/*
 * Decompiled with CFR 0.152.
 */
package de.cuioss.tools.io;

import de.cuioss.tools.io.IOStreams;
import de.cuioss.tools.io.StructuredFilename;
import de.cuioss.tools.logging.CuiLogger;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.StandardCopyOption;
import java.nio.file.attribute.FileAttribute;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.Objects;
import lombok.Generated;
import lombok.NonNull;

public final class MorePaths {
    private static final CuiLogger log = new CuiLogger(MorePaths.class);
    public static final String BACKUP_DIR_NAME = ".backup";
    public static final String MSG_DIRECTORY_NOT_ACCESSIBLE = "File or Directory {} is not accessible, reason: {}";
    public static final String BACKUP_FILE_SUFFIX = ".bck_";

    public static Path getRealPathSafely(Path path) {
        Objects.requireNonNull(path, "Path must not be null");
        try {
            return path.toRealPath(new LinkOption[0]);
        }
        catch (IOException e) {
            log.warn("Unable to resolve real path for '{}', due to '{}'. Returning absolutePath.", path, e.getMessage(), e);
            return path.toAbsolutePath();
        }
    }

    public static Path getRealPathSafely(String first, String ... more) {
        return MorePaths.getRealPathSafely(Paths.get(first, more));
    }

    public static Path getRealPathSafely(File file) {
        Objects.requireNonNull(file, "File must not be null");
        return MorePaths.getRealPathSafely(file.toPath());
    }

    public static boolean checkAccessiblePath(@NonNull Path path, boolean checkForDirectory, boolean verbose) {
        if (path == null) {
            throw new NullPointerException("path is marked non-null but is null");
        }
        if (!MorePaths.checkReadablePath(path, checkForDirectory, verbose)) {
            return false;
        }
        File pathFile = path.toFile();
        String absolutePath = pathFile.getAbsolutePath();
        if (!pathFile.canWrite()) {
            if (verbose) {
                log.warn(MSG_DIRECTORY_NOT_ACCESSIBLE, absolutePath, "Not Writable");
            }
            return false;
        }
        log.debug("{} denotes an existing file / directory with read and write permissions", absolutePath);
        return true;
    }

    public static boolean checkReadablePath(@NonNull Path path, boolean checkForDirectory, boolean verbose) {
        if (path == null) {
            throw new NullPointerException("path is marked non-null but is null");
        }
        File pathFile = path.toFile();
        String absolutePath = pathFile.getAbsolutePath();
        if (!pathFile.exists()) {
            if (verbose) {
                log.warn(MSG_DIRECTORY_NOT_ACCESSIBLE, absolutePath, "Not Existing");
            }
            return false;
        }
        if (checkForDirectory) {
            if (!pathFile.isDirectory()) {
                if (verbose) {
                    log.warn(MSG_DIRECTORY_NOT_ACCESSIBLE, absolutePath, "Not a directory");
                }
                return false;
            }
        } else if (!pathFile.isFile()) {
            if (verbose) {
                log.warn(MSG_DIRECTORY_NOT_ACCESSIBLE, absolutePath, "Not a file");
            }
            return false;
        }
        if (!pathFile.canRead()) {
            if (verbose) {
                log.warn(MSG_DIRECTORY_NOT_ACCESSIBLE, absolutePath, "Not Readable");
            }
            return false;
        }
        log.debug("{} denotes an existing file / directory with read permissions", absolutePath);
        return true;
    }

    public static boolean checkExecutablePath(@NonNull Path path, boolean verbose) {
        if (path == null) {
            throw new NullPointerException("path is marked non-null but is null");
        }
        File pathFile = path.toFile();
        String absolutePath = pathFile.getAbsolutePath();
        if (!pathFile.exists()) {
            if (verbose) {
                log.warn(MSG_DIRECTORY_NOT_ACCESSIBLE, absolutePath, "Not Existing");
            }
            return false;
        }
        if (!pathFile.isFile()) {
            if (verbose) {
                log.warn(MSG_DIRECTORY_NOT_ACCESSIBLE, absolutePath, "Not a file");
            }
            return false;
        }
        if (!pathFile.canExecute()) {
            if (verbose) {
                log.warn(MSG_DIRECTORY_NOT_ACCESSIBLE, absolutePath, "Not Executable");
            }
            return false;
        }
        log.debug("{} denotes an existing file / directory with execute permission", absolutePath);
        return true;
    }

    public static Path getBackupDirectoryForPath(Path directory) {
        if (!MorePaths.checkAccessiblePath(directory, true, true)) {
            throw new IllegalArgumentException(String.format("Given path '%s' does not denote an existing writable directory", directory.toFile().getAbsolutePath()));
        }
        Path backup = directory.resolve(BACKUP_DIR_NAME);
        File backupAsFile = backup.toFile();
        if (!backupAsFile.exists() && !backupAsFile.mkdir()) {
            throw new IllegalStateException(String.format("Unable to create directory '%s'", backup.toFile().getAbsolutePath()));
        }
        return backup;
    }

    public static Path backupFile(Path path) throws IOException {
        MorePaths.assertAccessibleFile(path);
        Path backupDir = MorePaths.getBackupDirectoryForPath(path.getParent());
        Path backupFile = MorePaths.createNonExistingPath(backupDir, path.getFileName() + BACKUP_FILE_SUFFIX + new SimpleDateFormat("yyyyMMddHHmmss").format(new Date()));
        Files.copy(path, backupFile, StandardCopyOption.REPLACE_EXISTING, StandardCopyOption.COPY_ATTRIBUTES);
        log.debug("Created backup from '{}' at '{}'", path.toFile().getAbsolutePath(), backupFile.toFile().getAbsolutePath());
        return backupFile;
    }

    public static Path copyToTempLocation(Path path) throws IOException {
        MorePaths.assertAccessibleFile(path);
        StructuredFilename filename = new StructuredFilename(path.getFileName());
        Path tempFile = Files.createTempFile(filename.getNamePart(), filename.getSuffix(), new FileAttribute[0]);
        Files.copy(path, tempFile, StandardCopyOption.REPLACE_EXISTING, StandardCopyOption.COPY_ATTRIBUTES);
        log.debug("Created temp-file from '{}' at '{}'", path.toFile().getAbsolutePath(), tempFile.toFile().getAbsolutePath());
        return tempFile;
    }

    public static void assertAccessibleFile(Path path) {
        Objects.requireNonNull(path, "path");
        if (!MorePaths.checkAccessiblePath(path, false, true)) {
            throw new IllegalArgumentException(String.format("Given path '%s' does not denote an existing readable file", path.toFile().getAbsolutePath()));
        }
    }

    static Path createNonExistingPath(Path parentDir, String fileName) {
        Path backupFile = parentDir.resolve(fileName);
        if (!backupFile.toFile().exists()) {
            return backupFile;
        }
        for (int counter = 1; counter < 20; ++counter) {
            String newName = fileName + "_" + counter;
            Path newBackupFile = parentDir.resolve(newName);
            if (newBackupFile.toFile().exists()) continue;
            return newBackupFile;
        }
        throw new IllegalStateException(String.format("Unable to determine a non-existing file within '%s' for file-name '%s'", parentDir.toFile().getAbsolutePath(), fileName));
    }

    public static boolean deleteQuietly(Path path) {
        log.trace("Deleting file {}", path);
        if (path == null) {
            return false;
        }
        File file = path.toFile();
        String absolutePath = file.getAbsolutePath();
        if (!file.exists()) {
            log.trace("Path {} does not exist", absolutePath);
            return false;
        }
        boolean recursiveSucceful = true;
        try {
            if (file.isDirectory()) {
                log.trace("Path {} is directory, checking children", absolutePath);
                for (String child : file.list()) {
                    if (MorePaths.deleteQuietly(path.resolve(child))) continue;
                    recursiveSucceful = false;
                }
            }
        }
        catch (Exception e) {
            log.trace(e, "Unable to check Path {} whether it is a directory", absolutePath);
        }
        try {
            if (Files.deleteIfExists(path)) {
                log.trace("Successully deleted path {}", absolutePath);
            } else {
                recursiveSucceful = false;
            }
        }
        catch (Exception e) {
            log.trace(e, "Unable to delete Path {}", absolutePath);
            return false;
        }
        return recursiveSucceful;
    }

    public static boolean contentEquals(Path path1, Path path2) throws IOException {
        Objects.requireNonNull(path1);
        Objects.requireNonNull(path2);
        File file1 = path1.toFile();
        File file2 = path2.toFile();
        boolean file1Exists = file1.exists();
        if (file1Exists != file2.exists()) {
            return false;
        }
        if (!file1Exists) {
            return true;
        }
        if (file1.isDirectory() || file2.isDirectory()) {
            throw new IOException("Can't compare directories, only files");
        }
        if (file1.length() != file2.length()) {
            return false;
        }
        if (file1.getCanonicalFile().equals(file2.getCanonicalFile())) {
            return true;
        }
        try (FileInputStream input1 = new FileInputStream(file1);){
            boolean bl;
            try (FileInputStream input2 = new FileInputStream(file2);){
                bl = IOStreams.contentEquals(input1, input2);
            }
            return bl;
        }
    }

    public static void saveAndBackup(Path filePath, FileWriteHandler fileWriteHandler) throws IOException {
        Path temp = MorePaths.copyToTempLocation(filePath);
        fileWriteHandler.write(temp);
        MorePaths.backupFile(filePath);
        Files.copy(temp, filePath, StandardCopyOption.REPLACE_EXISTING, StandardCopyOption.COPY_ATTRIBUTES);
    }

    public static boolean isSameFile(Path path, Path path2) {
        if (null == path && null == path2) {
            return true;
        }
        if (null != path && null != path2) {
            if (!path.toFile().exists() || !path2.toFile().exists()) {
                log.debug("Comparing paths with #equals, as at least one path does not exist. path_a={}, path_b={}", path, path2);
                return path.equals(path2);
            }
            try {
                return Files.isSameFile(path, path2);
            }
            catch (IOException e) {
                log.error(e, "Portal-123: Unable to compare path_a={} and path_b={}", path, path2);
            }
        } else {
            log.trace("at least one path is null: path_a={}, path_b={}", path, path2);
        }
        return false;
    }

    @Generated
    private MorePaths() {
        throw new UnsupportedOperationException("This is a utility class and cannot be instantiated");
    }

    public static interface FileWriteHandler {
        public void write(Path var1) throws IOException;
    }
}

