/*
 * Decompiled with CFR 0.152.
 */
package de.cuioss.tools.io;

import de.cuioss.tools.ToolsLogMessages;
import de.cuioss.tools.io.IOStreams;
import de.cuioss.tools.io.PathTraversalSecurity;
import de.cuioss.tools.io.StructuredFilename;
import de.cuioss.tools.logging.CuiLogger;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.StandardCopyOption;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.Objects;
import lombok.Generated;
import lombok.NonNull;

public final class MorePaths {
    private static final CuiLogger LOGGER = new CuiLogger(MorePaths.class);
    public static final String BACKUP_DIR_NAME = ".backup";
    public static final String BACKUP_FILE_SUFFIX = ".bck_";

    public static Path getRealPathSafely(Path path) {
        Objects.requireNonNull(path, "Path must not be null");
        try {
            return path.toRealPath(new LinkOption[0]);
        }
        catch (IOException e) {
            LOGGER.warn((Throwable)e, ToolsLogMessages.WARN.REAL_PATH_RESOLUTION_FAILED, path, e.getMessage());
            return path.toAbsolutePath();
        }
    }

    public static Path getRealPathSafely(String first, String ... more) {
        return MorePaths.getRealPathSafely(Path.of(first, more));
    }

    public static Path getRealPathSafely(File file) {
        Objects.requireNonNull(file, "File must not be null");
        return MorePaths.getRealPathSafely(file.toPath());
    }

    public static boolean checkAccessiblePath(@NonNull Path path, boolean checkForDirectory, boolean verbose) {
        if (path == null) {
            throw new NullPointerException("path is marked non-null but is null");
        }
        if (!MorePaths.checkReadablePath(path, checkForDirectory, verbose)) {
            return false;
        }
        File pathFile = path.toFile();
        String absolutePath = pathFile.getAbsolutePath();
        if (!pathFile.canWrite()) {
            if (verbose) {
                LOGGER.warn(ToolsLogMessages.WARN.PATH_NOT_ACCESSIBLE, absolutePath, "Not Writable");
            }
            return false;
        }
        LOGGER.debug("%s denotes an existing file / directory with read and write permissions", absolutePath);
        return true;
    }

    public static boolean checkReadablePath(@NonNull Path path, boolean checkForDirectory, boolean verbose) {
        if (path == null) {
            throw new NullPointerException("path is marked non-null but is null");
        }
        File pathFile = path.toFile();
        String absolutePath = pathFile.getAbsolutePath();
        if (!pathFile.exists()) {
            if (verbose) {
                LOGGER.warn(ToolsLogMessages.WARN.PATH_NOT_ACCESSIBLE, absolutePath, "Not Existing");
            }
            return false;
        }
        if (checkForDirectory) {
            if (!pathFile.isDirectory()) {
                if (verbose) {
                    LOGGER.warn(ToolsLogMessages.WARN.PATH_NOT_ACCESSIBLE, absolutePath, "Not a directory");
                }
                return false;
            }
        } else if (!pathFile.isFile()) {
            if (verbose) {
                LOGGER.warn(ToolsLogMessages.WARN.PATH_NOT_ACCESSIBLE, absolutePath, "Not a file");
            }
            return false;
        }
        if (!pathFile.canRead()) {
            if (verbose) {
                LOGGER.warn(ToolsLogMessages.WARN.PATH_NOT_ACCESSIBLE, absolutePath, "Not Readable");
            }
            return false;
        }
        LOGGER.debug("%s denotes an existing file / directory with read permissions", absolutePath);
        return true;
    }

    public static boolean checkExecutablePath(@NonNull Path path, boolean verbose) {
        if (path == null) {
            throw new NullPointerException("path is marked non-null but is null");
        }
        File pathFile = path.toFile();
        String absolutePath = pathFile.getAbsolutePath();
        if (!pathFile.exists()) {
            if (verbose) {
                LOGGER.warn(ToolsLogMessages.WARN.PATH_NOT_ACCESSIBLE, absolutePath, "Not Existing");
            }
            return false;
        }
        if (!pathFile.isFile()) {
            if (verbose) {
                LOGGER.warn(ToolsLogMessages.WARN.PATH_NOT_ACCESSIBLE, absolutePath, "Not a file");
            }
            return false;
        }
        if (!pathFile.canExecute()) {
            if (verbose) {
                LOGGER.warn(ToolsLogMessages.WARN.PATH_NOT_ACCESSIBLE, absolutePath, "Not Executable");
            }
            return false;
        }
        LOGGER.debug("%s denotes an existing file / directory with execute permission", absolutePath);
        return true;
    }

    public static Path getBackupDirectoryForPath(Path directory) {
        if (!MorePaths.checkAccessiblePath(directory, true, true)) {
            throw new IllegalArgumentException("Given path '%s' does not denote an existing writable directory".formatted(directory.toFile().getAbsolutePath()));
        }
        Path backup = directory.resolve(BACKUP_DIR_NAME);
        File backupAsFile = backup.toFile();
        if (!backupAsFile.exists() && !backupAsFile.mkdir()) {
            throw new IllegalStateException("Unable to create directory '%s'".formatted(backup.toFile().getAbsolutePath()));
        }
        return backup;
    }

    public static Path backupFile(Path path) throws IOException {
        MorePaths.assertAccessibleFile(path);
        Path backupDir = MorePaths.getBackupDirectoryForPath(path.getParent());
        Path backupFile = MorePaths.createNonExistingPath(backupDir, String.valueOf(path.getFileName()) + BACKUP_FILE_SUFFIX + new SimpleDateFormat("yyyyMMddHHmmss").format(new Date()));
        Files.copy(path, backupFile, StandardCopyOption.REPLACE_EXISTING, StandardCopyOption.COPY_ATTRIBUTES);
        LOGGER.debug("Created backup from '%s' at '%s'", path.toFile().getAbsolutePath(), backupFile.toFile().getAbsolutePath());
        return backupFile;
    }

    public static Path copyToTempLocation(Path path) throws IOException {
        MorePaths.assertAccessibleFile(path);
        Path normalizedPath = path.normalize();
        Path realPath = normalizedPath.toRealPath(new LinkOption[0]);
        Path fileName = realPath.getFileName();
        if (fileName == null) {
            throw new IllegalArgumentException("Invalid path: no filename found");
        }
        StructuredFilename filename = new StructuredFilename(fileName);
        String namePart = filename.getNamePart();
        String suffix = filename.getSuffix();
        PathTraversalSecurity.validatePathSegment(namePart);
        PathTraversalSecurity.validatePathSegment(suffix);
        Path tempFile = PathTraversalSecurity.createSecureTempFile(namePart, suffix);
        Files.copy(realPath, tempFile, StandardCopyOption.REPLACE_EXISTING, StandardCopyOption.COPY_ATTRIBUTES);
        LOGGER.debug("Created temp-file from '%s' at '%s'", realPath.toFile().getAbsolutePath(), tempFile.toFile().getAbsolutePath());
        return tempFile;
    }

    public static void assertAccessibleFile(Path path) {
        Objects.requireNonNull(path, "path");
        if (!MorePaths.checkAccessiblePath(path, false, true)) {
            throw new IllegalArgumentException("Given path '%s' does not denote an existing readable file".formatted(path.toFile().getAbsolutePath()));
        }
    }

    static Path createNonExistingPath(Path parentDir, String fileName) {
        Path backupFile = parentDir.resolve(fileName);
        if (!backupFile.toFile().exists()) {
            return backupFile;
        }
        for (int counter = 1; counter < 20; ++counter) {
            String newName = fileName + "_" + counter;
            Path newBackupFile = parentDir.resolve(newName);
            if (newBackupFile.toFile().exists()) continue;
            return newBackupFile;
        }
        throw new IllegalStateException("Unable to determine a non-existing file within '%s' for file-name '%s'".formatted(parentDir.toFile().getAbsolutePath(), fileName));
    }

    public static boolean deleteQuietly(Path path) {
        LOGGER.trace("Deleting file %s", path);
        if (path == null) {
            return false;
        }
        File file = path.toFile();
        String absolutePath = file.getAbsolutePath();
        if (!file.exists()) {
            LOGGER.trace("Path %s does not exist", absolutePath);
            return false;
        }
        boolean recursiveSucceful = true;
        try {
            if (file.isDirectory()) {
                LOGGER.trace("Path %s is directory, checking children", absolutePath);
                for (String child : file.list()) {
                    if (MorePaths.deleteQuietly(path.resolve(child))) continue;
                    recursiveSucceful = false;
                }
            }
        }
        catch (SecurityException | UnsupportedOperationException e) {
            LOGGER.trace(e, "Unable to check Path %s whether it is a directory", absolutePath);
        }
        try {
            if (Files.deleteIfExists(path)) {
                LOGGER.trace("Successully deleted path %s", absolutePath);
            } else {
                recursiveSucceful = false;
            }
        }
        catch (IOException | SecurityException e) {
            LOGGER.trace(e, "Unable to delete Path %s", absolutePath);
            return false;
        }
        return recursiveSucceful;
    }

    public static boolean contentEquals(Path path1, Path path2) throws IOException {
        Objects.requireNonNull(path1);
        Objects.requireNonNull(path2);
        File file1 = path1.toFile();
        File file2 = path2.toFile();
        boolean file1Exists = file1.exists();
        if (file1Exists != file2.exists()) {
            return false;
        }
        if (!file1Exists) {
            return true;
        }
        if (file1.isDirectory() || file2.isDirectory()) {
            throw new IOException("Can't compare directories, only files");
        }
        if (file1.length() != file2.length()) {
            return false;
        }
        if (file1.getCanonicalFile().equals(file2.getCanonicalFile())) {
            return true;
        }
        try (FileInputStream input1 = new FileInputStream(file1);){
            boolean bl;
            try (FileInputStream input2 = new FileInputStream(file2);){
                bl = IOStreams.contentEquals(input1, input2);
            }
            return bl;
        }
    }

    public static void saveAndBackup(Path filePath, FileWriteHandler fileWriteHandler) throws IOException {
        Path temp = MorePaths.copyToTempLocation(filePath);
        fileWriteHandler.write(temp);
        MorePaths.backupFile(filePath);
        Files.copy(temp, filePath, StandardCopyOption.REPLACE_EXISTING, StandardCopyOption.COPY_ATTRIBUTES);
    }

    public static boolean isSameFile(Path path, Path path2) {
        if (null == path && null == path2) {
            return true;
        }
        if (null != path && null != path2) {
            if (!path.toFile().exists() || !path2.toFile().exists()) {
                LOGGER.debug("Comparing paths with #equals, as at least one path does not exist. path_a=%s, path_b=%s", path, path2);
                return path.equals(path2);
            }
            try {
                return Files.isSameFile(path, path2);
            }
            catch (IOException e) {
                LOGGER.error((Throwable)e, ToolsLogMessages.ERROR.PATH_COMPARISON_FAILED, path, path2);
            }
        } else {
            LOGGER.trace("at least one path is null: path_a=%s, path_b=%s", path, path2);
        }
        return false;
    }

    @Generated
    private MorePaths() {
        throw new UnsupportedOperationException("This is a utility class and cannot be instantiated");
    }

    public static interface FileWriteHandler {
        public void write(Path var1) throws IOException;
    }
}

