//++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
// This file is a part of the 'esoco-business' project.
// Copyright 2018 Elmar Sonnenschein, esoco GmbH, Flensburg, Germany
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//	  http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.
//++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
package de.esoco.entity;

import de.esoco.entity.EntityDefinition.DisplayMode;
import de.esoco.lib.collection.CollectionUtil;
import de.esoco.lib.expression.Conversions;
import de.esoco.lib.expression.Predicate;
import de.esoco.lib.expression.Predicates;
import de.esoco.lib.json.JsonBuilder;
import de.esoco.lib.property.MutableProperties;
import de.esoco.lib.property.PropertyName;
import de.esoco.lib.property.StringProperties;
import de.esoco.lib.text.TextConvert;
import de.esoco.storage.AfterStoreHandler;
import de.esoco.storage.StorageException;
import org.obrel.core.IntermediateRelation;
import org.obrel.core.Relatable;
import org.obrel.core.Relation;
import org.obrel.core.RelationType;
import org.obrel.core.SerializableRelatedObject;
import org.obrel.type.ListenerTypes;
import org.obrel.type.MetaTypes;
import org.obrel.type.StandardTypes;

import java.io.PrintStream;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.Date;
import java.util.List;
import java.util.Map;

import static de.esoco.entity.EntityRelationTypes.CACHE_ENTITY;
import static de.esoco.entity.EntityRelationTypes.DEPENDENT_STORE_ENTITIES;
import static de.esoco.entity.EntityRelationTypes.ENTITY_STORE_ORIGIN;
import static de.esoco.entity.EntityRelationTypes.EXTRA_ATTRIBUTES_MODIFIED;
import static de.esoco.entity.EntityRelationTypes.EXTRA_ATTRIBUTES_READ;
import static de.esoco.entity.EntityRelationTypes.EXTRA_ATTRIBUTE_MAP;
import static de.esoco.entity.EntityRelationTypes.REMOVED_CHILDREN;
import static de.esoco.entity.EntityRelationTypes.SKIP_NEXT_CHANGE_LOGGING;
import static de.esoco.lib.expression.Predicates.equalTo;
import static de.esoco.lib.expression.Predicates.notNull;
import static de.esoco.storage.StorageRelationTypes.PERSISTENT;
import static de.esoco.storage.StorageRelationTypes.STORAGE_MAPPING;
import static org.obrel.type.MetaTypes.INITIALIZING;
import static org.obrel.type.MetaTypes.LOCKED;
import static org.obrel.type.MetaTypes.MODIFIED;
import static org.obrel.type.StandardTypes.PREVIOUS_VALUE;

/**
 * The base class for persistent entities that are created from entity
 * definitions. It can either be used directly or subclassed for entity classes
 * that want to provide additional methods. But all standard attributes from the
 * definition can be accessed directly with the {@link Relatable} methods.
 *
 * <p>Besides the standard attributes from the entity definition an entity can
 * have arbitrary extra attributes which will only be stored if they are set on
 * an entity. These extra attributes can be set and retrieved with the methods
 * {@link #getExtraAttribute(RelationType, Object)} and
 * {@link #setExtraAttribute(RelationType, Object)}.</p>
 *
 * <p>Extra attribute relation types must be created through the factory
 * methods in the class {@link ExtraAttributes} because any other kind of
 * relation type in an entity will be recognized as a standard attribute when
 * analyzed by {@link EntityDefinition}.</p>
 *
 * @author eso
 */
public class Entity extends SerializableRelatedObject
	implements AfterStoreHandler {

	/**
	 * The ID field name in JSON representations of entities, especially as
	 * generated by the method {@link #createChangeDescription()}.
	 */
	public static final String JSON_ID_FIELD = "_id";

	/**
	 * The type field name in JSON representations of entities, especially as
	 * generated by the method {@link #createChangeDescription()}.
	 */
	public static final String JSON_TYPE_FIELD = "_type";

	/**
	 * The name field name in JSON representations of entities, especially as
	 * generated by the method {@link #createChangeDescription()}.
	 */
	public static final String JSON_NAME_FIELD = "_name";

	/**
	 * The JSON field name new value of a change log generated by the method
	 * {@link #createChangeDescription()}.
	 */
	public static final String JSON_CHANGE_NEW_VALUE = "new";

	/**
	 * The JSON field name new value of a change log generated by the method
	 * {@link #createChangeDescription()}.
	 */
	public static final String JSON_CHANGE_OLD_VALUE = "old";

	/**
	 * The JSON field name suffix for removed children in a change log
	 * generated
	 * by the method {@link #createChangeDescription()}.
	 */
	public static final String JSON_REMOVED_CHILDREN_SUFFIX = "-removed";

	private static final long serialVersionUID = 1L;

	private static final String JSON_INDENT = "  ";

	static {
		Conversions.registerStringConversion(Entity.class,
			EntityFunctions.entityToString());
	}

	/**
	 * Initialize this instance.
	 */
	public Entity() {
		// modified flag must exist to indicate the modification tracking
		// support to the storage framework, therefore set to FALSE
		set(MODIFIED, Boolean.FALSE);
		get(ListenerTypes.RELATION_LISTENERS).add(getDefinition());
	}

	/**
	 * Convenience method to set a boolean display property.
	 *
	 * @see #setAttributeDisplayProperty(Class, PropertyName, Object,
	 * RelationType...)
	 */
	@SuppressWarnings("boxing")
	public static void setAttributeDisplayFlag(
		Class<? extends Entity> entityClass, PropertyName<Boolean> property,
		RelationType<?>... attributes) {
		setAttributeDisplayProperty(entityClass, property, true, attributes);
	}

	/**
	 * Convenience method to set an integer display property.
	 *
	 * @see #setAttributeDisplayProperty(Class, PropertyName, Object,
	 * RelationType...)
	 */
	@SuppressWarnings("boxing")
	public static void setAttributeDisplayProperty(
		Class<? extends Entity> entityClass, int value,
		PropertyName<Integer> property, RelationType<?>... attributes) {
		setAttributeDisplayProperty(entityClass, property, value, attributes);
	}

	/**
	 * Helper method for subclasses to set a display property on attributes. If
	 * the attribute doesn't have display attributes yet they will be created
	 * and stored in the {@link EntityRelationTypes#DISPLAY_PROPERTIES}
	 * relation.
	 *
	 * <p>ATTENTION: this setting affects all occurrences of the given
	 * attribute. Properties of attributes that are re-used in different
	 * contexts (e.g. relation types defined in {@link StandardTypes} or
	 * similar) should be set with a
	 * {@link EntityDefinition#ATTRIBUTE_DISPLAY_PROPERTIES_FIELD} instead to
	 * prevent side effects.</p>
	 *
	 * @param entityClass The entity class for which to set the property
	 * @param property    The property to set
	 * @param value       The property values
	 * @param attributes  The attributes to set the display property on
	 */
	public static <T> void setAttributeDisplayProperty(
		Class<? extends Entity> entityClass, PropertyName<T> property, T value,
		RelationType<?>... attributes) {
		Map<RelationType<?>, MutableProperties> displayPropertiesMap =
			EntityDefinition.getAttributeDisplayProperties(entityClass);

		for (RelationType<?> attribute : attributes) {
			MutableProperties properties = displayPropertiesMap.get(attribute);

			if (properties == null) {
				properties = new StringProperties();
				displayPropertiesMap.put(attribute, properties);
			}

			properties.setProperty(property, value);
		}
	}

	/**
	 * A convenience method to add a single child to an entity. Invokes the
	 * method {@link #addChildren(RelationType, Entity...)} .
	 *
	 * @param childAttr The attribute that references the child type
	 * @param child     The child object to add
	 * @throws IllegalArgumentException If one of the parameters is invalid or
	 *                                  if the child is already stored in this
	 *                                  entity
	 */
	@SuppressWarnings("unchecked")
	public final <C extends Entity> void addChild(
		RelationType<List<C>> childAttr, C child) {
		addChildren(childAttr, child);
	}

	/**
	 * A convenience method to add variable number of children to an entity.
	 * Invokes the method {@link #addChildren(RelationType, List)}.
	 *
	 * @param childAttr The attribute that references the child type
	 * @param children  The new child objects to add
	 * @throws IllegalArgumentException If one of the parameters is invalid or
	 *                                  if one of the children is already
	 *                                  stored
	 *                                  in this entity
	 */
	@SuppressWarnings("unchecked")
	public final <C extends Entity> void addChildren(
		RelationType<List<C>> childAttr, C... children) {
		addChildren(childAttr, Arrays.asList(children));
	}

	/**
	 * Adds child entities to this entity. This will set the parent references
	 * in the children to this instance. If a child already exists in this
	 * entity an exception will be thrown. It is recommended that entity
	 * children are always added through this method unless an entity subclass
	 * implements (and documents) more specific methods for the management of
	 * child entities.
	 *
	 * @param childAttr The attribute that references the child type
	 * @param children  The new child objects to add to this entity
	 * @throws IllegalArgumentException If one of the parameters is invalid or
	 *                                  if one of the children is already
	 *                                  stored
	 *                                  in this entity
	 */
	public <C extends Entity> void addChildren(RelationType<List<C>> childAttr,
		List<C> children) {
		if (childAttr == null || children == null || children.size() == 0) {
			throw new IllegalArgumentException(
				"Arguments must not be NULL or empty");
		}

		boolean initializing = hasFlag(INITIALIZING);

		// mark as modified to perform child count update in storage framework
		// do before child list modification to allow the modification tracking
		// in EntityDefinition to prohibit the change by throwing an exception
		if (!initializing) {
			set(MODIFIED);
		}

		@SuppressWarnings("unchecked")
		EntityDefinition<Entity> def =
			(EntityDefinition<Entity>) getDefinition();

		EntityDefinition<?> childDef =
			(EntityDefinition<?>) childAttr.get(STORAGE_MAPPING);

		def.initChildren(this, children, childDef, initializing);
		get(childAttr).addAll(children);
	}

	/**
	 * Implemented to store all new and modified extra attributes.
	 *
	 * @see AfterStoreHandler#afterStore()
	 */
	@Override
	public void afterStore() throws Exception {
		if (hasFlag(EXTRA_ATTRIBUTES_MODIFIED)) {
			Entity storeOrigin = getUpwards(ENTITY_STORE_ORIGIN);

			for (ExtraAttribute extraAttribute : get(
				EXTRA_ATTRIBUTE_MAP).values()) {
				if (extraAttribute.hasFlag(MODIFIED)) {
					EntityManager.storeEntity(extraAttribute, storeOrigin);
				}
			}

			set(EXTRA_ATTRIBUTES_MODIFIED, Boolean.FALSE);
		}
	}

	/**
	 * A method that appends a string description of a certain attribute to a
	 * {@link StringBuilder}. The attribute description will appended as
	 * defined
	 * by the given format string which must contain two %s references. The
	 * first one will be replaced by the attribute name after converting it
	 * with
	 * the method {@link TextConvert#capitalize(String, String)} and the second
	 * is replaced with the attribute value {@link Object#toString()}. NULL
	 * values will only be included if the boolean parameter is TRUE. Otherwise
	 * no attribute description will be appended at all.
	 *
	 * @param stringBuilder The string builder to append the attribute
	 *                      description to
	 * @param attribute     The relation type of the attribute to append
	 * @param format        The format string for the formatting of the
	 *                      attribute name and value
	 * @param includeNull   TRUE to include NULL values, FALSE to omit them
	 */
	public void appendAttribute(StringBuilder stringBuilder,
		RelationType<?> attribute, String format, boolean includeNull) {
		Object value = get(attribute);

		if (includeNull || value != null) {
			String name = TextConvert.capitalize(
				TextConvert.lastElementOf(attribute.getName()), " ");

			if (value == null) {
				value = "NULL";
			}

			stringBuilder.append(String.format(format, name, value));
		}
	}

	/**
	 * Creates a string for the attributes of a certain display mode.
	 *
	 * @param displayMode The display mode
	 * @param separator   The separator string between attributes
	 * @return The resulting string
	 */
	public String attributeString(DisplayMode displayMode, String separator) {
		return attributeString(
			getDefinition().getDisplayAttributes(displayMode), separator);
	}

	/**
	 * Creates a string for a certain list of entity attributes. NULL values
	 * will be omitted from the output.
	 *
	 * @param attributes The attribute list
	 * @param separator  The separator string between attributes
	 * @return The resulting string
	 */
	public String attributeString(List<RelationType<?>> attributes,
		String separator) {
		StringBuilder result = new StringBuilder();

		for (RelationType<?> attribute : attributes) {
			String value = getAttributeValue(attribute);

			if (value != null) {
				result.append(value).append(separator);
			} else {
				result.append(separator);
			}
		}

		if (result.length() > 0) {
			result.setLength(result.length() - separator.length());
		}

		return result.toString();
	}

	/**
	 * This method can be overridden by subclasses to check if the instance's
	 * hierarchy needs an update. In such a case the method should return the
	 * topmost parent of the hierarchy which needs to be stored to make all
	 * changes persistent. Otherwise it should return itself to continue with
	 * the standard update. The default implementation returns the topmost
	 * parent entity in this entity's hierarchy that has been marked with the
	 * {@link MetaTypes#MODIFIED} flag.
	 *
	 * @return The topmost entity in the hierarchy to update or THIS
	 */
	public Entity checkForHierarchyUpdate() {
		Entity parent = getParent();
		Entity updateEntity = this;

		if (parent != null && parent.hasFlag(MODIFIED)) {
			updateEntity = parent.checkForHierarchyUpdate();
		}

		return updateEntity;
	}

	/**
	 * Collects all children of a certain type in the hierarchy of this entity
	 * that match a certain predicate.
	 *
	 * @param childAttribute The child attribute to descend the hierarchy at
	 * @param criteria       The predicate to evaluate the child entities with
	 * @return A new list containing the resulting child entities (may be empty
	 * but will never be NULL)
	 */
	public <E extends Entity> List<E> collectDownwards(
		RelationType<List<E>> childAttribute, Predicate<? super E> criteria) {
		return EntityManager.collectDownwards(get(childAttribute),
			childAttribute, criteria);
	}

	/**
	 * Creates a JSON string containing the modified values of this entity and
	 * it's hierarchy. Value changes are recorded by the entity modification
	 * tracking with annotations of type {@link StandardTypes#PREVIOUS_VALUE}.
	 *
	 * @return The resulting JSON string
	 */
	public String createChangeDescription() {
		return toJson("", isPersistent(), true);
	}

	/**
	 * Overridden to check whether this entity is equal to another entity.
	 *
	 * @see Object#equals(Object)
	 */
	@Override
	public boolean equals(Object object) {
		if (this == object) {
			return true;
		}

		if (object == null || getClass() != object.getClass()) {
			return false;
		}

		Entity other = (Entity) object;
		long id = getId();

		boolean modified = hasFlag(MODIFIED);
		boolean otherModified = other.hasFlag(MODIFIED);

		if ((!modified && otherModified) || (modified && !otherModified)) {
			return false;
		} else if (id != 0 && !(modified || otherModified)) {
			return id == other.getId();
		} else {
			return attributesEqual(other);
		}
	}

	/**
	 * Searches for a child entity in a hierarchy of children that matches a
	 * certain predicate. If the given children are not hierarchical only the
	 * flat list of children will be searched. This method is based on the
	 * {@link EntityManager#findDownwards(List, RelationType, Predicate)}
	 * method.
	 *
	 * @param childAttribute The child attribute to search and descend
	 *                       hierarchically
	 * @param criteria       The predicate to evaluate the children with
	 * @return The first matching child or NULL if none could be found
	 */
	public <E extends Entity> E findDownwards(
		RelationType<List<E>> childAttribute, Predicate<? super E> criteria) {
		return EntityManager.findDownwards(get(childAttribute), childAttribute,
			criteria);
	}

	/**
	 * Invokes {@link EntityFunctions#format(Entity)} on this instance.
	 *
	 * @return A formatted string representation of this entity
	 */
	public String format() {
		return EntityFunctions.format(this);
	}

	/**
	 * Returns a child that is valid on a certain date. The child must be a
	 * subclass of {@link PeriodEntity} to support the date check.
	 *
	 * @param childAttr     The child attribute to search for
	 * @param date          The date to return the child for
	 * @param extraCriteria An optional predicate to check valid children with
	 *                      or NULL for none
	 * @param upwards       TRUE to look upwards in the hierarchy if not found
	 *                      in this entity
	 * @return The matching child or NULL if none could be found
	 */
	public <C extends PeriodEntity> C getChild(RelationType<List<C>> childAttr,
		Date date, Predicate<? super C> extraCriteria, boolean upwards) {
		C result = null;

		for (C child : get(childAttr)) {
			if (child.isValidOn(date) && (extraCriteria == null ||
				extraCriteria.evaluate(child) == Boolean.TRUE)) {
				result = child;

				break;
			}
		}

		if (result == null && upwards) {
			RelationType<? extends Entity> parentAttr =
				getDefinition().getParentAttribute(getDefinition());

			if (parentAttr != null) {
				Entity parent = get(parentAttr);

				if (parent != null) {
					result = parent.getChild(childAttr, date, extraCriteria,
						upwards);
				}
			}
		}

		return result;
	}

	/**
	 * Returns the hierarchy of all or certain children of this entity. The
	 * returned list will contain all of this entity's direct and indirect
	 * children of the given attribute. If the predicate parameter is not NULL
	 * only children will be returned of which the predicate evaluation yields
	 * TRUE. If the evaluation of a certain child is FALSE it's own children
	 * will still be evaluated.
	 *
	 * <p>The returned list will never be NULL and can be manipulated freely.
	 * </p>
	 *
	 * @param childAttribute The attribute of the children to return
	 * @param criteria       The optional criteria or NULL for all children
	 * @return The list of children
	 */
	@SuppressWarnings("boxing")
	public <C extends Entity> List<C> getChildHierarchy(
		RelationType<List<C>> childAttribute, Predicate<? super C> criteria) {
		List<C> children = get(childAttribute);
		List<C> hierarchy = new ArrayList<C>(children.size());

		for (C child : children) {
			if (criteria != null && criteria.evaluate(child)) {
				hierarchy.add(child);
			}

			hierarchy.addAll(child.getChildHierarchy(childAttribute,
				criteria));
		}

		return hierarchy;
	}

	/**
	 * Returns the entity definition for this entity class.
	 *
	 * @return The entity definition
	 */
	public EntityDefinition<?> getDefinition() {
		return EntityManager.getEntityDefinition(getClass());
	}

	/**
	 * Returns the value of a certain extra attribute of this entity.
	 *
	 * @param key          The key that identifies the attribute
	 * @param defaultValue The default value to return if no extra attribute
	 *                     with the given key exists
	 * @return The attribute value or the default value if the attribute is not
	 * set
	 * @throws StorageException If retrieving the extra attributes of this
	 *                          entity fails
	 */
	@SuppressWarnings("unchecked")
	public <T> T getExtraAttribute(RelationType<T> key, T defaultValue)
		throws StorageException {
		ExtraAttribute extraAttribute =
			getExtraAttributeMap().get(key.getSimpleName());

		T value = defaultValue;

		if (extraAttribute != null) {
			value = (T) extraAttribute.get(ExtraAttribute.VALUE);
		}

		return value;
	}

	/**
	 * Returns the value of an extra attribute of this entity for a certain
	 * owner entity.
	 *
	 * @param owner        The owner entity of the extra attribute
	 * @param key          The key that identifies the attribute
	 * @param defaultValue The default value to return if no extra attribute
	 *                     with the given key exists
	 * @param fallback     TRUE to return the default extra attribute if no
	 *                     attribute for the given owner exists; FALSE to just
	 *                     return the default value
	 * @return The attribute value or the default value if the attribute is not
	 * set
	 * @throws StorageException If querying the extra attribute fails
	 */
	@SuppressWarnings("unchecked")
	public <T> T getExtraAttributeFor(Entity owner, RelationType<T> key,
		T defaultValue, boolean fallback) throws StorageException {
		ExtraAttribute extraAttribute = queryExtraAttributeFor(owner, key);

		T value = defaultValue;

		if (extraAttribute != null) {
			value = (T) extraAttribute.get(ExtraAttribute.VALUE);
		} else if (fallback) {
			value = getExtraAttribute(key, value);
		}

		return value;
	}

	/**
	 * Returns an extra attribute value from the hierarchy of this entity. If
	 * the value of the given relation type is NULL in this entity it will be
	 * looked up recursively in the parent of the same type (i.e. with the same
	 * entity definition) if such exists.
	 *
	 * @param typedKey The extra attribute key
	 * @return The extra attribute value
	 * @throws StorageException If retrieving the extra attributes of this
	 *                          entity fails
	 */
	public <T> T getExtraAttributeUpwards(RelationType<T> typedKey)
		throws StorageException {
		T value = getExtraAttribute(typedKey, null);

		if (value == null) {
			Entity parent = getParent();

			if ((parent != null) &&
				(parent.getDefinition() == getDefinition())) {
				value = parent.getExtraAttributeUpwards(typedKey);
			}
		}

		return value;
	}

	/**
	 * Returns the keys of the extra attributes that are set on this entity.
	 * The
	 * values of the extra attribute can then be queried through the method
	 * {@link #getExtraAttribute(RelationType, Object)}. The returned
	 * collection
	 * can be modified freely by the caller.
	 *
	 * @return A collection of the current extra attribute keys
	 * @throws StorageException If retrieving the extra attributes of this
	 *                          entity fails
	 */
	public Collection<RelationType<?>> getExtraAttributes()
		throws StorageException {
		return CollectionUtil.map(getExtraAttributeMap().values(),
			ExtraAttribute.KEY);
	}

	/**
	 * Returns the global ID of this entity.
	 *
	 * @see EntityManager#getGlobalEntityId(Entity)
	 */
	public String getGlobalId() {
		return EntityManager.getGlobalEntityId(this);
	}

	/**
	 * Returns the ID of this entity.
	 *
	 * @return The entity ID
	 */
	public long getId() {
		return get(getIdAttribute()).longValue();
	}

	/**
	 * Returns the ID attribute of this entity.
	 *
	 * @return The ID attribute
	 */
	public RelationType<? extends Number> getIdAttribute() {
		return getDefinition().getIdAttribute();
	}

	/**
	 * Returns the parent entity of this entity if available. This will either
	 * be the hierarchy parent or, if it is NULL and a master attribute is
	 * present, the master entity.
	 *
	 * @return The parent entity or NULL if this entity has no parent or if
	 * none
	 * is set
	 */
	public Entity getParent() {
		return getDefinition().getParent(this);
	}

	/**
	 * Returns the previous value of a modified attribute.
	 *
	 * @param attribute The attribute to get the previous value of
	 * @return The previous value (may be NULL)
	 */
	@SuppressWarnings("unchecked")
	public <T> T getPrevious(RelationType<T> attribute) {
		Relation<T> attrRelation = getRelation(attribute);

		return attrRelation != null ?
		       (T) attrRelation.get(PREVIOUS_VALUE) :
		       null;
	}

	/**
	 * Returns the root parent of this entity's hierarchy. This will either be
	 * the topmost parent of the same entity type or, if a master attribute is
	 * present, the master entity.
	 *
	 * @return The root parent of this entity
	 */
	public Entity getRoot() {
		Entity root = this;

		while (root.getParent() != null) {
			root = root.getParent();
		}

		return root;
	}

	/**
	 * Returns the type attribute of this entity.
	 *
	 * @return The type attribute
	 */
	public RelationType<Enum<?>> getTypeAttribute() {
		return getDefinition().getTypeAttribute();
	}

	/**
	 * Returns an attribute value from the hierarchy of this entity. If the
	 * value of the given relation type is NULL in this entity it will be
	 * looked
	 * up recursively in the parent of the same type (i.e. with the same entity
	 * definition) if such exists.
	 *
	 * @param type The relation type of the attribute to return the value of
	 * @return The attribute value (NULL if not set in the hierarchy)
	 */
	public <T> T getUpwards(RelationType<T> type) {
		T value = get(type);

		if (value == null) {
			Entity parent = getParent();

			if (parent != null && parent.getDefinition() == getDefinition()) {
				value = parent.getUpwards(type);
			}
		}

		return value;
	}

	/**
	 * A shortcut for {@link #getExtraAttribute(RelationType, Object)} that
	 * converts storage exception into runtime exceptions. Extra attributes
	 * should typically be accessible as their entity has obviously already
	 * been
	 * retrieved from the storage so the exception handling is normally
	 * superfluous. Nonetheless developers need to be aware that accessing an
	 * extra attribute also can cause a storage access (at least for the first
	 * XA).
	 *
	 * @see #getExtraAttribute(RelationType, Object)
	 */
	public <T> T getXA(RelationType<T> key, T defaultValue) {
		try {
			return getExtraAttribute(key, defaultValue);
		} catch (StorageException e) {
			throw new IllegalStateException(e);
		}
	}

	/**
	 * Checks whether changes to this entity should be logged the history is
	 * enabled. The default implementation returns TRUE unless the relation
	 * {@link EntityRelationTypes#SKIP_NEXT_CHANGE_LOGGING} is set on this
	 * instance. This flag will be removed when this entity is stored with
	 * {@link EntityManager#storeEntity(Entity, Entity, boolean)}.
	 *
	 * @return TRUE if history change logging is enabled for this entity
	 */
	public boolean hasChangeLogging() {
		return !hasFlag(SKIP_NEXT_CHANGE_LOGGING);
	}

	/**
	 * Checks whether a certain extra attribute has been set on this entity.
	 *
	 * @param key The key that identifies the attribute
	 * @return TRUE if the attribute exists
	 * @throws StorageException If retrieving the extra attributes of this
	 *                          entity fails
	 */
	public boolean hasExtraAttribute(RelationType<?> key)
		throws StorageException {
		return getExtraAttributeMap().containsKey(key.getSimpleName());
	}

	/**
	 * A convenience method to check a boolean extra attribute.
	 *
	 * @param key The key of the boolean extra attribute
	 * @return TRUE only if the extra attribute exists and has the value TRUE
	 * @throws StorageException If reading the extra attributes fails
	 */
	@SuppressWarnings("boxing")
	public boolean hasExtraAttributeFlag(RelationType<Boolean> key)
		throws StorageException {
		return getExtraAttribute(key, false);
	}

	/**
	 * A shortcut for {@link #hasExtraAttribute(RelationType)} that converts
	 * storage exception into runtime exceptions. Extra attributes should
	 * typically be accessible as their entity has obviously already been
	 * retrieved from the storage so the exception handling is normally
	 * superfluous. Nonetheless developers need to be aware that accessing an
	 * extra attribute also can cause a storage access (at least for the first
	 * XA).
	 *
	 * @see #hasExtraAttribute(RelationType)
	 */
	public boolean hasXA(RelationType<?> key) {
		try {
			return hasExtraAttribute(key);
		} catch (StorageException e) {
			throw new IllegalStateException(e);
		}
	}

	/**
	 * Returns a hash code that is consistent with the {@link #equals(Object)}
	 * implementation.
	 *
	 * @see Object#hashCode()
	 */
	@Override
	public int hashCode() {
		int hashCode = 37 * getClass().hashCode();

		if (hasFlag(MODIFIED)) {
			hashCode = 37 * hashCode + attributesHashCode();
		} else {
			long id = getId();

			hashCode = 37 * hashCode + (int) (id ^ (id >>> 32));
		}

		return hashCode;
	}

	/**
	 * Checks whether this entity is currently cached.
	 *
	 * @return TRUE if this instance is cached
	 */
	public final boolean isCached() {
		return hasFlag(CACHE_ENTITY);
	}

	/**
	 * Checks whether this entity is part of a certain hierarchy. That is the
	 * case if this entity is either a the same as the given parent or a
	 * descendant of it.
	 *
	 * @param hierarchyParent The hierarchy parent to check this entity against
	 * @return TRUE if this entity is a part of the given entity's hierarchy
	 */
	public boolean isHierarchyElement(Entity hierarchyParent) {
		Entity entity = this;

		while (entity != null && entity.getId() != hierarchyParent.getId()) {
			entity = entity.getParent();
		}

		return entity != null;
	}

	/**
	 * Checks whether this entity is already persistent.
	 *
	 * @return TRUE if this instance is persistent
	 */
	public final boolean isModified() {
		return hasFlag(MODIFIED);
	}

	/**
	 * Checks whether this entity is already persistent.
	 *
	 * @return TRUE if this instance is persistent
	 */
	public final boolean isPersistent() {
		return hasFlag(PERSISTENT);
	}

	/**
	 * Checks whether this entity is the root in a hierarchy.
	 *
	 * @return TRUE if this entity is the hierarchy root
	 */
	public boolean isRoot() {
		EntityDefinition<?> definition = getDefinition();
		RelationType<?> parentAttr = definition.getParentAttribute();
		RelationType<?> masterAttr = definition.getMasterAttribute();

		return (parentAttr == null || getAttributeValue(parentAttr) == null) &&
			(masterAttr == null || getAttributeValue(masterAttr) == null);
	}

	/**
	 * Tries to acquire a modification lock on this entity.
	 *
	 * @return TRUE if the lock could be acquired, FALSE if the entity is
	 * already locked
	 */
	public boolean lock() {
		boolean success;

		try {
			// this method doesn't need to be synchronized because
			// beginEntityModification already is; if it returns successfully
			// it will prevent parallel executions of lock() from succeeding
			EntityManager.beginEntityModification(this);
			success = true;
			set(LOCKED);
		} catch (ConcurrentEntityModificationException e) {
			success = false;
		}

		return success;
	}

	/**
	 * Prints this entity and the hierarchy of it's children to a print stream.
	 * This method is intended to be used for debugging and informational
	 * purposes only. The format of the output may change any time.
	 *
	 * @param out The stream to print to
	 */
	public void printHierarchy(PrintStream out) {
		printHierarchy(out, null);
	}

	/**
	 * A convenience method that removes all children of a certain type and
	 * handles the necessary defensive copying of the child list.
	 *
	 * @param childAttr The child attribute to remove all children of
	 */
	public <C extends Entity> void removeAllChildren(
		RelationType<List<C>> childAttr) {
		List<C> children = get(childAttr);

		if (children != null && children.size() > 0) {
			removeChildren(childAttr, new ArrayList<>(children));
		}
	}

	/**
	 * Convenience method to remove a single child from this entity.
	 *
	 * @see #removeChildren(RelationType, List)
	 */
	public <C extends Entity> void removeChild(RelationType<List<C>> childAttr,
		C child) {
		removeChildren(childAttr, Collections.singletonList(child));
	}

	/**
	 * Removes child entities from this entity. This will set the parent
	 * references in the children to NULL. It is recommended that entity
	 * children are always removed through this method unless an entity
	 * subclass
	 * implements (and documents) more specific methods for the management of
	 * child entities. ATTENTION: it is important that the children to be
	 * removed are either actual children of this entity or are equals
	 * according
	 * to the implementation of {@link #equals(Object)}.
	 *
	 * <p>To perform an update of the detached children that are no longer
	 * contained in this instances list of children they will be added to the
	 * relation {@link EntityRelationTypes#DEPENDENT_STORE_ENTITIES} so that
	 * they will be updated when this entity is stored.</p>
	 *
	 * @param childAttr The attribute that references the child type
	 * @param children  The new child objects to add to this entity
	 * @throws IllegalArgumentException If one of the parameters is invalid or
	 *                                  if one of the children is already
	 *                                  stored
	 *                                  in this entity
	 */
	public <C extends Entity> void removeChildren(
		RelationType<List<C>> childAttr, List<C> children) {
		if (childAttr == null || children == null || children.size() == 0) {
			throw new IllegalArgumentException(
				"Arguments must not be NULL or empty");
		}

		// mark as modified to perform child count update in storage framework
		// do before child list modification to allow the modification tracking
		// in EntityDefinition to prohibit the change by throwing an exception
		if (!hasFlag(INITIALIZING)) {
			set(MODIFIED);
		}

		@SuppressWarnings("unchecked")
		EntityDefinition<Entity> def =
			(EntityDefinition<Entity>) getDefinition();

		def.detachChildren(this, childAttr, children);
		get(childAttr).removeAll(children);
		get(DEPENDENT_STORE_ENTITIES).addAll(children);
		getRelation(childAttr).get(REMOVED_CHILDREN).addAll(children);
	}

	/**
	 * Resets the persistent state and removes all modifications markers on
	 * this
	 * entity by invoking {@link EntityManager#resetEntity(Entity)} on this
	 * instance. This will also restore the persistent state of this entity
	 * including it's children. Invoking code should therefore not longer hold
	 * references to children of this entity but instead read them from the
	 * parent after the reset if necessary.
	 *
	 * <p>This method will only reset</p>
	 *
	 * @throws StorageException If restoring the persistent state fails
	 */
	public void reset() throws StorageException {
		if (isModified()) {
			EntityManager.resetEntity(this);
		}
	}

	/**
	 * Resets all modifications of this entity and it's hierarchy by invoking
	 * {@link #reset()} on this instance and clearing modification markers on
	 * children held by the {@link EntityManager}. This will also restore the
	 * persistent state of this entity including it's children. Invoking code
	 * should therefore not longer hold references to children of this entity
	 * but instead read them from the parent after the reset if necessary.
	 *
	 * @throws StorageException If restoring the persistent state fails
	 */
	public void resetHierachy() throws StorageException {
		resetHierarchyModifications();
		reset();
	}

	/**
	 * Stores an extra attribute of this entity. If an attribute with the given
	 * key already exists it will be updated, otherwise a new attribute will be
	 * created.
	 *
	 * @param key   The key that identifies the attribute
	 * @param value The value of the new attribute
	 * @throws StorageException If retrieving the extra attributes of this
	 *                          entity fails
	 */
	public <T> void setExtraAttribute(RelationType<T> key, T value)
		throws StorageException {
		Map<String, ExtraAttribute> attributes = getExtraAttributeMap();

		ExtraAttribute extraAttribute = attributes.get(key.getSimpleName());

		if (extraAttribute == null && value != null) {
			extraAttribute = new ExtraAttribute();

			extraAttribute.set(ExtraAttribute.ENTITY, this);
			extraAttribute.set(ExtraAttribute.KEY, key);
		}

		if (extraAttribute != null) {
			// set as modified before extra attribute is updated to allow the
			// modification tracking in EntityDefinition to prohibit the change
			// by throwing an exception
			set(MODIFIED);

			// always mark attribute as modified even if value is detected as
			// unmodified by the framework to ensure the saving of mutable
			// values like collections
			extraAttribute.set(MODIFIED);
			set(EXTRA_ATTRIBUTES_MODIFIED);

			extraAttribute.set(ExtraAttribute.VALUE, value);
			attributes.put(key.getSimpleName(), extraAttribute);
		}
	}

	/**
	 * Sets an extra attribute for this entity with a certain owner entity.
	 * Other than {@link #setExtraAttribute(RelationType, Object)} this method
	 * directly stores the new or modified extra attribute to avoid the need
	 * for
	 * caching. Caching is not necessary because owner-specific extra
	 * attributes
	 * are typically used less often than standard extra attributes.
	 *
	 * @param owner        The owner of the extra attribute
	 * @param key          The key that identifies the attribute
	 * @param value        The value of the extra attribute
	 * @param changeOrigin The entity that caused this change
	 * @throws StorageException If querying for an existing extra attribute
	 *                          fails
	 */
	public <T> void setExtraAttributeFor(Entity owner, RelationType<T> key,
		T value, Entity changeOrigin) throws StorageException {
		ExtraAttribute extraAttribute = null;
		boolean addToDependentStoreEntities = true;

		// search for an existing extra attribute that hasn't been stored yet
		if (hasRelation(DEPENDENT_STORE_ENTITIES)) {
			for (Entity entity : get(DEPENDENT_STORE_ENTITIES)) {
				if (entity instanceof ExtraAttribute &&
					entity.get(ExtraAttribute.KEY) == key &&
					entity.get(ExtraAttribute.OWNER) == owner) {
					assert entity.get(ExtraAttribute.ENTITY) == this;

					extraAttribute = (ExtraAttribute) entity;
					addToDependentStoreEntities = false;

					break;
				}
			}
		}

		if (extraAttribute == null) {
			extraAttribute = queryExtraAttributeFor(owner, key);
		}

		if (extraAttribute == null && value != null) {
			extraAttribute = new ExtraAttribute();

			extraAttribute.set(ExtraAttribute.ENTITY, this);
			extraAttribute.set(ExtraAttribute.OWNER, owner);
			extraAttribute.set(ExtraAttribute.KEY, key);
		}

		if (extraAttribute != null) {
			// always mark as modified to ensure the saving of mutable values
			extraAttribute.set(MODIFIED);
			extraAttribute.set(ExtraAttribute.VALUE, value);

			if (addToDependentStoreEntities) {
				get(DEPENDENT_STORE_ENTITIES).add(extraAttribute);
			}
		}
	}

	/**
	 * Sets a certain attribute value in all entities of this entity's
	 * hierarchy, including this parent entity. If a criteria predicate is
	 * given
	 * only the entities for which the predicate evaluation yields TRUE will
	 * have the attribute value set.
	 *
	 * @param attribute The attribute to set
	 * @param value     The attribute value
	 * @param criteria  Optional criteria for the entities affected or NULL for
	 *                  all
	 */
	@SuppressWarnings("boxing")
	public <T> void setHierarchical(RelationType<T> attribute, T value,
		Predicate<? super Entity> criteria) {
		if (criteria == null || criteria.evaluate(this)) {
			set(attribute, value);
		}

		RelationType<List<Entity>> childAttr =
			getDefinition().getHierarchyChildAttribute();

		if (childAttr != null) {
			for (Entity child : get(childAttr)) {
				child.setHierarchical(attribute, value, criteria);
			}
		}
	}

	/**
	 * Makes the hierarchy of this entity immutable by setting the flag
	 * {@link MetaTypes#IMMUTABLE} recursively on this entity and it's children
	 * (but not on other entity relations).
	 */
	public void setHierarchyImmutable() {
		set(MetaTypes.IMMUTABLE);

		for (RelationType<List<Entity>> childAttr :
			getDefinition().getChildAttributes()) {
			if (hasRelation(childAttr)) {
				for (Entity child : get(childAttr)) {
					child.setHierarchyImmutable();
				}
			}
		}
	}

	/**
	 * A shortcut for {@link #setExtraAttribute(RelationType, Object)} that
	 * converts storage exception into runtime exceptions. Extra attributes
	 * should typically be accessible as their entity has obviously already
	 * been
	 * retrieved from the storage so the exception handling is normally
	 * superfluous. Nonetheless developers need to be aware that accessing an
	 * extra attribute also can cause a storage access (at least for the first
	 * XA).
	 *
	 * @see #setExtraAttribute(RelationType, Object)
	 */
	public <T> void setXA(RelationType<T> key, T value) {
		try {
			setExtraAttribute(key, value);
		} catch (StorageException e) {
			throw new IllegalStateException(e);
		}
	}

	/**
	 * Creates a {@link Predicate} from this entity using the current attribute
	 * values and the {@link Predicates#equalTo(Object)} method to construct
	 * it.
	 *
	 * <p>The attribute {@link EntityRelationTypes#ENTITY_ID} is always
	 * ignored!
	 * </p>
	 *
	 * @param ignoreNullValues If TRUE attribute values that are NULL are
	 *                         ignored when building the {@link Predicate}
	 * @return A {@link Predicate} from this entity using the current attribute
	 * values and the {@link Predicates#equalTo(Object)} method.
	 */
	public <E extends Entity> Predicate<E> toPredicate(
		boolean ignoreNullValues) {
		List<RelationType<?>> attributes =
			new ArrayList<RelationType<?>>(getDefinition().getAttributes());

		attributes.remove(EntityRelationTypes.ENTITY_ID);

		return toPredicate(ignoreNullValues,
			attributes.toArray(new RelationType<?>[0]));
	}

	/**
	 * Creates a {@link Predicate} from this entity using the current attribute
	 * values for the given attributes and the
	 * {@link Predicates#equalTo(Object)} method to construct it.
	 *
	 * @param ignoreNullValues If TRUE attribute values that are NULL are
	 *                         ignored when building the {@link Predicate}
	 * @param attributes       The attributes to use.
	 * @return A {@link Predicate} from this entity using the current attribute
	 * values for the given attributes and the
	 * {@link Predicates#equalTo(Object)} method.
	 */
	public <E extends Entity> Predicate<E> toPredicate(boolean ignoreNullValues,
		RelationType<?>... attributes) {
		Predicate<E> predicate = notNull();

		for (RelationType<?> attribute : attributes) {
			if (!ignoreNullValues || get(attribute) != null) {
				predicate =
					predicate.and(attribute.is(equalTo(get(attribute))));
			}
		}

		return predicate;
	}

	/**
	 * Returns a string representation of this entity.
	 *
	 * @return A string describing this entity
	 */
	@Override
	public String toString() {
		Object id = isPersistent() ? get(getIdAttribute()) : "<NEW>";
		String desc = getDescription();
		String result;

		if (desc.length() > 0) {
			result =
				String.format("%s[%s(%s)]", getDefinition().getEntityName(),
					this, id);
		} else {
			result =
				String.format("%s[%s]", getDefinition().getEntityName(), id);
		}

		return result;
	}

	/**
	 * Converts an entity to a string that includes certain attributes.
	 *
	 * @param attributes The attributes to include in the string
	 * @param separator  The separator string between attributes
	 * @return The string for the given display mode
	 */
	public String toString(Collection<RelationType<?>> attributes,
		String separator) {
		StringBuilder result =
			new StringBuilder(getDefinition().getEntityName());

		result.append('[');

		if (attributes != null) {
			for (RelationType<?> attribute : attributes) {
				appendAttribute(attribute, separator, result);
			}

			if (result.length() > 0) {
				result.setLength(result.length() - separator.length());
			}
		}

		result.append(']');

		return result.toString();
	}

	/**
	 * Converts an entity to a string according to a certain display mode.
	 *
	 * @param displayMode The display mode to create the string for
	 * @param separator   The separator string between attributes
	 * @return The string for the given display mode
	 */
	public final String toString(DisplayMode displayMode, String separator) {
		return toString(getDefinition().getDisplayAttributes(displayMode),
			separator);
	}

	/**
	 * Converts an entity to a string that includes certain attributes.
	 *
	 * @param separator  The separator string between attributes
	 * @param attributes The attributes to include in the string
	 * @return The string for the given display mode
	 */
	public String toString(String separator, RelationType<?>... attributes) {
		return toString(Arrays.asList(attributes), separator);
	}

	/**
	 * Tries to release a modification lock on this entity. The lock must have
	 * been acquired previously with {@link #lock()} by the same context
	 * (process, thread).
	 *
	 * @throws ConcurrentEntityModificationException If the entity lock is not
	 *                                               held by the current
	 *                                               context
	 *                                               (which can only happen if
	 *                                               locking calls are handled
	 *                                               incorrectly)
	 */
	public void unlock() {
		EntityManager.endEntityModification(this);
		deleteRelation(LOCKED);
	}

	/**
	 * Checks whether an attribute of this instance has a defined value and
	 * throws an exception if that values is not equal to the given value. If
	 * the current value is not set or NULL no exception is thrown.
	 *
	 * @param attr  The attribute to test
	 * @param value The value to check the attribute value against
	 * @throws IllegalStateException If the attribute value differs from the
	 *                               given value
	 */
	public <T> void verifyAttribute(RelationType<? super T> attr, T value) {
		Object currentValue = this.get(attr);

		if (currentValue != null && !currentValue.equals(value)) {
			String message =
				String.format("%s attribute %s != %s", this, currentValue,
					value);

			throw new IllegalStateException(message);
		}
	}

	/**
	 * Returns a description of this entity. The default implementation returns
	 * the result of {@link EntityFunctions#toString()} but subclasses may
	 * override this method to change or modify the result.
	 *
	 * @return The entity description
	 */
	protected String getDescription() {
		return EntityFunctions.format(this);
	}

	/**
	 * Appends a string description for a certain entity attribute to a string
	 * buffer.
	 *
	 * @param attribute The element to append the description of
	 * @param separator The separator string
	 * @param builder   The string buffer to append to
	 */
	void appendAttribute(RelationType<?> attribute, String separator,
		StringBuilder builder) {
		String value = getAttributeValue(attribute);

		builder.append(attribute.getSimpleName());
		builder.append('=');
		builder.append(value);
		builder.append(separator);
	}

	/**
	 * Internal method to return the extra attributes of this entity. If the
	 * entity attributes haven't been retrieved yet this method will initialize
	 * the attribute map by invoking the private method readExtraAttributes().
	 *
	 * @return The extra attributes
	 * @throws StorageException If retrieving the attributes fails
	 */
	synchronized Map<String, ExtraAttribute> getExtraAttributeMap()
		throws StorageException {
		if (isPersistent() && !hasRelation(EXTRA_ATTRIBUTES_READ)) {
			readExtraAttributes();
		}

		return get(EXTRA_ATTRIBUTE_MAP);
	}

	/**
	 * Appends a JSON string for a certain attribute to a string builder.
	 *
	 * @param changes      The string builder to append to
	 * @param indent       The indentation
	 * @param attr         The name of the attribute to append
	 * @param attrRelation The attribute relation
	 * @param changesOnly  TRUE to include only changed attributes
	 */
	private void appendJsonAttribute(StringBuilder changes, String indent,
		String attr, Relation<?> attrRelation, boolean changesOnly) {
		Object newValue = attrRelation.getTarget();

		boolean updated =
			(changesOnly && attrRelation.hasRelation(PREVIOUS_VALUE));

		if (updated || newValue != null) {
			JsonBuilder json = new JsonBuilder();

			json.appendText(indent);
			json.appendName(attr);

			if (updated) {
				json.appendText("{\n");
				json.appendText(indent);
				json.appendText(JSON_INDENT);
				json.appendText("\"");
				json.appendText(JSON_CHANGE_NEW_VALUE);
				json.appendText("\": ");
			}

			if (newValue instanceof Relatable) {
				newValue = newValue.toString();
			}

			json.append(newValue);

			if (updated) {
				Object prevValue = attrRelation.get(PREVIOUS_VALUE);

				if (prevValue instanceof Relatable) {
					prevValue = prevValue.toString();
				}

				json.appendText(",\n");
				json.appendText(indent);
				json.appendText(JSON_INDENT);
				json.appendText("\"");
				json.appendText(JSON_CHANGE_OLD_VALUE);
				json.appendText("\": ");
				json.append(prevValue);
				json.appendText("\n");
				json.appendText(indent);
				json.appendText("}");
			}

			json.appendText(",\n");
			changes.append(json);
		}
	}

	/**
	 * Appends the JSON representation of this entity's attributes to a string
	 * builder.
	 *
	 * @param changes     The string builder
	 * @param definition  The definition of this entity
	 * @param indent      The indentation
	 * @param changesOnly TRUE to include only changed attributes
	 */
	private void appendJsonAttributes(StringBuilder changes,
		EntityDefinition<?> definition, String indent, boolean changesOnly) {
		for (RelationType<?> attribute : definition.getAttributes()) {
			Relation<?> relation = getRelation(attribute);

			if (relation != null &&
				(!changesOnly || relation.hasRelation(PREVIOUS_VALUE))) {
				appendJsonAttribute(changes, indent, attribute.getSimpleName(),
					relation, changesOnly);
			}
		}
	}

	/**
	 * Appends a JSON string with child entity informations to a string
	 * builder.
	 *
	 * @param changes        The string builder
	 * @param indent         The indentation
	 * @param childAttribute The child attribute
	 * @param changesOnly    TRUE to include only changed attributes
	 */
	private void appendJsonChildren(StringBuilder changes, String indent,
		RelationType<List<Entity>> childAttribute, boolean changesOnly) {
		List<Entity> children = get(childAttribute);
		StringBuilder childChanges = new StringBuilder();
		String childAttr = childAttribute.getSimpleName();

		for (Entity child : children) {
			boolean childChangesOnly = changesOnly && child.isPersistent();

			String childChange = child.toJson(indent, childChangesOnly, false);

			if (!childChange.isEmpty()) {
				childChanges.append(childChange);
				childChanges.append(",\n");
			}
		}

		if (childChanges.length() > 0) {
			childChanges.setLength(childChanges.length() - 2);

			changes.append(indent);
			changes.append('"');
			changes.append(childAttr);
			changes.append("\": [\n");
			changes.append(childChanges);
			changes.append("\n");
			changes.append(indent);
			changes.append("],\n");
		}

		List<Entity> removedChildren =
			getRelation(childAttribute).get(REMOVED_CHILDREN);

		if (!removedChildren.isEmpty()) {
			String subIndent = indent + JSON_INDENT;

			changes.append(indent);
			changes.append("\"");
			changes.append(childAttr);
			changes.append(JSON_REMOVED_CHILDREN_SUFFIX);
			changes.append("\": [\n");

			for (Entity removed : removedChildren) {
				changes.append(removed.toJson(subIndent, false, false));
				changes.append(",\n");
			}

			changes.setLength(changes.length() - 2);
			changes.append('\n');
			changes.append(indent);
			changes.append("],\n");
		}
	}

	/**
	 * Appends the change descriptions for modified extra attributes to a
	 * string
	 * builder.
	 *
	 * @param changes     The string builder
	 * @param indent      The indentation
	 * @param changesOnly TRUE to include only changed attributes
	 */
	private void appendJsonExtraAttribute(StringBuilder changes, String indent,
		boolean changesOnly) {
		for (ExtraAttribute extraAttribute : get(
			EXTRA_ATTRIBUTE_MAP).values()) {
			Relation<Object> extraAttrRelation =
				extraAttribute.getRelation(ExtraAttribute.VALUE);

			if (extraAttribute.hasFlag(MODIFIED) && (!changesOnly ||
				extraAttrRelation.hasRelation(PREVIOUS_VALUE))) {
				String name = extraAttribute.get(ExtraAttribute.KEY).getName();

				appendJsonAttribute(changes, indent, name, extraAttrRelation,
					changesOnly);
			}
		}
	}

	/**
	 * Performs an equality comparison of this instance's attributes with
	 * another entity, including extra attributes and children.
	 *
	 * @param other The entity to compare with
	 * @return TRUE if all attributes are equal
	 */
	private boolean attributesEqual(Entity other) {
		List<RelationType<?>> compareAttributes = getCompareAttributes();

		boolean equal = compareAttributes.equals(other.getCompareAttributes());

		if (equal) {
			for (RelationType<?> attr : compareAttributes) {
				Object value = get(attr);
				Object otherValue = other.get(attr);

				if (value == null && otherValue != null ||
					value != null && !value.equals(otherValue)) {
					equal = false;

					break;
				}
			}

			if (equal) {
				Map<String, ExtraAttribute> extraAttributeMap =
					get(EXTRA_ATTRIBUTE_MAP);

				Map<String, ExtraAttribute> otherExtraAttributeMap =
					other.get(EXTRA_ATTRIBUTE_MAP);

				if (extraAttributeMap != null) {
					equal = extraAttributeMap.equals(otherExtraAttributeMap);
				} else {
					equal = otherExtraAttributeMap == null;
				}
			}
		}

		return equal;
	}

	/**
	 * Calculates the hash code of this entities attributes, including extra
	 * attributes and children.
	 *
	 * @return The attribute hash code
	 */
	private int attributesHashCode() {
		int hashCode = getCompareAttributes().hashCode();

		Map<String, ExtraAttribute> extraAttributeMap =
			get(EXTRA_ATTRIBUTE_MAP);

		if (extraAttributeMap != null) {
			hashCode = 37 * hashCode + extraAttributeMap.hashCode();
		}

		return hashCode;
	}

	/**
	 * Returns the value of an attribute.
	 *
	 * @param attribute The attribute
	 * @return The attribute value
	 */
	@SuppressWarnings("boxing")
	private String getAttributeValue(RelationType<?> attribute) {
		Relation<?> relation = getRelation(attribute);
		Object value = null;

		if (relation != null) {
			value = relation.getTarget();
		}

		if (value == null && relation instanceof IntermediateRelation<?, ?>) {
			value =
				((IntermediateRelation<?, ?>) relation).getIntermediateTarget();
		}

		if (value instanceof Entity) {
			value = ((Entity) value).getId();
		}

		return value != null ?
		       value.toString().replaceAll("[\n\r]", "¶") :
		       null;
	}

	/**
	 * Returns a list of all entity attributes (including extra attributes)
	 * that
	 * can be used for equality comparison by {@link #equals(Object)} and the
	 * hash code calculation by {@link #hashCode()} without causing an endless
	 * recursion.
	 *
	 * @return The list of all comparable (extra) attributes
	 */
	private ArrayList<RelationType<?>> getCompareAttributes() {
		EntityDefinition<?> definition = getDefinition();

		ArrayList<RelationType<?>> attributes =
			new ArrayList<RelationType<?>>(definition.getAttributes());

		attributes.remove(definition.getParentAttribute());

		return attributes;
	}

	/**
	 * Prints this entity and the hierarchy of it's children to a print stream.
	 * This method is intended to be used for debugging and informational
	 * purposes only. The format of the output may change any time.
	 *
	 * @param out    The stream to print to
	 * @param indent The indentation of the output
	 */
	private void printHierarchy(PrintStream out, String indent) {
		EntityDefinition<?> definition = getDefinition();
		String prefix = indent != null ? indent + "+--" : "";

		indent = indent != null ? indent + "   " : "";

		out.println(prefix + toString(definition.getAttributes(), ","));

		for (RelationType<?> childType : definition.getChildAttributes()) {
			@SuppressWarnings("unchecked")
			Collection<Entity> children =
				get((RelationType<Collection<Entity>>) childType);

			for (Entity child : children) {
				child.printHierarchy(out, indent);
			}
		}
	}

	/**
	 * Internal method to query a single extra attribute for a certain owner
	 * entity.
	 *
	 * @param owner The owner entity
	 * @param key   The extra attribute key
	 * @return The matching extra attribute or NULL if none exists
	 * @throws StorageException If the query fails
	 */
	private <T> ExtraAttribute queryExtraAttributeFor(Entity owner,
		RelationType<T> key) throws StorageException {
		ExtraAttribute extraAttribute = null;

		if (isPersistent()) {
			Predicate<Relatable> extraAttr = ExtraAttribute.ENTITY
				.is(equalTo(this))
				.and(ExtraAttribute.OWNER.is(equalTo(owner)))
				.and(ExtraAttribute.KEY.is(equalTo(key)));

			extraAttribute =
				EntityManager.queryEntity(ExtraAttribute.class, extraAttr,
					true);
		}

		return extraAttribute;
	}

	/**
	 * Reads the extra attributes of this instance and stores them internally.
	 * This method is synchronized to prevent concurrent modifications of the
	 * extra attribute management variables.
	 *
	 * @throws StorageException If querying the extra attributes fails
	 */
	private void readExtraAttributes() throws StorageException {
		assert !hasFlag(EXTRA_ATTRIBUTES_MODIFIED) :
			"Invalid state: extra attributes have been modified";

		Predicate<Relatable> extraAttr = ExtraAttribute.ENTITY
			.is(equalTo(this))
			.and(ExtraAttribute.HAS_NO_OWNER);

		List<ExtraAttribute> extraAttributes =
			EntityManager.queryEntities(ExtraAttribute.class, extraAttr,
				Integer.MAX_VALUE);

		Map<String, ExtraAttribute> extraAttributeMap =
			get(EXTRA_ATTRIBUTE_MAP);

		for (ExtraAttribute attribute : extraAttributes) {
			Relation<?> keyRelation =
				attribute.getRelation(ExtraAttribute.KEY);

			String key = null;

			if (keyRelation instanceof IntermediateRelation) {
				key = ((IntermediateRelation<?, ?>) keyRelation)
					.getIntermediateTarget()
					.toString();
			}

			// always check for NULL because the intermediate target will be
			// NULL if the relation has already been resolved
			if (key == null) {
				key = attribute.get(ExtraAttribute.KEY).getSimpleName();
			}

			assert key != null :
				"Undefined extra attribute key for " + attribute;

			extraAttributeMap.put(key, attribute);
		}

		set(EXTRA_ATTRIBUTES_READ);
	}

	/**
	 * Internal method to resets all modifications of this entity's hierarchy.
	 * Used by {@link #resetHierachy()}.
	 *
	 * @throws StorageException If restoring the persistent state fails
	 * @see #resetHierachy()
	 */
	private void resetHierarchyModifications() {
		for (RelationType<List<Entity>> childAttr :
			getDefinition().getChildAttributes()) {
			for (Entity child : get(childAttr)) {
				child.resetHierarchyModifications();

				if (child.isModified()) {
					EntityManager.endEntityModification(child);
				}
			}
		}
	}

	/**
	 * Internal method to create a JSON representation of this entity and it's
	 * hierarchy. If the onlyChanges parameter is TRUE only the modified
	 * attributes will be included.
	 *
	 * @param indent            The indentation of the resulting string
	 * @param changesOnly       TRUE to only included changed attributes
	 * @param includeEntityName TRUE to include the entity name into the
	 *                          generated JSON, FALSE to omit it
	 * @return The resulting string
	 */
	private String toJson(String indent, boolean changesOnly,
		boolean includeEntityName) {
		EntityDefinition<?> definition = getDefinition();
		StringBuilder changes = new StringBuilder();
		String subIndent = indent + JSON_INDENT;

		if (!changesOnly || hasFlag(MODIFIED)) {
			appendJsonAttributes(changes, definition, subIndent, changesOnly);
		}

		if (hasFlag(EXTRA_ATTRIBUTES_MODIFIED)) {
			appendJsonExtraAttribute(changes, subIndent, changesOnly);
		}

		Collection<RelationType<List<Entity>>> childAttributes =
			definition.getChildAttributes();

		if (childAttributes != null) {
			for (RelationType<List<Entity>> childAttr : childAttributes) {
				appendJsonChildren(changes, subIndent, childAttr, changesOnly);
			}
		}

		int length = changes.length();

		if (length > 0 || !isPersistent()) {
			if (length >= 2) {
				// remove trailing ,\n
				changes.setLength(length - 2);
			}

			@SuppressWarnings("boxing")
			Object id = isPersistent() ? getId() : "\"<NEW>\"";

			String description = getDescription();
			String entity =
				String.format("%s{\n%s\"%s\": %s,\n", indent, subIndent,
					JSON_ID_FIELD, id);

			if (includeEntityName) {
				entity =
					String.format("%s%s\"%s\": \"%s\",\n", entity, subIndent,
						JSON_TYPE_FIELD, definition.getEntityName());
			}

			if (description.length() > 0) {
				entity =
					String.format("%s%s\"%s\": \"%s\",\n", entity, subIndent,
						JSON_NAME_FIELD, description);
			}

			changes.insert(0, entity);
			changes.append("\n");
			changes.append(indent);
			changes.append("}");
		}

		return changes.toString();
	}
}
