//++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
// This file is a part of the 'esoco-business' project.
// Copyright 2019 Elmar Sonnenschein, esoco GmbH, Flensburg, Germany
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//	  http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.
//++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
package de.esoco.entity;

import de.esoco.entity.EntityDefinition.DisplayMode;
import de.esoco.lib.expression.Function;
import de.esoco.lib.expression.Predicate;
import de.esoco.lib.property.MutableProperties;
import de.esoco.lib.property.PropertyName;
import de.esoco.lib.property.StringProperties;
import de.esoco.storage.QueryPredicate;
import org.obrel.core.Annotations.RelationTypeNamespace;
import org.obrel.core.RelationType;
import org.obrel.core.RelationTypeModifier;
import org.obrel.core.RelationTypes;
import org.obrel.type.MetaTypes;
import org.obrel.type.StandardTypes;

import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.Set;

import static de.esoco.entity.ExtraAttributes.newExtraAttribute;
import static de.esoco.entity.ExtraAttributes.newOrderedSetExtraAttribute;
import static de.esoco.storage.StorageRelationTypes.STORAGE_DATATYPE;
import static de.esoco.storage.StorageRelationTypes.STORAGE_NAME;
import static org.obrel.core.RelationTypeModifier.FINAL;
import static org.obrel.core.RelationTypes.newFlagType;
import static org.obrel.core.RelationTypes.newInitialValueType;
import static org.obrel.core.RelationTypes.newListType;
import static org.obrel.core.RelationTypes.newLongType;
import static org.obrel.core.RelationTypes.newMapType;
import static org.obrel.core.RelationTypes.newSetType;
import static org.obrel.core.RelationTypes.newType;
import static org.obrel.type.MetaTypes.AUTOGENERATED;
import static org.obrel.type.MetaTypes.CHILD_ATTRIBUTE;
import static org.obrel.type.MetaTypes.OBJECT_ID_ATTRIBUTE;
import static org.obrel.type.MetaTypes.PARENT_ATTRIBUTE;

/**
 * Contains entity-specific relation type declarations and factory methods to
 * create such relation types.
 *
 * @author eso
 */
@RelationTypeNamespace("de.esoco.entity")
public class EntityRelationTypes {

	/**
	 * Enumeration of the possible modes for hierarchical queries. The values
	 * are:
	 *
	 * <ul>
	 *   <li>{@link #NEVER}: never perform a hierarchical query.</li>
	 *   <li>{@link #UNCONSTRAINED}: only perform a hierarchical query if no
	 *     additional constraints have been provided.</li>
	 *   <li>{@link #ALWAYS}: always perform a hierarchical query.</li>
	 * </ul>
	 */
	public enum HierarchicalQueryMode {NEVER, UNCONSTRAINED, ALWAYS}

	/**
	 * Meta-type that marks a relation type that represents the root in a
	 * hierarchy
	 */
	public static final RelationType<Boolean> ROOT_ATTRIBUTE =
		newFlagType(FINAL);

	/**
	 * A flag that indicates that an entity has been placed in the entity cache
	 * of {@link EntityManager}.
	 */
	public static final RelationType<Boolean> CACHE_ENTITY = newFlagType();

	/**
	 * The entity ID of an entity's parent entity.
	 */
	public static final RelationType<Long> PARENT_ENTITY_ID = newLongType();

	/**
	 * The entity ID of an entity's master entity.
	 */
	public static final RelationType<Long> MASTER_ENTITY_ID = newLongType();

	/**
	 * A generic reference to a single entity,
	 */
	public static final RelationType<Entity> ENTITY = entityAttribute();

	/**
	 * An entity class.
	 */
	public static final RelationType<Class<? extends Entity>> ENTITY_CLASS =
		newType();

	/**
	 * A predicate for entities.
	 */
	public static final RelationType<Predicate<? super Entity>>
		ENTITY_PREDICATE = newType();

	/**
	 * A query predicate for entity storage queries.
	 */
	public static final RelationType<QueryPredicate<? extends Entity>>
		ENTITY_QUERY_PREDICATE = newType();

	/**
	 * A previously executed entity query predicate.
	 */
	public static final RelationType<QueryPredicate<? extends Entity>>
		LAST_ENTITY_QUERY_PREDICATE = newType();

	/**
	 * A predicate that defines the sort order of entities
	 */
	public static final RelationType<Predicate<? super Entity>>
		ENTITY_SORT_PREDICATE = newType();

	/**
	 * A flag that will skip the automatic change logging of an entity the next
	 * time it is stored with
	 * {@link EntityManager#storeEntity(Entity, Entity, boolean)}. The flag
	 * will
	 * automatically be removed after the entity has been stored so that it
	 * must
	 * always be set before an entity is stored to skip the creation of
	 * history.
	 */
	public static final RelationType<Boolean> SKIP_NEXT_CHANGE_LOGGING =
		newType();

	/**
	 * A flag that prevents the locking of an entity by
	 * {@link EntityManager} if
	 * it is set on the entity. This is intended to be used for bulk processing
	 * of entities. Applying code must ensure that the flag is removed
	 * afterwards, especially for cached entities. If necessary it must also
	 * ensure that the affected entities are not modified concurrently while
	 * the
	 * locking is disabled (e.g. through
	 * {@link EntityManager#setEntityModificationLock(String, Predicate)}).
	 */
	public static final RelationType<Boolean> NO_ENTITY_LOCKING = newType();

	/**
	 * A flag to display entity references in annotated entities as IDs instead
	 * of describing strings.
	 */
	public static final RelationType<Boolean> DISPLAY_ENTITY_IDS =
		newFlagType();

	/**
	 * A property that contains a list of entity attribute access functions.
	 * These can either be attribute relation types (because relation types
	 * implement the function interface) or more complex functions that access
	 * the entity hierarchy or convert values (for example). This type is not
	 * defined with the standard factory method for list properties to avoid
	 * the
	 * creation of a default value. Accessing a non-existing relation of this
	 * type will therefore return NULL.
	 */
	public static final RelationType<List<Function<? super Entity, ?>>>
		ENTITY_ATTRIBUTES = newType();

	/**
	 * The mode to be used when displaying entity attributes. The default value
	 * is {@link DisplayMode#COMPACT}
	 */
	public static final RelationType<DisplayMode> ENTITY_DISPLAY_MODE =
		newInitialValueType(DisplayMode.COMPACT);

	/**
	 * A relation type to set the {@link HierarchicalQueryMode} on a query
	 * predicate. The default value is {@link HierarchicalQueryMode#NEVER}.
	 */
	public static final RelationType<HierarchicalQueryMode>
		HIERARCHICAL_QUERY_MODE =
		newInitialValueType(HierarchicalQueryMode.NEVER);

	/**
	 * A predicate that defines the roots of a hierarchy. It constrains the
	 * root
	 * entities that are displayed in the hierarchical query modes
	 * {@link HierarchicalQueryMode#ALWAYS ALWAYS} and
	 * {@link HierarchicalQueryMode#UNCONSTRAINED UNCONSTRAINED}. If not set
	 * the
	 * default root criterion PARENT = NULL will be used.
	 */
	public static final RelationType<Predicate<? super Entity>>
		HIERARCHY_ROOT_PREDICATE = newType();

	/**
	 * A predicate that defines the children in a hierarchy. It constrains the
	 * child entities that are displayed in the hierarchical query modes
	 * {@link HierarchicalQueryMode#ALWAYS ALWAYS} and
	 * {@link HierarchicalQueryMode#UNCONSTRAINED UNCONSTRAINED}. If not set
	 * all
	 * children in the hierarchy will be displayed.
	 */
	public static final RelationType<Predicate<? super Entity>>
		HIERARCHY_CHILD_PREDICATE = newType();

	/**
	 * Will contain the entity that caused the storing of an entity during an
	 * invocation of {@link EntityManager#storeEntity(Entity, Entity)}.
	 */
	public static final RelationType<Entity> ENTITY_STORE_ORIGIN =
		entityAttribute();

	/**
	 * A collection of entities that should be stored if the target entity is
	 * stored. Evaluated by
	 * {@link EntityManager#storeEntity(Entity, Entity)} to
	 * store the dependent entities in the order in which they are listed after
	 * the main entity. After the dependent entities have been stored this
	 * relation will be removed from the target.
	 */
	public static final RelationType<List<Entity>> DEPENDENT_STORE_ENTITIES =
		newListType();

	/**
	 * A mapping from global entity IDs to child entities that have been
	 * removed
	 * from a parent. Used internally by the change tracking of the framework.
	 */
	public static final RelationType<List<Entity>> REMOVED_CHILDREN =
		newListType();

	/**
	 * Refers to a set of properties that can be used as a parameter annotation
	 * to define display properties for the rendering in user interfaces.
	 */
	public static final RelationType<MutableProperties> DISPLAY_PROPERTIES =
		newType();

	/**
	 * A string identifier that is set on entities to detect concurrent
	 * modifications of entities.
	 */
	public static final RelationType<String> ENTITY_MODIFICATION_HANDLE =
		newType();

	/**
	 * The set of modified entities that have been successfully updated within
	 * an entity modification context but may need to be reset if the enclosing
	 * transaction failed.
	 */
	public static final RelationType<Set<Entity>> CONTEXT_UPDATED_ENTITIES =
		newSetType(false);

	/**
	 * A mapping from global entity IDs to entities that have been modified
	 * within an entity modification context.
	 */
	public static final RelationType<Map<String, Entity>>
		CONTEXT_MODIFIED_ENTITIES = newMapType(false);

	//- Extra attribute management types

	/**
	 * The default attribute for an auto-incremented long integer entity ID.
	 * This attribute will be added automatically to all entities that do not
	 * have a specific ID attribute. Only used internally by the package.
	 * Applications should always query the actual ID attribute from the entity
	 * definition with the method {@link EntityDefinition#getIdAttribute()}.
	 */
	public static final RelationType<Long> ENTITY_ID = longAutoIdAttribute();

	/**
	 * An entity that is targeted by another entity for a certain purpose.
	 */
	public static final RelationType<Entity> TARGET =
		arbitraryEntityAttribute();

	/**
	 * The date of the last modification of an entity.
	 */
	public static final RelationType<Date> LAST_CHANGE = newType();

	//- Standard entity attributes

	/**
	 * A flag indicating that an entity is locked for (typically concurrent)
	 * modifications. Related application code must be aware of this flag and
	 * prevent modifications if it is set.
	 */
	public static final RelationType<Boolean> MODIFICATION_LOCK =
		newExtraAttribute();

	/**
	 * Entity is hidden from display.
	 */
	public static final RelationType<Boolean> HIDE_ENTITY =
		newExtraAttribute();

	/**
	 * A set of tags that categorize an entity.
	 */
	public static final RelationType<Set<String>> ENTITY_TAGS =
		newOrderedSetExtraAttribute();

	//- Standard extra attributes

	/**
	 * Child entities that have bee removed from a parent.
	 */
	public static final RelationType<List<Entity>> REMOVED_CHILD_ENTITIES =
		newListType();

	/**
	 * Will be set to TRUE after the extra attributes have been read from the
	 * storage.
	 */
	static final RelationType<Boolean> EXTRA_ATTRIBUTES_READ =
		newFlagType(FINAL);

	/**
	 * Will be set to TRUE when extra attributes have been modified.
	 */
	static final RelationType<Boolean> EXTRA_ATTRIBUTES_MODIFIED =
		newFlagType();

	/**
	 * Contains the mapping from extra attribute keys to extra attributes.
	 */
	static final RelationType<Map<String, ExtraAttribute>> EXTRA_ATTRIBUTE_MAP =
		newMapType(false);

	static {
		RelationTypes.init(EntityRelationTypes.class);

		ENTITY_ID.set(STORAGE_NAME, "id");
	}

	/**
	 * Private, only static use.
	 */
	private EntityRelationTypes() {
	}

	/**
	 * A factory method to create a new relation type for entity attributes
	 * that
	 * reference arbitrary entity instances. The storage datatype for such
	 * attributes will be a string that holds the global entity ID.
	 *
	 * @see #entityAttribute(RelationTypeModifier...)
	 */
	public static RelationType<Entity> arbitraryEntityAttribute(
		RelationTypeModifier... flags) {
		return RelationTypes
			.<Entity>newType(flags)
			.annotate(STORAGE_DATATYPE, String.class);
	}

	/**
	 * Factory method that creates an integer relation type for automatically
	 * generated object identifiers. The flags {@link MetaTypes#AUTOGENERATED}
	 * and {@link MetaTypes#OBJECT_ID_ATTRIBUTE} will be set on the returned
	 * type. The initial value of relations with this type will be zero.
	 *
	 * @see RelationTypes#newRelationType(String, Class,
	 * RelationTypeModifier...)
	 */
	public static RelationType<Integer> autoIdAttribute(
		RelationTypeModifier... flags) {
		return idAttribute(flags).annotate(AUTOGENERATED);
	}

	/**
	 * A factory method that creates a new relation type for relations to a
	 * list
	 * of child entities. To prevent initialization cycles the child entity
	 * definition cannot be set directly through this method. Instead, a
	 * special
	 * constructor of {@link EntityDefinition} must be used to define the child
	 * entity relation and set the entity definitions on the attributes.
	 *
	 * @param flags The optional type flags
	 * @return A new relation type instance
	 * @see RelationTypes#newListType(String, Class, RelationTypeModifier...)
	 */
	public static <T extends Entity> RelationType<List<T>> childAttribute(
		RelationTypeModifier... flags) {
		return RelationTypes.<T>newListType(flags).annotate(CHILD_ATTRIBUTE);
	}

	/**
	 * A factory method to create a new relation type for entity attributes
	 * that
	 * reference an entity instance of a specific entity subclass.
	 *
	 * @param flags The optional type flags
	 * @return A new relation type instance
	 * @see RelationTypes#newRelationType(String, Class,
	 * RelationTypeModifier...)
	 */
	public static <T extends Entity> RelationType<T> entityAttribute(
		RelationTypeModifier... flags) {
		return RelationTypes
			.<T>newType(flags)
			.annotate(STORAGE_DATATYPE, long.class);
	}

	/**
	 * A factory method that creates an integer relation type for object
	 * identifiers. The flag {@link MetaTypes#OBJECT_ID_ATTRIBUTE} will be set
	 * on the returned type. The initial value of relations with this type will
	 * be zero.
	 *
	 * @see RelationTypes#newRelationType(String, Class,
	 * RelationTypeModifier...)
	 */
	public static RelationType<Integer> idAttribute(
		RelationTypeModifier... flags) {
		return RelationTypes
			.newIntType(flags)
			.annotate(OBJECT_ID_ATTRIBUTE)
			.annotate(STORAGE_DATATYPE, Integer.class);
	}

	/**
	 * Package-internal initialization.
	 */
	static void init() {
	}

	/**
	 * Factory method that creates a long relation type for automatically
	 * generated object identifiers. The flags {@link MetaTypes#AUTOGENERATED}
	 * and {@link MetaTypes#OBJECT_ID_ATTRIBUTE} will be set on the returned
	 * type. The initial value of relations with this type will be zero.
	 *
	 * @see RelationTypes#newRelationType(String, Class,
	 * RelationTypeModifier...)
	 */
	public static RelationType<Long> longAutoIdAttribute(
		RelationTypeModifier... flags) {
		return longIdAttribute(flags).annotate(AUTOGENERATED);
	}

	/**
	 * A factory method that creates a long relation type for object
	 * identifiers. The flag {@link MetaTypes#OBJECT_ID_ATTRIBUTE} will be set
	 * on the returned type. The initial value of relations with this type will
	 * be zero.
	 *
	 * @see RelationTypes#newRelationType(String, Class,
	 * RelationTypeModifier...)
	 */
	public static RelationType<Long> longIdAttribute(
		RelationTypeModifier... flags) {
		return RelationTypes
			.newLongType(flags)
			.annotate(OBJECT_ID_ATTRIBUTE)
			.annotate(STORAGE_DATATYPE, Long.class);
	}

	/**
	 * A factory method that creates a new relation type for relations to the
	 * parent entity in a hierarchy. To prevent initialization cycles the
	 * entity
	 * definition of the parent attribute cannot be set directly in this
	 * method.
	 * Instead this will be handled by the {@link EntityDefinition} class.
	 *
	 * @param flags The optional type flags
	 * @return A new relation type instance
	 * @see RelationTypes#newRelationType(String, Class,
	 * RelationTypeModifier...)
	 */
	public static <T extends Entity> RelationType<T> parentAttribute(
		RelationTypeModifier... flags) {
		return EntityRelationTypes
			.<T>entityAttribute(flags)
			.annotate(PARENT_ATTRIBUTE);
	}

	/**
	 * Creates a new relation type for the root entity in a hierarchy
	 * similar to
	 * {@link #parentAttribute(RelationTypeModifier...)}.
	 *
	 * @see #parentAttribute(RelationTypeModifier...)
	 */
	public static <T extends Entity> RelationType<T> rootAttribute(
		RelationTypeModifier... flags) {
		return EntityRelationTypes
			.<T>entityAttribute(flags)
			.annotate(ROOT_ATTRIBUTE);
	}

	/**
	 * Convenience method to set a boolean display property.
	 *
	 * @see #setAttributeDisplayProperty(PropertyName, Object, RelationType...)
	 */
	@SuppressWarnings("boxing")
	public static void setAttributeDisplayFlag(PropertyName<Boolean> property,
		RelationType<?>... attributes) {
		setAttributeDisplayProperty(property, true, attributes);
	}

	/**
	 * Convenience method to set an integer display property.
	 *
	 * @see #setAttributeDisplayProperty(PropertyName, Object, RelationType...)
	 */
	public static void setAttributeDisplayProperty(int value,
		PropertyName<Integer> property, RelationType<?>... attributes) {
		setAttributeDisplayProperty(property, Integer.valueOf(value),
			attributes);
	}

	/**
	 * Helper method for subclasses to set a display property on attributes. If
	 * the attribute doesn't have display attributes yet they will be created
	 * and stored in the {@link EntityRelationTypes#DISPLAY_PROPERTIES}
	 * relation.
	 *
	 * <p>ATTENTION: this setting affects all occurrences of the given
	 * attribute. Properties of attributes that are re-used in different
	 * contexts (e.g. relation types defined in {@link StandardTypes} or
	 * similar) should be set with a
	 * {@link EntityDefinition#ATTRIBUTE_DISPLAY_PROPERTIES_FIELD} instead to
	 * prevent side effects.</p>
	 *
	 * @param property   The property to set
	 * @param value      The property values
	 * @param attributes The attributes to set the display property on
	 */
	public static <T> void setAttributeDisplayProperty(PropertyName<T> property,
		T value, RelationType<?>... attributes) {
		for (RelationType<?> attribute : attributes) {
			MutableProperties properties = attribute.get(DISPLAY_PROPERTIES);

			if (properties == null) {
				properties = new StringProperties();
				attribute.set(DISPLAY_PROPERTIES, properties);
			}

			properties.setProperty(property, value);
		}
	}
}
