//++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
// This file is a part of the 'esoco-business' project.
// Copyright 2018 Elmar Sonnenschein, esoco GmbH, Flensburg, Germany
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//	  http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.
//++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
package de.esoco.lib.comm;

import de.esoco.entity.Entity;
import de.esoco.entity.EntityFunctions;
import de.esoco.lib.comm.GraylogEndpoint.SendGraylogMessage;
import de.esoco.lib.comm.GraylogMessage.Level;
import de.esoco.lib.logging.BusinessLogAspect;
import de.esoco.lib.logging.Log;
import de.esoco.lib.logging.LogLevel;
import de.esoco.lib.logging.LogRecord;

import java.util.Collection;
import java.util.Objects;

/**
 * A logging implementation that sends log messages to a Graylog server. The
 * endpoint address of the Graylog server must be set on this instance with the
 * relation type {@link CommunicationRelationTypes#ENDPOINT_ADDRESS} before
 * invoking the {@link #initLogging()} method.
 *
 * @author eso
 */
public class GraylogLogging extends BusinessLogAspect<GraylogMessage> {

	private Endpoint graylogEndpoint;

	private SendGraylogMessage sendGraylogMessage;

	/**
	 * Default constructor.
	 */
	public GraylogLogging() {
	}

	/**
	 * Creates a new instance with a certain Graylog server endpoint address.
	 *
	 * @param graylogEndpointAddress The Graylog server address
	 */
	public GraylogLogging(String graylogEndpointAddress) {
		set(CommunicationRelationTypes.ENDPOINT_ADDRESS,
			Objects.requireNonNull(graylogEndpointAddress));
	}

	/**
	 * Adds the endpoint address to the string representation generated by the
	 * superclass.
	 *
	 * @return The string representation for this instance
	 */
	@Override
	public String toString() {
		return String.format("%s(%s)", super.toString(),
			get(CommunicationRelationTypes.ENDPOINT_ADDRESS));
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	@SuppressWarnings("boxing")
	protected GraylogMessage createLogObject(LogRecord logRecord) {
		LogLevel logLevel = logRecord.getLevel();

		Level graylogLevel;

		switch (logLevel) {
			case DEBUG:
				graylogLevel = Level.DEBUG;
				break;

			case ERROR:
				graylogLevel = Level.ERROR;
				break;

			case FATAL:
				graylogLevel = Level.EMERGENCY;
				break;

			case TRACE:
				graylogLevel = Level.NOTICE;
				break;

			case WARN:
				graylogLevel = Level.WARNING;
				break;

			default:
				graylogLevel = Level.INFORMATIONAL;
		}

		GraylogMessage message =
			new GraylogMessage(graylogLevel, logRecord.getMessage(), null);

		message.set(GraylogMessage.TIMESTAMP, logRecord.getTime());
		message.set(GraylogMessage._FILE_NAME, logRecord.getSourceFileName());
		message.set(GraylogMessage._LINE_NUMBER, logRecord.getLineNumber());

		Entity logSource = getLogSource();

		if (logSource != null) {
			message.set(GraylogMessage._ORIGIN,
				EntityFunctions.format(logSource));
		}

		if (logLevel.compareTo(get(MIN_STACK_LOG_LEVEL)) >= 0) {
			message.set(GraylogMessage.FULL_MESSAGE,
				Log.CAUSE_TRACE.evaluate(logRecord));
		}

		return message;
	}

	/**
	 * Creates the target endpoint from the endpoint address that must be
	 * stored
	 * in {@link CommunicationRelationTypes#ENDPOINT_ADDRESS}.
	 *
	 * @see BusinessLogAspect#init()
	 */
	@Override
	protected void init() {
		String endpointAddress =
			get(CommunicationRelationTypes.ENDPOINT_ADDRESS);

		Objects.requireNonNull(endpointAddress);

		graylogEndpoint = Endpoint.at(endpointAddress);
		sendGraylogMessage = GraylogEndpoint.sendMessage();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	protected void processLogObjects(Collection<GraylogMessage> messages) {
		try (Connection connection = graylogEndpoint.connect(this)) {
			for (GraylogMessage message : messages) {
				sendGraylogMessage.sendTo(connection, message);
			}
		}
	}
}
