package de.flapdoodle.embed.mongo.client;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;

/**
 * Immutable implementation of {@link AuthenticationSetup}.
 * <p>
 * Use the builder to create immutable instances:
 * {@code ImmutableAuthenticationSetup.builder()}.
 * Use the static factory method to create immutable instances:
 * {@code ImmutableAuthenticationSetup.of()}.
 */
@SuppressWarnings({"all"})
public final class ImmutableAuthenticationSetup extends AuthenticationSetup {
  private final UsernamePassword admin;
  private final List<AuthenticationSetup.Entry> entries;

  private ImmutableAuthenticationSetup(UsernamePassword admin) {
    this.admin = Objects.requireNonNull(admin, "admin");
    this.entries = createUnmodifiableList(false, createSafeList(super.entries(), true, false));
  }

  private ImmutableAuthenticationSetup(ImmutableAuthenticationSetup.Builder builder) {
    this.admin = builder.admin;
    this.entries = builder.entriesIsSet()
        ? createUnmodifiableList(true, builder.entries)
        : createUnmodifiableList(false, createSafeList(super.entries(), true, false));
  }

  private ImmutableAuthenticationSetup(
      UsernamePassword admin,
      List<AuthenticationSetup.Entry> entries) {
    this.admin = admin;
    this.entries = entries;
  }

  /**
   * @return The value of the {@code admin} attribute
   */
  @Override
  protected UsernamePassword admin() {
    return admin;
  }

  /**
   * @return The value of the {@code entries} attribute
   */
  @Override
  protected List<AuthenticationSetup.Entry> entries() {
    return entries;
  }

  /**
   * Copy the current immutable object by setting a value for the {@link AuthenticationSetup#admin() admin} attribute.
   * A shallow reference equality check is used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for admin
   * @return A modified copy of the {@code this} object
   */
  public final ImmutableAuthenticationSetup withAdmin(UsernamePassword value) {
    if (this.admin == value) return this;
    UsernamePassword newValue = Objects.requireNonNull(value, "admin");
    return new ImmutableAuthenticationSetup(newValue, this.entries);
  }

  /**
   * Copy the current immutable object with elements that replace the content of {@link AuthenticationSetup#entries() entries}.
   * @param elements The elements to set
   * @return A modified copy of {@code this} object
   */
  public final ImmutableAuthenticationSetup withEntries(AuthenticationSetup.Entry... elements) {
    List<AuthenticationSetup.Entry> newValue = createUnmodifiableList(false, createSafeList(Arrays.asList(elements), true, false));
    return new ImmutableAuthenticationSetup(this.admin, newValue);
  }

  /**
   * Copy the current immutable object with elements that replace the content of {@link AuthenticationSetup#entries() entries}.
   * A shallow reference equality check is used to prevent copying of the same value by returning {@code this}.
   * @param elements An iterable of entries elements to set
   * @return A modified copy of {@code this} object
   */
  public final ImmutableAuthenticationSetup withEntries(Iterable<? extends AuthenticationSetup.Entry> elements) {
    if (this.entries == elements) return this;
    List<AuthenticationSetup.Entry> newValue = createUnmodifiableList(false, createSafeList(elements, true, false));
    return new ImmutableAuthenticationSetup(this.admin, newValue);
  }

  /**
   * This instance is equal to all instances of {@code ImmutableAuthenticationSetup} that have equal attribute values.
   * @return {@code true} if {@code this} is equal to {@code another} instance
   */
  @Override
  public boolean equals(Object another) {
    if (this == another) return true;
    return another instanceof ImmutableAuthenticationSetup
        && equalTo(0, (ImmutableAuthenticationSetup) another);
  }

  private boolean equalTo(int synthetic, ImmutableAuthenticationSetup another) {
    return admin.equals(another.admin)
        && entries.equals(another.entries);
  }

  /**
   * Computes a hash code from attributes: {@code admin}, {@code entries}.
   * @return hashCode value
   */
  @Override
  public int hashCode() {
    int h = 5381;
    h += (h << 5) + admin.hashCode();
    h += (h << 5) + entries.hashCode();
    return h;
  }

  /**
   * Prints the immutable value {@code AuthenticationSetup} with attribute values.
   * @return A string representation of the value
   */
  @Override
  public String toString() {
    return "AuthenticationSetup{"
        + "admin=" + admin
        + ", entries=" + entries
        + "}";
  }

  /**
   * Construct a new immutable {@code AuthenticationSetup} instance.
   * @param admin The value for the {@code admin} attribute
   * @return An immutable AuthenticationSetup instance
   */
  public static ImmutableAuthenticationSetup of(UsernamePassword admin) {
    return new ImmutableAuthenticationSetup(admin);
  }

  /**
   * Creates an immutable copy of a {@link AuthenticationSetup} value.
   * Uses accessors to get values to initialize the new immutable instance.
   * If an instance is already immutable, it is returned as is.
   * @param instance The instance to copy
   * @return A copied immutable AuthenticationSetup instance
   */
  public static ImmutableAuthenticationSetup copyOf(AuthenticationSetup instance) {
    if (instance instanceof ImmutableAuthenticationSetup) {
      return (ImmutableAuthenticationSetup) instance;
    }
    return ImmutableAuthenticationSetup.builder()
        .from(instance)
        .build();
  }

  /**
   * Creates a builder for {@link ImmutableAuthenticationSetup ImmutableAuthenticationSetup}.
   * <pre>
   * ImmutableAuthenticationSetup.builder()
   *    .admin(de.flapdoodle.embed.mongo.client.UsernamePassword) // required {@link AuthenticationSetup#admin() admin}
   *    .addEntries|addAllEntries(de.flapdoodle.embed.mongo.client.AuthenticationSetup.Entry) // {@link AuthenticationSetup#entries() entries} elements
   *    .build();
   * </pre>
   * @return A new ImmutableAuthenticationSetup builder
   */
  public static ImmutableAuthenticationSetup.Builder builder() {
    return new ImmutableAuthenticationSetup.Builder();
  }

  /**
   * Builds instances of type {@link ImmutableAuthenticationSetup ImmutableAuthenticationSetup}.
   * Initialize attributes and then invoke the {@link #build()} method to create an
   * immutable instance.
   * <p><em>{@code Builder} is not thread-safe and generally should not be stored in a field or collection,
   * but instead used immediately to create instances.</em>
   */
  public static final class Builder {
    private static final long INIT_BIT_ADMIN = 0x1L;
    private static final long OPT_BIT_ENTRIES = 0x1L;
    private long initBits = 0x1L;
    private long optBits;

    private UsernamePassword admin;
    private List<AuthenticationSetup.Entry> entries = new ArrayList<AuthenticationSetup.Entry>();

    private Builder() {
    }

    /**
     * Fill a builder with attribute values from the provided {@code AuthenticationSetup} instance.
     * Regular attribute values will be replaced with those from the given instance.
     * Absent optional values will not replace present values.
     * Collection elements and entries will be added, not replaced.
     * @param instance The instance from which to copy values
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder from(AuthenticationSetup instance) {
      Objects.requireNonNull(instance, "instance");
      this.admin(instance.admin());
      addAllEntries(instance.entries());
      return this;
    }

    /**
     * Initializes the value for the {@link AuthenticationSetup#admin() admin} attribute.
     * @param admin The value for admin 
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder admin(UsernamePassword admin) {
      this.admin = Objects.requireNonNull(admin, "admin");
      initBits &= ~INIT_BIT_ADMIN;
      return this;
    }

    /**
     * Adds one element to {@link AuthenticationSetup#entries() entries} list.
     * @param element A entries element
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder addEntries(AuthenticationSetup.Entry element) {
      this.entries.add(Objects.requireNonNull(element, "entries element"));
      optBits |= OPT_BIT_ENTRIES;
      return this;
    }

    /**
     * Adds elements to {@link AuthenticationSetup#entries() entries} list.
     * @param elements An array of entries elements
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder addEntries(AuthenticationSetup.Entry... elements) {
      for (AuthenticationSetup.Entry element : elements) {
        this.entries.add(Objects.requireNonNull(element, "entries element"));
      }
      optBits |= OPT_BIT_ENTRIES;
      return this;
    }


    /**
     * Sets or replaces all elements for {@link AuthenticationSetup#entries() entries} list.
     * @param elements An iterable of entries elements
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder entries(Iterable<? extends AuthenticationSetup.Entry> elements) {
      this.entries.clear();
      return addAllEntries(elements);
    }

    /**
     * Adds elements to {@link AuthenticationSetup#entries() entries} list.
     * @param elements An iterable of entries elements
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder addAllEntries(Iterable<? extends AuthenticationSetup.Entry> elements) {
      for (AuthenticationSetup.Entry element : elements) {
        this.entries.add(Objects.requireNonNull(element, "entries element"));
      }
      optBits |= OPT_BIT_ENTRIES;
      return this;
    }

    /**
     * Builds a new {@link ImmutableAuthenticationSetup ImmutableAuthenticationSetup}.
     * @return An immutable instance of AuthenticationSetup
     * @throws java.lang.IllegalStateException if any required attributes are missing
     */
    public ImmutableAuthenticationSetup build() {
      if (initBits != 0) {
        throw new IllegalStateException(formatRequiredAttributesMessage());
      }
      return new ImmutableAuthenticationSetup(this);
    }

    private boolean entriesIsSet() {
      return (optBits & OPT_BIT_ENTRIES) != 0;
    }

    private String formatRequiredAttributesMessage() {
      List<String> attributes = new ArrayList<>();
      if ((initBits & INIT_BIT_ADMIN) != 0) attributes.add("admin");
      return "Cannot build AuthenticationSetup, some of required attributes are not set " + attributes;
    }
  }

  private static <T> List<T> createSafeList(Iterable<? extends T> iterable, boolean checkNulls, boolean skipNulls) {
    ArrayList<T> list;
    if (iterable instanceof Collection<?>) {
      int size = ((Collection<?>) iterable).size();
      if (size == 0) return Collections.emptyList();
      list = new ArrayList<>(size);
    } else {
      list = new ArrayList<>();
    }
    for (T element : iterable) {
      if (skipNulls && element == null) continue;
      if (checkNulls) Objects.requireNonNull(element, "element");
      list.add(element);
    }
    return list;
  }

  private static <T> List<T> createUnmodifiableList(boolean clone, List<T> list) {
    switch(list.size()) {
    case 0: return Collections.emptyList();
    case 1: return Collections.singletonList(list.get(0));
    default:
      if (clone) {
        return Collections.unmodifiableList(new ArrayList<>(list));
      } else {
        if (list instanceof ArrayList<?>) {
          ((ArrayList<?>) list).trimToSize();
        }
        return Collections.unmodifiableList(list);
      }
    }
  }
}
