package de.flapdoodle.embed.mongo.config;

import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.Optional;

/**
 * Immutable implementation of {@link Net}.
 * <p>
 * Use the builder to create immutable instances:
 * {@code ImmutableNet.builder()}.
 */
@SuppressWarnings({"all"})
public final class ImmutableNet extends Net {
  private final String bindIp;
  private final int port;
  private final boolean isIpv6;

  private ImmutableNet(String bindIp, int port, boolean isIpv6) {
    this.bindIp = bindIp;
    this.port = port;
    this.isIpv6 = isIpv6;
  }

  /**
   * @return The value of the {@code bindIp} attribute
   */
  @Override
  public Optional<String> getBindIp() {
    return Optional.ofNullable(bindIp);
  }

  /**
   * @return The value of the {@code port} attribute
   */
  @Override
  public int getPort() {
    return port;
  }

  /**
   * @return The value of the {@code isIpv6} attribute
   */
  @Override
  public boolean isIpv6() {
    return isIpv6;
  }

  /**
   * Copy the current immutable object by setting a <i>present</i> value for the optional {@link Net#getBindIp() bindIp} attribute.
   * @param value The value for bindIp
   * @return A modified copy of {@code this} object
   */
  public final ImmutableNet withBindIp(String value) {
    String newValue = Objects.requireNonNull(value, "bindIp");
    if (Objects.equals(this.bindIp, newValue)) return this;
    return new ImmutableNet(newValue, this.port, this.isIpv6);
  }

  /**
   * Copy the current immutable object by setting an optional value for the {@link Net#getBindIp() bindIp} attribute.
   * An equality check is used on inner nullable value to prevent copying of the same value by returning {@code this}.
   * @param optional A value for bindIp
   * @return A modified copy of {@code this} object
   */
  public final ImmutableNet withBindIp(Optional<String> optional) {
    String value = optional.orElse(null);
    if (Objects.equals(this.bindIp, value)) return this;
    return new ImmutableNet(value, this.port, this.isIpv6);
  }

  /**
   * Copy the current immutable object by setting a value for the {@link Net#getPort() port} attribute.
   * A value equality check is used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for port
   * @return A modified copy of the {@code this} object
   */
  public final ImmutableNet withPort(int value) {
    if (this.port == value) return this;
    return new ImmutableNet(this.bindIp, value, this.isIpv6);
  }

  /**
   * Copy the current immutable object by setting a value for the {@link Net#isIpv6() isIpv6} attribute.
   * A value equality check is used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for isIpv6
   * @return A modified copy of the {@code this} object
   */
  public final ImmutableNet withIsIpv6(boolean value) {
    if (this.isIpv6 == value) return this;
    return new ImmutableNet(this.bindIp, this.port, value);
  }

  /**
   * This instance is equal to all instances of {@code ImmutableNet} that have equal attribute values.
   * @return {@code true} if {@code this} is equal to {@code another} instance
   */
  @Override
  public boolean equals(Object another) {
    if (this == another) return true;
    return another instanceof ImmutableNet
        && equalTo(0, (ImmutableNet) another);
  }

  private boolean equalTo(int synthetic, ImmutableNet another) {
    return Objects.equals(bindIp, another.bindIp)
        && port == another.port
        && isIpv6 == another.isIpv6;
  }

  /**
   * Computes a hash code from attributes: {@code bindIp}, {@code port}, {@code isIpv6}.
   * @return hashCode value
   */
  @Override
  public int hashCode() {
    int h = 5381;
    h += (h << 5) + Objects.hashCode(bindIp);
    h += (h << 5) + port;
    h += (h << 5) + Boolean.hashCode(isIpv6);
    return h;
  }

  /**
   * Prints the immutable value {@code Net} with attribute values.
   * @return A string representation of the value
   */
  @Override
  public String toString() {
    StringBuilder builder = new StringBuilder("Net{");
    if (bindIp != null) {
      builder.append("bindIp=").append(bindIp);
    }
    if (builder.length() > 4) builder.append(", ");
    builder.append("port=").append(port);
    builder.append(", ");
    builder.append("isIpv6=").append(isIpv6);
    return builder.append("}").toString();
  }

  /**
   * Creates an immutable copy of a {@link Net} value.
   * Uses accessors to get values to initialize the new immutable instance.
   * If an instance is already immutable, it is returned as is.
   * @param instance The instance to copy
   * @return A copied immutable Net instance
   */
  public static ImmutableNet copyOf(Net instance) {
    if (instance instanceof ImmutableNet) {
      return (ImmutableNet) instance;
    }
    return ImmutableNet.builder()
        .from(instance)
        .build();
  }

  /**
   * Creates a builder for {@link ImmutableNet ImmutableNet}.
   * <pre>
   * ImmutableNet.builder()
   *    .bindIp(String) // optional {@link Net#getBindIp() bindIp}
   *    .port(int) // required {@link Net#getPort() port}
   *    .isIpv6(boolean) // required {@link Net#isIpv6() isIpv6}
   *    .build();
   * </pre>
   * @return A new ImmutableNet builder
   */
  public static ImmutableNet.Builder builder() {
    return new ImmutableNet.Builder();
  }

  /**
   * Builds instances of type {@link ImmutableNet ImmutableNet}.
   * Initialize attributes and then invoke the {@link #build()} method to create an
   * immutable instance.
   * <p><em>{@code Builder} is not thread-safe and generally should not be stored in a field or collection,
   * but instead used immediately to create instances.</em>
   */
  public static final class Builder {
    private static final long INIT_BIT_PORT = 0x1L;
    private static final long INIT_BIT_IS_IPV6 = 0x2L;
    private long initBits = 0x3L;

    private String bindIp;
    private int port;
    private boolean isIpv6;

    private Builder() {
    }

    /**
     * Fill a builder with attribute values from the provided {@code Net} instance.
     * Regular attribute values will be replaced with those from the given instance.
     * Absent optional values will not replace present values.
     * @param instance The instance from which to copy values
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder from(Net instance) {
      Objects.requireNonNull(instance, "instance");
      Optional<String> bindIpOptional = instance.getBindIp();
      if (bindIpOptional.isPresent()) {
        bindIp(bindIpOptional);
      }
      this.port(instance.getPort());
      this.isIpv6(instance.isIpv6());
      return this;
    }

    /**
     * Initializes the optional value {@link Net#getBindIp() bindIp} to bindIp.
     * @param bindIp The value for bindIp
     * @return {@code this} builder for chained invocation
     */
    public final Builder bindIp(String bindIp) {
      this.bindIp = Objects.requireNonNull(bindIp, "bindIp");
      return this;
    }

    /**
     * Initializes the optional value {@link Net#getBindIp() bindIp} to bindIp.
     * @param bindIp The value for bindIp
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder bindIp(Optional<String> bindIp) {
      this.bindIp = bindIp.orElse(null);
      return this;
    }

    /**
     * Initializes the value for the {@link Net#getPort() port} attribute.
     * @param port The value for port 
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder port(int port) {
      this.port = port;
      initBits &= ~INIT_BIT_PORT;
      return this;
    }

    /**
     * Initializes the value for the {@link Net#isIpv6() isIpv6} attribute.
     * @param isIpv6 The value for isIpv6 
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder isIpv6(boolean isIpv6) {
      this.isIpv6 = isIpv6;
      initBits &= ~INIT_BIT_IS_IPV6;
      return this;
    }

    /**
     * Builds a new {@link ImmutableNet ImmutableNet}.
     * @return An immutable instance of Net
     * @throws java.lang.IllegalStateException if any required attributes are missing
     */
    public ImmutableNet build() {
      if (initBits != 0) {
        throw new IllegalStateException(formatRequiredAttributesMessage());
      }
      return new ImmutableNet(bindIp, port, isIpv6);
    }

    private String formatRequiredAttributesMessage() {
      List<String> attributes = new ArrayList<>();
      if ((initBits & INIT_BIT_PORT) != 0) attributes.add("port");
      if ((initBits & INIT_BIT_IS_IPV6) != 0) attributes.add("isIpv6");
      return "Cannot build Net, some of required attributes are not set " + attributes;
    }
  }
}
