/*
 * Decompiled with CFR 0.152.
 */
package de.governikus.panstar.sdk.utils.crypto;

import de.governikus.panstar.sdk.utils.constant.Common;
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.net.MalformedURLException;
import java.net.URL;
import java.security.Key;
import java.security.KeyFactory;
import java.security.KeyPair;
import java.security.KeyStore;
import java.security.KeyStoreException;
import java.security.NoSuchAlgorithmException;
import java.security.PrivateKey;
import java.security.UnrecoverableKeyException;
import java.security.cert.Certificate;
import java.security.cert.CertificateException;
import java.security.cert.CertificateFactory;
import java.security.cert.X509Certificate;
import java.security.spec.InvalidKeySpecException;
import java.security.spec.PKCS8EncodedKeySpec;
import java.util.Optional;
import org.apache.commons.lang3.ArrayUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class KeystoreLoader {
    private static final Logger LOG = LoggerFactory.getLogger(KeystoreLoader.class);
    public static final String PKCS_12 = "PKCS12";
    public static final String JKS = "JKS";

    private KeystoreLoader() {
    }

    public static Optional<KeyStore> loadKeystore(String path, String type, String password) {
        block11: {
            Optional<KeyStore> optional;
            block10: {
                InputStream inputStream = KeystoreLoader.constructURL(path).openStream();
                try {
                    KeyStore keyStore = KeyStore.getInstance(type);
                    keyStore.load(inputStream, password.toCharArray());
                    optional = Optional.of(keyStore);
                    if (inputStream == null) break block10;
                }
                catch (Throwable throwable) {
                    try {
                        if (inputStream != null) {
                            try {
                                inputStream.close();
                            }
                            catch (Throwable throwable2) {
                                throwable.addSuppressed(throwable2);
                            }
                        }
                        throw throwable;
                    }
                    catch (KeyStoreException e) {
                        LOG.debug("Can not get a key store from instance {}", (Object)type, (Object)e);
                        break block11;
                    }
                    catch (NoSuchAlgorithmException e) {
                        LOG.debug("Can not load key store from path {}. Algorithm is unknown", (Object)path, (Object)e);
                        break block11;
                    }
                    catch (IOException | CertificateException e) {
                        LOG.debug("Can not load key store from path {}", (Object)path, (Object)e);
                    }
                }
                inputStream.close();
            }
            return optional;
        }
        LOG.debug("No key store for path {} loaded. Key store is null", (Object)path);
        return Optional.empty();
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    public static Optional<KeyStore> loadKeystore(InputStream keystoreInputStream, String type, String password) {
        try (InputStream inputStream = keystoreInputStream;){
            if (keystoreInputStream == null || keystoreInputStream.available() == 0) {
                LOG.debug("Can not load key store from inputStream. InputStream is null.");
                Optional<KeyStore> optional2 = Optional.empty();
                return optional2;
            }
            KeyStore keyStore = KeyStore.getInstance(type);
            keyStore.load(inputStream, password.toCharArray());
            Optional<KeyStore> optional = Optional.of(keyStore);
            return optional;
        }
        catch (KeyStoreException e) {
            LOG.debug("Can not get a key store from instance {}", (Object)type, (Object)e);
        }
        catch (NoSuchAlgorithmException e) {
            LOG.debug("Can not load key store from inputStream. Algorithm is unknown", (Throwable)e);
        }
        catch (IOException | CertificateException e) {
            LOG.debug("Can not load key store from inputStream", (Throwable)e);
        }
        LOG.debug("No key store for inputStream loaded.");
        return Optional.empty();
    }

    public static Optional<PrivateKey> loadPrivateKeyFromKeystore(String path, String type, String storePassword, String alias, String keyPassword) {
        Optional<KeyStore> ks = KeystoreLoader.loadKeystore(path, type, storePassword);
        return KeystoreLoader.loadPrivateKeyInternal(alias, keyPassword, ks);
    }

    public static Optional<PrivateKey> loadPrivateKeyFromKeystore(InputStream keystoreInputStream, String type, String storePassword, String alias, String keyPassword) {
        Optional<KeyStore> ks = KeystoreLoader.loadKeystore(keystoreInputStream, type, storePassword);
        return KeystoreLoader.loadPrivateKeyInternal(alias, keyPassword, ks);
    }

    private static Optional<PrivateKey> loadPrivateKeyInternal(String alias, String keyPassword, Optional<KeyStore> ks) {
        if (ks.isEmpty()) {
            return Optional.empty();
        }
        try {
            return Optional.ofNullable((PrivateKey)ks.get().getKey(alias, keyPassword.toCharArray()));
        }
        catch (KeyStoreException | NoSuchAlgorithmException | UnrecoverableKeyException e) {
            LOG.debug("Can not load key {} from keystore", (Object)alias, (Object)e);
            return Optional.empty();
        }
    }

    public static Optional<X509Certificate> loadX509CertificateFromKeystore(String path, String type, String storePassword, String alias) {
        Optional<KeyStore> ks = KeystoreLoader.loadKeystore(path, type, storePassword);
        return KeystoreLoader.loadX509CertificateInternal(alias, ks);
    }

    public static Optional<X509Certificate> loadX509CertificateFromKeystore(InputStream keystoreInputStream, String type, String storePassword, String alias) {
        Optional<KeyStore> ks = KeystoreLoader.loadKeystore(keystoreInputStream, type, storePassword);
        return KeystoreLoader.loadX509CertificateInternal(alias, ks);
    }

    private static Optional<X509Certificate> loadX509CertificateInternal(String alias, Optional<KeyStore> ks) {
        if (ks.isEmpty()) {
            return Optional.empty();
        }
        try {
            return Optional.ofNullable((X509Certificate)ks.get().getCertificate(alias));
        }
        catch (KeyStoreException e) {
            LOG.debug("Can not load certificate {} from keystore", (Object)alias, (Object)e);
            return Optional.empty();
        }
    }

    public static Optional<KeyPair> loadKeyPairFromKeyStore(KeyStore keyStore, String alias, String keyPassword) {
        if (keyStore == null) {
            LOG.debug("Key store cannot be null");
            return Optional.empty();
        }
        try {
            Certificate certificate = keyStore.getCertificate(alias);
            if (certificate == null) {
                LOG.debug("Cannot load certificate from key store. No certificate with alias {} present in key store", (Object)alias);
                return Optional.empty();
            }
            Key key = keyStore.getKey(alias, keyPassword == null ? null : keyPassword.toCharArray());
            if (key == null) {
                LOG.debug("Cannot load key from key store. No key entry with alias {} present in key store", (Object)alias);
                return Optional.empty();
            }
            LOG.debug("Key pair with alias {} successfully loaded from key store", (Object)alias);
            return Optional.of(new KeyPair(certificate.getPublicKey(), (PrivateKey)key));
        }
        catch (KeyStoreException e) {
            LOG.debug("Keystore has not been initialized (loaded). Cannot load key pair from key store", (Throwable)e);
        }
        catch (NoSuchAlgorithmException e) {
            LOG.debug("Algorithm for recovering the key cannot be found", (Throwable)e);
        }
        catch (UnrecoverableKeyException e) {
            LOG.debug("Key cannot be recovered", (Throwable)e);
        }
        return Optional.empty();
    }

    public static Optional<PrivateKey> loadPrivateKey(byte[] keyBytes, String alg) {
        try {
            return Optional.of(KeyFactory.getInstance(alg, Common.BOUNCY_PROVIDER).generatePrivate(new PKCS8EncodedKeySpec(keyBytes)));
        }
        catch (NoSuchAlgorithmException | InvalidKeySpecException e) {
            LOG.debug("Key cannot be loaded", (Throwable)e);
            return Optional.empty();
        }
    }

    public static Optional<X509Certificate> loadX509Certificate(byte[] certBytes) {
        if (ArrayUtils.isEmpty((byte[])certBytes)) {
            LOG.debug("Certificate bytes cannot be null");
            return Optional.empty();
        }
        return KeystoreLoader.loadX509Certificate(new ByteArrayInputStream(certBytes));
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    public static Optional<X509Certificate> loadX509Certificate(InputStream certificateStream) {
        try (InputStream inputStream = certificateStream;){
            if (certificateStream == null || certificateStream.available() == 0) {
                LOG.debug("Can not generate certificate from input stream. Input stream is empty");
                Optional<X509Certificate> optional2 = Optional.empty();
                return optional2;
            }
            Optional<X509Certificate> optional = Optional.of((X509Certificate)CertificateFactory.getInstance("X509", Common.BOUNCY_PROVIDER).generateCertificate(inputStream));
            return optional;
        }
        catch (CertificateException e) {
            LOG.debug("Can not generate certificate from input stream", (Throwable)e);
            return Optional.empty();
        }
        catch (IOException e) {
            LOG.debug("Can not read input stream from input stream", (Throwable)e);
        }
        return Optional.empty();
    }

    public static Optional<X509Certificate> loadX509Certificate(String path) {
        block10: {
            Optional<X509Certificate> optional;
            block9: {
                InputStream inputStream = KeystoreLoader.constructURL(path).openStream();
                try {
                    optional = Optional.of((X509Certificate)CertificateFactory.getInstance("X509", Common.BOUNCY_PROVIDER).generateCertificate(inputStream));
                    if (inputStream == null) break block9;
                }
                catch (Throwable throwable) {
                    try {
                        if (inputStream != null) {
                            try {
                                inputStream.close();
                            }
                            catch (Throwable throwable2) {
                                throwable.addSuppressed(throwable2);
                            }
                        }
                        throw throwable;
                    }
                    catch (CertificateException e) {
                        LOG.debug("Can not generate certificate from path {}", (Object)path, (Object)e);
                        break block10;
                    }
                    catch (IOException e) {
                        LOG.debug("Can not read input stream from path {}", (Object)path, (Object)e);
                    }
                }
                inputStream.close();
            }
            return optional;
        }
        LOG.debug("No certificate for path {} loaded. Certificate is null", (Object)path);
        return Optional.empty();
    }

    private static URL constructURL(String path) throws MalformedURLException {
        if (path.startsWith("classpath:")) {
            return KeystoreLoader.class.getResource(path.substring(10));
        }
        return new URL(path);
    }

    public static Optional<KeyStore> toKeystore(byte[] key, byte[] certificate, String alias, String password, String keyStoreType) {
        Optional<X509Certificate> cert = KeystoreLoader.loadX509Certificate(certificate);
        if (cert.isEmpty()) {
            LOG.debug("Can not create keystore. Certificate not available.");
            return Optional.empty();
        }
        Optional<PrivateKey> privateKey = KeystoreLoader.loadPrivateKey(key, cert.get().getPublicKey().getAlgorithm());
        if (privateKey.isEmpty()) {
            LOG.debug("Can not create keystore. Key not available.");
            return Optional.empty();
        }
        Certificate[] certificateChain = new X509Certificate[]{cert.get()};
        try {
            KeyStore javaKeyStore = KeyStore.getInstance(keyStoreType);
            javaKeyStore.load(null, password.toCharArray());
            javaKeyStore.setKeyEntry(alias, privateKey.get(), password.toCharArray(), certificateChain);
            return Optional.of(javaKeyStore);
        }
        catch (NoSuchAlgorithmException e) {
            LOG.debug("Can not create keystore.", (Throwable)e);
        }
        catch (KeyStoreException e) {
            LOG.debug("Can not create keystore. Invalid type {} or password {}", new Object[]{keyStoreType, password, e});
        }
        catch (CertificateException e) {
            LOG.debug("Could not create keystore. Error while loading certificate", (Throwable)e);
        }
        catch (IOException e) {
            LOG.debug("Could not create keystore", (Throwable)e);
        }
        return Optional.empty();
    }

    public static byte[] getBytes(KeyStore keyStore, String password) {
        if (keyStore == null) {
            return new byte[0];
        }
        byte[] keyStoreBytes = null;
        try (ByteArrayOutputStream outputStream = new ByteArrayOutputStream();){
            keyStore.store(outputStream, password.toCharArray());
            keyStoreBytes = outputStream.toByteArray();
        }
        catch (NoSuchAlgorithmException e) {
            LOG.debug("Could not get bytes from keystore. Data integrity algorithm could not be found", (Throwable)e);
        }
        catch (CertificateException e) {
            LOG.debug("Could not get bytes from keystore. Error storing certificate", (Throwable)e);
        }
        catch (IOException e) {
            LOG.debug("Could not get bytes from keystore. IO Error", (Throwable)e);
        }
        catch (KeyStoreException e) {
            LOG.debug("Could not get bytes from keystore. Keystore not properly initialized", (Throwable)e);
        }
        return keyStoreBytes;
    }
}

