/*
 * Decompiled with CFR 0.152.
 */
package de.governikus.panstar.sdk.utils.saml;

import de.governikus.panstar.sdk.utils.constant.Common;
import de.governikus.panstar.sdk.utils.exception.SAMLInternalErrorException;
import java.net.URISyntaxException;
import java.net.URLEncoder;
import java.nio.charset.StandardCharsets;
import java.security.InvalidKeyException;
import java.security.NoSuchAlgorithmException;
import java.security.PrivateKey;
import java.security.Signature;
import java.security.SignatureException;
import java.security.cert.X509Certificate;
import java.util.Base64;
import java.util.zip.DataFormatException;
import java.util.zip.Deflater;
import java.util.zip.Inflater;
import org.apache.commons.lang3.StringUtils;
import org.apache.http.client.utils.URIBuilder;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public final class SAMLUtils {
    private static final Logger LOG = LoggerFactory.getLogger(SAMLUtils.class);
    public static final String PARAM_SAMLREQUEST = "SAMLRequest";
    public static final String PARAM_SIGALG = "SigAlg";
    public static final String PARAM_SIGNATURE = "Signature";
    public static final String PARAM_RELAYSTATE = "RelayState";
    public static final String PARAM_SAMLRESPONSE = "SAMLResponse";

    public static byte[] inflate(String input) throws DataFormatException {
        byte[] b = Base64.getDecoder().decode(input);
        Inflater decompressor = new Inflater(true);
        decompressor.setInput(b);
        byte[] output = new byte[10 * b.length];
        int length = decompressor.inflate(output);
        byte[] result = new byte[length];
        System.arraycopy(output, 0, result, 0, length);
        return result;
    }

    public static String deflate(byte[] input) {
        byte[] output = new byte[2 * input.length];
        Deflater compresser = new Deflater(3, true);
        compresser.setInput(input);
        compresser.finish();
        int compressedDataLength = compresser.deflate(output);
        byte[] result = new byte[compressedDataLength];
        System.arraycopy(output, 0, result, 0, compressedDataLength);
        return Base64.getEncoder().encodeToString(result);
    }

    public static boolean checkQuerySignature(String samlRequestBase64, String relayState, String sigAlgo, String sigValueBase64, X509Certificate sigCert, boolean isRequest) {
        if (samlRequestBase64 == null || sigAlgo == null || sigValueBase64 == null || sigCert == null) {
            return false;
        }
        try {
            String algoName = switch (sigAlgo) {
                case "http://www.w3.org/2007/05/xmldsig-more#sha256-rsa-MGF1" -> "SHA256withRSASSA-PSS";
                case "http://www.w3.org/2007/05/xmldsig-more#sha384-rsa-MGF1" -> "SHA384withRSASSA-PSS";
                case "http://www.w3.org/2007/05/xmldsig-more#sha512-rsa-MGF1" -> "SHA512withRSASSA-PSS";
                case "http://www.w3.org/2001/04/xmldsig-more#ecdsa-sha256" -> "SHA256withECDSA";
                case "http://www.w3.org/2001/04/xmldsig-more#ecdsa-sha384" -> "SHA384withECDSA";
                case "http://www.w3.org/2001/04/xmldsig-more#ecdsa-sha512" -> "SHA512withECDSA";
                default -> throw new UnsupportedOperationException("unsupported signature algorithm " + sigAlgo);
            };
            byte[] value = Base64.getDecoder().decode(sigValueBase64);
            Signature sig = Signature.getInstance(algoName, Common.BOUNCY_PROVIDER);
            sig.initVerify(sigCert);
            String signed = SAMLUtils.getSignedURL(samlRequestBase64, relayState, sigAlgo, isRequest);
            sig.update(signed.getBytes(StandardCharsets.UTF_8));
            return sig.verify(value);
        }
        catch (Exception e) {
            LOG.debug("Ex:", (Throwable)e);
            return false;
        }
    }

    public static String signQueryParameter(String url, boolean isRequest, String saml, String relayState, PrivateKey privateKey, String digestAlgorithm) throws SAMLInternalErrorException {
        try {
            StringBuilder result = new StringBuilder();
            String messageParameter = isRequest ? PARAM_SAMLREQUEST : PARAM_SAMLRESPONSE;
            SAMLUtils.appendParam(result, messageParameter, saml);
            if (StringUtils.isNotBlank((CharSequence)relayState)) {
                SAMLUtils.appendParam(result, PARAM_RELAYSTATE, relayState);
            }
            SAMLUtils.signQuery(result, privateKey, digestAlgorithm, saml, relayState, isRequest);
            URIBuilder uriBuilder = new URIBuilder(url);
            String querySeparator = uriBuilder.isQueryEmpty() ? "?" : "&";
            return url + querySeparator + result;
        }
        catch (URISyntaxException e) {
            throw new SAMLInternalErrorException(e);
        }
    }

    private static void signQuery(StringBuilder result, PrivateKey privateKey, String digestAlg, String saml, String relayState, boolean isRequest) throws SAMLInternalErrorException {
        String sigAlg;
        String algorithm;
        String algoName = privateKey.getAlgorithm();
        if ("RSA".equalsIgnoreCase(algoName)) {
            switch (digestAlg) {
                case "SHA256": 
                case "SHA-256": {
                    algorithm = "http://www.w3.org/2007/05/xmldsig-more#sha256-rsa-MGF1";
                    sigAlg = "SHA256withRSASSA-PSS";
                    break;
                }
                case "SHA-384": 
                case "SHA384": {
                    algorithm = "http://www.w3.org/2007/05/xmldsig-more#sha384-rsa-MGF1";
                    sigAlg = "SHA384withRSASSA-PSS";
                    break;
                }
                case "SHA512": 
                case "SHA-512": {
                    algorithm = "http://www.w3.org/2007/05/xmldsig-more#sha512-rsa-MGF1";
                    sigAlg = "SHA512withRSASSA-PSS";
                    break;
                }
                default: {
                    throw new SAMLInternalErrorException("Given digest algorithm " + digestAlg + " not supported");
                }
            }
        } else if ("EC".equals(algoName)) {
            switch (digestAlg) {
                case "SHA256": 
                case "SHA-256": {
                    algorithm = "http://www.w3.org/2001/04/xmldsig-more#ecdsa-sha256";
                    sigAlg = "SHA256withECDSA";
                    break;
                }
                case "SHA-384": 
                case "SHA384": {
                    algorithm = "http://www.w3.org/2001/04/xmldsig-more#ecdsa-sha384";
                    sigAlg = "SHA384withECDSA";
                    break;
                }
                case "SHA512": 
                case "SHA-512": {
                    algorithm = "http://www.w3.org/2001/04/xmldsig-more#ecdsa-sha512";
                    sigAlg = "SHA512withECDSA";
                    break;
                }
                default: {
                    throw new SAMLInternalErrorException("Given digest algorithm " + digestAlg + " not supported");
                }
            }
        } else {
            throw new SAMLInternalErrorException("Unsupported signature algorithm: " + algoName);
        }
        SAMLUtils.appendParam(result, PARAM_SIGALG, algorithm);
        try {
            Signature signature = privateKey.getClass().getName().contains("P11Key") ? Signature.getInstance(sigAlg) : Signature.getInstance(sigAlg, Common.BOUNCY_PROVIDER);
            signature.initSign(privateKey);
            signature.update(SAMLUtils.getSignedURL(saml, relayState, algorithm, isRequest).getBytes(StandardCharsets.UTF_8));
            byte[] signatureValue = signature.sign();
            SAMLUtils.appendParam(result, PARAM_SIGNATURE, Base64.getEncoder().encodeToString(signatureValue));
        }
        catch (NoSuchAlgorithmException e) {
            throw new SAMLInternalErrorException("Cannot get signature instance: No such Algorithm", e);
        }
        catch (InvalidKeyException e) {
            throw new SAMLInternalErrorException("Cannot initialize signing object: Invalid key", e);
        }
        catch (SignatureException e) {
            throw new SAMLInternalErrorException("Cannot sign query", e);
        }
    }

    private static void appendParam(StringBuilder result, String name, String value) {
        if (!result.isEmpty()) {
            result.append('&');
        }
        result.append(name).append('=').append(URLEncoder.encode(value, StandardCharsets.UTF_8));
    }

    static String getSignedURL(String message, String relayState, String sigAlg, boolean isRequest) {
        StringBuilder signed = new StringBuilder();
        signed.append(isRequest ? PARAM_SAMLREQUEST : PARAM_SAMLRESPONSE).append('=').append(SAMLUtils.urlEncode(message));
        if (StringUtils.isNotBlank((CharSequence)relayState)) {
            signed.append('&').append(PARAM_RELAYSTATE).append('=').append(SAMLUtils.urlEncode(relayState));
        }
        signed.append('&').append(PARAM_SIGALG).append('=').append(SAMLUtils.urlEncode(sigAlg));
        return signed.toString();
    }

    private static String urlEncode(String message) {
        return URLEncoder.encode(message, StandardCharsets.UTF_8);
    }

    private SAMLUtils() {
    }
}

