/*
 * Copyright (c) 2008, intarsys consulting GmbH
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Public License as published by the 
 * Free Software Foundation; either version 3 of the License, 
 * or (at your option) any later version.
 * <p/>
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  
 * 
 */
package de.intarsys.pdf.platform.cwt.adapter.swt;

import java.awt.image.BufferedImage;
import java.lang.ref.SoftReference;
import java.util.HashMap;

import org.eclipse.swt.graphics.Color;
import org.eclipse.swt.graphics.Device;
import org.eclipse.swt.graphics.GC;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.graphics.ImageData;
import org.eclipse.swt.graphics.Pattern;
import org.eclipse.swt.graphics.Transform;

import de.intarsys.cwt.environment.IGraphicsContext;
import de.intarsys.cwt.environment.IGraphicsEnvironment;
import de.intarsys.cwt.swt.environment.CwtSwtGraphicsContext;
import de.intarsys.cwt.swt.environment.CwtSwtGraphicsEnvironment;
import de.intarsys.cwt.swt.image.CwtSwtImage;
import de.intarsys.cwt.swt.image.ImageConverterAwt2Swt;
import de.intarsys.pdf.content.CSException;
import de.intarsys.pdf.pd.PDAxialShading;
import de.intarsys.pdf.pd.PDFunction;
import de.intarsys.pdf.pd.PDImage;
import de.intarsys.pdf.pd.PDTilingPattern;
import de.intarsys.pdf.platform.cwt.adapter.CommonGraphicsEnvironmentAdapter;
import de.intarsys.pdf.platform.cwt.adapter.awt.AwtGraphicsEnvironmentAdapter;
import de.intarsys.pdf.platform.cwt.color.awt.AwtColorSpace;
import de.intarsys.pdf.platform.cwt.color.awt.AwtColorSpaceFactory;
import de.intarsys.pdf.platform.cwt.image.swt.ImageConverterPdf2Swt;
import de.intarsys.pdf.platform.cwt.paint.ColorPaint;
import de.intarsys.pdf.platform.cwt.paint.PatternPaint;
import de.intarsys.pdf.platform.cwt.paint.ShadingPaint;
import de.intarsys.pdf.platform.cwt.rendering.CSPlatformRenderer;
import de.intarsys.tools.attribute.Attribute;

public class SwtGraphicsEnvironmentAdapter extends
		CommonGraphicsEnvironmentAdapter {

	static class ImagePattern extends Pattern {

		private Image image;

		public ImagePattern(Device device, Image pImage) {
			super(device, pImage);
			image = pImage;
		}

		@Override
		public void dispose() {
			super.dispose();
			image.dispose();
		}

	}

	private static SwtGraphicsEnvironmentAdapter ACTIVE;

	private static final Attribute ATTR_EXCEPTION = new Attribute(
			"swtException"); //$NON-NLS-1$
	private static final Attribute ATTR_IMAGE = new Attribute("swtImage"); //$NON-NLS-1$

	public static SwtGraphicsEnvironmentAdapter get() {
		return ACTIVE;
	}

	public SwtGraphicsEnvironmentAdapter() {
		ACTIVE = this;
	}

	synchronized protected ImageData createImage(PDImage pdImage) {
		ImageData result = (ImageData) pdImage.getAttribute(ATTR_IMAGE);
		if (result == null) {
			RuntimeException ex = (RuntimeException) pdImage
					.getAttribute(ATTR_EXCEPTION);
			if (ex != null) {
				throw ex;
			}
			ImageConverterPdf2Swt converter = new ImageConverterPdf2Swt(pdImage);
			try {
				result = converter.getImageData();
				pdImage.setAttribute(ATTR_IMAGE, result);
			} catch (RuntimeException e) {
				pdImage.setAttribute(ATTR_EXCEPTION, e);
				throw e;
			}
		}
		return result;
	}

	protected ImageData createImage(PDImage pdImage, float[] rgb) {
		int red;
		int green;
		int blue;
		String key;
		ImageData image;

		red = (int) (rgb[0] * 255);
		green = (int) (rgb[1] * 255);
		blue = (int) (rgb[2] * 255);

		key = "swt-iimage-" + ((red << 16) | (green << 8) | blue); //$NON-NLS-1$
		SoftReference<ImageData> ref = (SoftReference<ImageData>) pdImage
				.getAttribute(key);
		if (ref != null) {
			image = ref.get();
			if (image != null) {
				return image;
			}
		}
		BufferedImage bufferedImage = AwtGraphicsEnvironmentAdapter.get()
				.createImage(pdImage, rgb);
		ImageConverterAwt2Swt converter = new ImageConverterAwt2Swt(
				bufferedImage);
		image = converter.getImageData();
		pdImage.setAttribute(key, new SoftReference<ImageData>(image));
		return image;
	}

	protected Color createResourceColorPaint(Device device, ColorPaint paint) {
		// os resource; do not cache; caller is responsible for dispose
		org.eclipse.swt.graphics.Color rgbColor;
		AwtColorSpace awtColorSpace = AwtColorSpaceFactory.get()
				.createColorSpace(paint.getPdColorSpace());
		float[] colorValues = paint.getColorValues();
		float[] rgb;
		if (colorValues == null) {
			rgb = new float[] { 0, 0, 0 };
		} else {
			try {
				rgb = awtColorSpace.getRGB(colorValues);
			} catch (RuntimeException ex) {
				// try to provide at least some remotely reasonable values
				rgb = AwtColorSpace.FALLBACK.toRGB(colorValues);
			}
		}
		rgbColor = new org.eclipse.swt.graphics.Color(device,
				(int) (rgb[0] * 255), (int) (rgb[1] * 255),
				(int) (rgb[2] * 255));
		/*
		 * would normally return a transparent gradient pattern if alpha is < 1
		 * but this doesn't work for printing; have to handle alpha elsewhere
		 */
		return rgbColor;
	}

	protected Pattern createResourceDummyPaint(Device device) {
		Pattern pattern;
		// use a half transparent grey as a dummy
		org.eclipse.swt.graphics.Color rgbColor;

		rgbColor = new org.eclipse.swt.graphics.Color(device, 128, 128, 128);
		pattern = new Pattern(device, 0, 0, 1, 0, rgbColor, 128, rgbColor, 128);
		rgbColor.dispose();
		return pattern;
	}

	protected Pattern createResourcePatternPaint(Device device,
			PatternPaint patternPaint) {
		// TODO 2 @ehk
		PDTilingPattern tilingPattern;
		Image image;
		GC gc;
		IGraphicsContext cwtGc;
		CSPlatformRenderer renderer;
		Pattern pattern;

		if (!(patternPaint.getPdPattern() instanceof PDTilingPattern)) {
			return createResourceDummyPaint(device);
		}

		/*
		 * the right way to draw a pattern would be to repeatedly draw its
		 * contents as if the pattern was a form. this here is only an
		 * approximation and won't work for some patterns.
		 */
		tilingPattern = (PDTilingPattern) patternPaint.getPdPattern();
		// live with the possible inaccuracy
		image = new Image(device, (int) tilingPattern.getXStep(),
				(int) tilingPattern.getYStep());

		gc = new GC(image);
		cwtGc = new CwtSwtGraphicsContext(gc, device);
		try {
			cwtGc.scale(1, -1);
			cwtGc.translate(0, 0 - (int) tilingPattern.getYStep());
			// matrix has to be applied elsewhere
			cwtGc.setClip(tilingPattern.getBoundingBox()
					.toNormalizedRectangle());
			renderer = new CSPlatformRenderer(new HashMap(), cwtGc);
			renderer.process(tilingPattern.getContentStream(), tilingPattern
					.getResources());
		} catch (CSException ex) {
			// TODO is exception handler catching this?
		} finally {
			cwtGc.dispose();
			gc.dispose();
		}

		pattern = new ImagePattern(device, image);
		return pattern;
	}

	protected Pattern createResourceShadingPaint(Device device,
			ShadingPaint paint) {
		// TODO 2 @ehk
		PDAxialShading axialShading;
		float[] coords;
		float[] domain;
		Pattern pattern;
		PDFunction function;
		float[] rgb;
		org.eclipse.swt.graphics.Color rgbColor1;
		org.eclipse.swt.graphics.Color rgbColor2;

		if (!(paint.getPdShading() instanceof PDAxialShading)) {
			return createResourceDummyPaint(device);
		}

		axialShading = (PDAxialShading) paint.getPdShading();
		coords = axialShading.getCoords();
		domain = axialShading.getDomain();
		function = axialShading.getFunction();
		AwtColorSpace awtColorSpace = AwtColorSpaceFactory.get()
				.createColorSpace(paint.getPdShading().getColorSpace());
		rgb = awtColorSpace
				.getRGB(function.evaluate(new float[] { domain[0] }));
		rgbColor1 = new org.eclipse.swt.graphics.Color(device,
				(int) (rgb[0] * 255), (int) (rgb[1] * 255),
				(int) (rgb[2] * 255));
		rgb = awtColorSpace
				.getRGB(function.evaluate(new float[] { domain[1] }));
		rgbColor2 = new org.eclipse.swt.graphics.Color(device,
				(int) (rgb[0] * 255), (int) (rgb[1] * 255),
				(int) (rgb[2] * 255));
		pattern = new Pattern(device, coords[0], coords[1], coords[2],
				coords[3], rgbColor1, (int) (paint.getAlphaValue() * 255),
				rgbColor2, (int) (paint.getAlphaValue() * 255));
		rgbColor1.dispose();
		rgbColor2.dispose();
		return pattern;
	}

	public void drawImage(IGraphicsContext graphicsContext, PDImage pdImage,
			float x, float y) {
		Transform original;
		Transform temporary;
		float scaledX;
		float scaledY;
		GC gc;

		ImageData imageData = createImage(pdImage);
		// do some post transformations..
		if (pdImage.isImageMask()) {
			// todo support masked patterns....
			float[] rgb = graphicsContext.getBackgroundColor()
					.getRGBColorComponents(null);
			imageData = createImage(pdImage, rgb);
		}

		float definedWidth = pdImage.getWidth();
		float definedHeight = pdImage.getHeight();
		float realWidth = imageData.width;
		float realHeight = imageData.height;
		scaledX = x;
		scaledY = y;

		original = null;
		temporary = null;
		gc = ((CwtSwtGraphicsContext) graphicsContext).getGc();
		if (definedHeight < realHeight || definedWidth < realWidth) {
			float[] elements;
			float xScale;
			float yScale;

			original = new Transform(gc.getDevice());
			gc.getTransform(original);
			elements = new float[6];
			original.getElements(elements);
			temporary = new Transform(gc.getDevice(), elements);

			xScale = definedWidth / realWidth;
			yScale = definedHeight / realHeight;
			temporary.scale(xScale, yScale);
			scaledX = x / xScale;
			scaledY = y / yScale;
			gc.setTransform(temporary);
			temporary.dispose();
		}

		try {
			new CwtSwtImage(imageData).drawFromGraphicsContext(graphicsContext,
					scaledX, scaledY);
		} finally {
			if (original != null) {
				gc.setTransform(original);
				original.dispose();
			}
		}
	}

	public IGraphicsEnvironment getGraphicsEnvironment() {
		return CwtSwtGraphicsEnvironment.get();
	}

	public void setBackgroundColorPaint(IGraphicsContext graphicsContext,
			ColorPaint paint) {
		CwtSwtGraphicsContext swtGC = (CwtSwtGraphicsContext) graphicsContext;
		Color resource = createResourceColorPaint(swtGC.getDevice(), paint);
		swtGC.setBackgroundColor(resource);
	}

	public void setBackgroundPatternPaint(IGraphicsContext graphicsContext,
			PatternPaint paint) {
		CwtSwtGraphicsContext swtGC = (CwtSwtGraphicsContext) graphicsContext;
		Pattern resource = createResourcePatternPaint(swtGC.getDevice(), paint);
		swtGC.setBackgroundPattern(resource);
	}

	public void setBackgroundShadingPaint(IGraphicsContext graphicsContext,
			ShadingPaint paint) {
		CwtSwtGraphicsContext swtGC = (CwtSwtGraphicsContext) graphicsContext;
		Pattern resource = createResourceShadingPaint(swtGC.getDevice(), paint);
		swtGC.setBackgroundPattern(resource);
	}

	public void setForegroundColorPaint(IGraphicsContext graphicsContext,
			ColorPaint paint) {
		CwtSwtGraphicsContext swtGC = (CwtSwtGraphicsContext) graphicsContext;
		Color resource = createResourceColorPaint(swtGC.getDevice(), paint);
		swtGC.setForegroundColor(resource);
	}

	public void setForegroundPatternPaint(IGraphicsContext graphicsContext,
			PatternPaint paint) {
		CwtSwtGraphicsContext swtGC = (CwtSwtGraphicsContext) graphicsContext;
		Pattern resource = createResourcePatternPaint(swtGC.getDevice(), paint);
		swtGC.setForegroundPattern(resource);
	}

	public void setForegroundShadingPaint(IGraphicsContext graphicsContext,
			ShadingPaint paint) {
		CwtSwtGraphicsContext swtGC = (CwtSwtGraphicsContext) graphicsContext;
		Pattern resource = createResourceShadingPaint(swtGC.getDevice(), paint);
		swtGC.setForegroundPattern(resource);
	}

}
