/*
 * Copyright (c) 2008, intarsys consulting GmbH
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Public License as published by the 
 * Free Software Foundation; either version 3 of the License, 
 * or (at your option) any later version.
 * <p/>
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  
 * 
 */
package de.intarsys.pdf.platform.cwt.font;

import java.util.Iterator;
import java.util.List;

import de.intarsys.pdf.cos.COSDocument;
import de.intarsys.pdf.cos.ICOSDocumentListener;
import de.intarsys.pdf.font.PDFont;
import de.intarsys.pdf.font.PDFontTools;
import de.intarsys.pdf.font.outlet.IFontFactory;
import de.intarsys.pdf.pd.PDDocument;

/**
 * An {@link IFontFactory} that tries to reuse fonts already available in a
 * {@link PDDocument}. If not found, the factory falls back to the
 * {@link StandardFontFactory}, looking up fonts in the system environment.
 * 
 */
public class DocumentFontFactory extends StandardFontFactory {

	/** the document whose fonts are managed */
	private final PDDocument doc;

	private ICOSDocumentListener listenDocumentChange = new ICOSDocumentListener() {

		public void changed(COSDocument document, Object slot, Object oldValue,
				Object newValue) {
			if (slot == COSDocument.SLOT_ALL) {
				reset();
			}
		}

	};

	protected DocumentFontFactory(PDDocument doc) {
		super();
		this.doc = doc;
		doc.cosGetDoc().addDocumentListener(listenDocumentChange);
		readFonts();
	}

	/**
	 * The document whose fonts are managed.
	 * 
	 * @return The document whose fonts are managed.
	 */
	public PDDocument getDoc() {
		return doc;
	}

	/**
	 * Iterate and register all fonts the document contains.
	 */
	protected void readFonts() {
		List fonts = PDFontTools.getFonts(getDoc());
		Iterator it = fonts.iterator();
		while (it.hasNext()) {
			PDFont font = (PDFont) it.next();
			if (isReusable(font)) {
				registerFont(font);
			}
		}
	}

	@Override
	public void reset() {
		super.reset();
		readFonts();
	}
}
