/*
 * Copyright (c) 2008, intarsys consulting GmbH
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Public License as published by the 
 * Free Software Foundation; either version 3 of the License, 
 * or (at your option) any later version.
 * <p/>
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  
 * 
 */
package de.intarsys.pdf.platform.cwt.font.awt;

import java.awt.Font;
import java.awt.geom.AffineTransform;
import java.awt.geom.Rectangle2D;

import de.intarsys.pdf.content.GraphicsState;
import de.intarsys.pdf.font.PDFont;
import de.intarsys.pdf.font.PDFontDescriptor;
import de.intarsys.pdf.font.PDGlyphs;
import de.intarsys.pdf.platform.cwt.font.IPlatformFont;
import de.intarsys.pdf.platform.cwt.font.IPlatformGlyphs;
import de.intarsys.pdf.platform.cwt.font.PlatformFontException;
import de.intarsys.pdf.platform.cwt.rendering.ICSPlatformDevice;
import de.intarsys.tools.attribute.Attribute;

public class AwtFont implements IPlatformFont {

	private static final Attribute ATTR_PLATFORMGLYPHS = new Attribute(
			"platformGlyphs"); //$NON-NLS-1$

	protected static final Attribute ATTR_SCALE = new Attribute(
			"platformFontScale");

	protected static final Attribute ATTR_TRANSFORM = new Attribute(
			"platformFontTransform");

	final private PDFont font;

	final private Font awtFont;

	private int highByte = -1;

	private double textSpaceFactor = 1;

	private PDGlyphs currentGlyphs = null;

	private boolean symbolic = false;

	protected AwtFont(PDFont font, Font face) {
		super();
		this.font = font;
		this.awtFont = face;
		// the factor for converting from awt to pdf text space
		textSpaceFactor = 1000.0 / 1;
		PDFontDescriptor fontDescriptor = null;
		try {
			fontDescriptor = font.getFontDescriptor();
			symbolic = fontDescriptor.isSymbolic();
		} catch (Exception e) {
			// support invalid documents around
		}
	}

	protected IPlatformGlyphs basicCreate(PDGlyphs glyphs)
			throws PlatformFontException {
		int codepoint = glyphs.getCodepoint();
		if (codepoint < 0) {
			return null;
		}
		return new AwtGlyphs(this, glyphs);
	}

	@Override
	synchronized public IPlatformGlyphs createPlatformGlyphs(PDGlyphs glyphs)
			throws PlatformFontException {
		IPlatformGlyphs result = (IPlatformGlyphs) glyphs
				.getAttribute(ATTR_PLATFORMGLYPHS);
		if (result == null) {
			result = basicCreate(glyphs);
			glyphs.setAttribute(ATTR_PLATFORMGLYPHS, result);
		}
		return result;
	}

	@Override
	public int getAscent() {
		// int result = getAwtFont().getAscender();
		int result = 10;
		return (int) (result * getTextSpaceFactor());
	}

	public Font getAwtFont() {
		return awtFont;
	}

	@Override
	public Rectangle2D getBBox() {
		// Rectangle2D fontBox = getAwtFont().getBBox();
		Rectangle2D fontBox = new Rectangle2D.Double(0, 0, 10, 12);
		int x = (int) (fontBox.getMinX() * getTextSpaceFactor());
		int y = (int) (fontBox.getMinY() * getTextSpaceFactor());
		int width = (int) (fontBox.getWidth() * getTextSpaceFactor());
		int height = (int) (fontBox.getHeight() * getTextSpaceFactor());
		return new Rectangle2D.Float(x, y, width, height);
	}

	@Override
	public int getCharIndex(char unicode) {
		return unicode;
	}

	@Override
	public int getDescent() {
		// int result = getAwtFont().getDescender();
		int result = 2;
		return (int) (result * getTextSpaceFactor());
	}

	@Override
	public int getHeight() {
		// int result = getAwtFont().getHeight();
		int result = 12;
		return (int) (result * getTextSpaceFactor());
	}

	public Object getImpl() {
		return getAwtFont();
	}

	@Override
	public PDFont getPDFont() {
		return font;
	}

	public double getTextSpaceFactor() {
		return textSpaceFactor;
	}

	@Override
	public int getUnderlinePosition() {
		return 0;
	}

	@Override
	public int getUnderlineThickness() {
		return 1;
	}

	@Override
	public boolean isBoldStyle() {
		// todo derive value
		return false;
	}

	@Override
	public boolean isItalicStyle() {
		// todo derive value
		return false;
	}

	@Override
	public boolean isMonospaced() {
		// todo derive value
		return false;
	}

	@Override
	public boolean isSymbolFont() {
		// todo derive value
		return false;
	}

	public boolean isSymbolic() {
		return symbolic;
	}

	@Override
	public void saveState(ICSPlatformDevice device, GraphicsState oldState,
			GraphicsState newState) {
		//
	}

	@Override
	public void textStateChanged(ICSPlatformDevice device) {
		GraphicsState graphicsState = device.getGraphicsState();
		float fontScale = graphicsState.textState.fontSize;
		AffineTransform tempTransform = AffineTransform
				.getScaleInstance(
						fontScale
								* graphicsState.textState.derivedHorizontalScalingFactor,
						-fontScale);
		graphicsState.setAttribute(ATTR_TRANSFORM, tempTransform);
	}

}
