/*
 * Copyright (c) 2008, intarsys consulting GmbH
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Public License as published by the 
 * Free Software Foundation; either version 3 of the License, 
 * or (at your option) any later version.
 * <p/>
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  
 * 
 */
package de.intarsys.pdf.platform.cwt.font.awt;

import java.awt.Font;
import java.io.ByteArrayInputStream;
import java.io.InputStream;
import java.util.logging.Level;
import java.util.logging.Logger;

import de.intarsys.cwt.font.IFont;
import de.intarsys.pdf.content.CSException;
import de.intarsys.pdf.cos.COSName;
import de.intarsys.pdf.cos.COSRuntimeException;
import de.intarsys.pdf.cos.COSStream;
import de.intarsys.pdf.font.PDFont;
import de.intarsys.pdf.font.PDFontDescriptor;
import de.intarsys.pdf.font.PDFontDescriptorEmbedded;
import de.intarsys.pdf.font.PDFontType3;
import de.intarsys.pdf.platform.cwt.font.IPlatformFont;
import de.intarsys.pdf.platform.cwt.font.IPlatformFontFactory;
import de.intarsys.pdf.platform.cwt.font.NullPlatformFontFactory;
import de.intarsys.pdf.platform.cwt.font.PlatformFontException;
import de.intarsys.pdf.platform.cwt.font.PlatformFontTools;
import de.intarsys.pdf.platform.cwt.font.type3.Type3FontFactory;
import de.intarsys.tools.attribute.Attribute;
import de.intarsys.tools.attribute.IAttributeSupport;
import de.intarsys.tools.stream.StreamTools;

public class AwtFontFactory implements IPlatformFontFactory {

	private static final Logger Log = PACKAGE.Log;

	private static final Attribute ATTR_PLATFORMFONT = new Attribute(
			"platformFont"); //$NON-NLS-1$

	static protected void registerPlatformFont(IAttributeSupport as,
			IPlatformFont object) {
		as.setAttribute(ATTR_PLATFORMFONT, object);
	}

	private Type3FontFactory type3FontFactory = new Type3FontFactory();

	private NullPlatformFontFactory nullFontFactory = new NullPlatformFontFactory();

	public AwtFontFactory() {
	}

	protected IPlatformFont basicCreate(PDFont font)
			throws PlatformFontException {
		if (font instanceof PDFontType3) {
			return type3FontFactory.createPlatformFont(font);
		}
		Font awtFont = null;
		PDFontDescriptor pdDesc = null;
		try {
			pdDesc = font.getFontDescriptor();
		} catch (COSRuntimeException e) {
			// there are documents around that are invalid but should be
			// displayed
			// todo 1 @mit warning
		}
		if (pdDesc != null) {
			byte[] fontdata = null;
			fontdata = pdDesc.getFontFile();
			if (fontdata != null) {
				awtFont = loadFace(Font.TYPE1_FONT, fontdata);
			} else {
				fontdata = pdDesc.getFontFile2();
				if (fontdata != null) {
					awtFont = loadFace(Font.TRUETYPE_FONT, fontdata);
				} else {
					COSStream stream = pdDesc.cosGetFontFile3();
					if (stream != null) {
						fontdata = stream.getDecodedBytes();
						COSName subtype = stream.getDict()
								.get(PDFontDescriptorEmbedded.DK_Subtype)
								.asName();
						if (PDFontDescriptorEmbedded.CN_SubType_Type1C
								.equals(subtype)) {
							awtFont = loadFace(Font.TYPE1_FONT, fontdata);
						} else if (PDFontDescriptorEmbedded.CN_SubType_CIDFontType0C
								.equals(subtype)) {
							// awtFont = loadFace(Font.TRUETYPE_FONT, fontdata);
						} else if (PDFontDescriptorEmbedded.CN_SubType_OpenType
								.equals(subtype)) {
							awtFont = loadFace(Font.TRUETYPE_FONT, fontdata);
						} else {
						}
					}
				}
			}
		}
		if (awtFont == null) {
			// no or invalid font data
			awtFont = getExternalFont(font);
		}
		if (awtFont == null) {
			// todo font replacement, error handling
			Log.log(Level.WARNING, "can't load font '" + font.getBaseFont()
					+ "'");
			return nullFontFactory.createPlatformFont(font);
		}
		// selectCharacterMap(font, face);
		AwtFont result = new AwtFont(font, awtFont);
		return result;
	}

	/*
	 * this is synchronized as two threads (e.g. viewing and printing) may race
	 * for the platform font.
	 * 
	 * @see
	 * de.intarsys.pdf.platform.common.IPlatformFontFactory#createPlatformFont
	 * (de.intarsys.pdf.font.PDFont)
	 */
	@Override
	synchronized public IPlatformFont createPlatformFont(PDFont font)
			throws PlatformFontException {
		IPlatformFont result = (IPlatformFont) font
				.getAttribute(ATTR_PLATFORMFONT);
		if (result == null) {
			result = basicCreate(font);
			registerPlatformFont(font, result);
		}
		return result;
	}

	protected Font getExternalFont(PDFont pdFont) throws CSException {
		IFont font = PlatformFontTools.getCWTFont(pdFont);
		if (font != null) {
			return loadFace(font);
		}
		return null;
	}

	protected Font loadFace(IFont font) {
		InputStream is = null;
		try {
			is = font.getFontProgram().getLocator().getInputStream();
			if ("TrueType".equals(font.getFontType())) {
				return Font.createFont(Font.TRUETYPE_FONT, is);
			} else {
				return Font.createFont(Font.TYPE1_FONT, is);
			}
		} catch (Exception e) {
			StreamTools.close(is);
			return null;
		}
	}

	protected Font loadFace(int format, byte[] data) {
		InputStream is = new ByteArrayInputStream(data);
		try {
			return Font.createFont(format, is);
		} catch (Exception e) {
			StreamTools.close(is);
			return null;
		}
	}

}
