/*
 * Copyright (c) 2008, intarsys consulting GmbH
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Public License as published by the 
 * Free Software Foundation; either version 3 of the License, 
 * or (at your option) any later version.
 * <p/>
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  
 * 
 */
package de.intarsys.pdf.platform.cwt.font.awt;

import java.awt.Font;
import java.awt.Shape;
import java.awt.font.FontRenderContext;
import java.awt.font.GlyphVector;
import java.awt.geom.AffineTransform;
import java.awt.geom.GeneralPath;

import de.intarsys.cwt.common.ShapeWrapper;
import de.intarsys.cwt.environment.IGraphicsContext;
import de.intarsys.pdf.content.GraphicsState;
import de.intarsys.pdf.content.TextState;
import de.intarsys.pdf.font.PDGlyphs;
import de.intarsys.pdf.platform.cwt.font.IPlatformFont;
import de.intarsys.pdf.platform.cwt.font.IPlatformGlyphs;
import de.intarsys.pdf.platform.cwt.font.PlatformFontException;
import de.intarsys.pdf.platform.cwt.rendering.ICSPlatformDevice;
import de.intarsys.tools.geometry.ApplyTransformationShape;
import de.intarsys.tools.geometry.TransformedShape;

public class AwtGlyphs implements IPlatformGlyphs {

	final private static Shape EMPTY = new GeneralPath();

	final private AwtFont font;

	final private PDGlyphs glyphs;

	private Shape shape;

	private int width;

	private static final AffineTransform TX = new AffineTransform(new double[] {
			1, 0, 0, 1, 0, 0 });

	protected AwtGlyphs(AwtFont font, PDGlyphs glyphs)
			throws PlatformFontException {
		super();
		this.font = font;
		this.glyphs = glyphs;
		init();
	}

	protected Shape createShape() {
		synchronized (font) {
			Font awtFont = getFont().getAwtFont();
			FontRenderContext frc = new FontRenderContext(TX, true, true);
			// int[] codes = new int[] { getGlyphs().getDecoded() };
			char[] chars = getGlyphs().getChars();
			GlyphVector v = awtFont.createGlyphVector(frc, chars);
			ShapeWrapper iShape = new ShapeWrapper(v.getOutline());
			return iShape;
		}
	}

	public AwtFont getFont() {
		return font;
	}

	@Override
	public PDGlyphs getGlyphs() {
		return glyphs;
	}

	@Override
	public IPlatformFont getPlatformFont() {
		return font;
	}

	public Shape getShape() {
		if (shape == null) {
			shape = createShape();
		}
		return shape;
	}

	@Override
	public int getWidth() {
		return width;
	}

	private void init() throws PlatformFontException {
		synchronized (font) {
			setWidth(10);
		}
	}

	@Override
	public void render(ICSPlatformDevice device) {
		Shape glyphShape = getShape();
		if (glyphShape == EMPTY) {
			// shape may be omitted, for example with whitespace
			return;
		}
		GraphicsState graphicsState = device.getGraphicsState();
		IGraphicsContext gc = device.getGraphicsContext();
		AffineTransform fontTransform = (AffineTransform) graphicsState
				.getAttribute(AwtFont.ATTR_TRANSFORM);
		TransformedShape tempShape = ApplyTransformationShape.create(
				glyphShape, fontTransform);
		int mode = graphicsState.textState.renderingMode & 3;
		if (mode == TextState.RENDERING_MODE_FILL) {
			gc.fill(tempShape);
		} else if (mode == TextState.RENDERING_MODE_STROKE) {
			gc.draw(tempShape);
		} else if (mode == TextState.RENDERING_MODE_FILL_STROKE) {
			gc.fill(tempShape);
			gc.draw(tempShape);
		} else {
			// no rendering
		}
		if ((graphicsState.textState.renderingMode & 4) == 4) {
			device.addTextClip(tempShape);
		}
	}

	public void setWidth(int width) {
		this.width = width;
	}
}
