/*
 * Copyright (c) 2008, intarsys consulting GmbH
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Public License as published by the 
 * Free Software Foundation; either version 3 of the License, 
 * or (at your option) any later version.
 * <p/>
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  
 * 
 */
package de.intarsys.pdf.platform.cwt.image.awt;

import java.awt.Point;
import java.awt.RenderingHints;
import java.awt.Transparency;
import java.awt.image.BufferedImage;
import java.awt.image.ColorModel;
import java.awt.image.ComponentColorModel;
import java.awt.image.DataBuffer;
import java.awt.image.DataBufferByte;
import java.awt.image.IndexColorModel;
import java.awt.image.Raster;
import java.awt.image.RenderedImage;
import java.awt.image.SampleModel;
import java.awt.image.WritableRaster;
import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;

import javax.imageio.ImageIO;
import javax.imageio.ImageReader;
import javax.imageio.stream.ImageInputStream;
import javax.media.jai.ImageLayout;
import javax.media.jai.Interpolation;
import javax.media.jai.JAI;
import javax.media.jai.operator.BandMergeDescriptor;
import javax.media.jai.operator.FormatDescriptor;
import javax.media.jai.operator.InvertDescriptor;
import javax.media.jai.operator.ScaleDescriptor;

import de.intarsys.cwt.awt.image.CwtAwtImageTools;
import de.intarsys.cwt.image.ImageException;
import de.intarsys.cwt.image.ImageTools;
import de.intarsys.pdf.cos.COSStream;
import de.intarsys.pdf.filter.Filter;
import de.intarsys.pdf.pd.PDCSDeviceCMYK;
import de.intarsys.pdf.pd.PDImage;
import de.intarsys.pdf.platform.cwt.color.awt.AwtColorSpace;
import de.intarsys.pdf.platform.cwt.color.awt.AwtColorSpaceFactory;

public class ImageConverterPdf2Awt {

	// arbitrary value - @see ImageConverterPdf2Awt#mergeMask()
	private static final float MAX_SCALE = 400000;

	protected static BufferedImage createBufferedImage(PDImage pdImage) {
		COSStream imgStream = pdImage.cosGetStream();
		if (!pdImage.hasTransparency()
				&& (imgStream.hasFilter(Filter.CN_Filter_DCTDecode)
						|| imgStream.hasFilter(Filter.CN_Filter_DCT) || imgStream
							.hasFilter(Filter.CN_Filter_JPXDecode))) {
			// shortcut for non-masked jpeg images
			try {
				return createBufferedImageFromJPEG(pdImage,
						pdImage.cosExtractJPEGStream());
			} catch (ImageException ex) {
				/*
				 * this is expected if jpeg has no color model. try again later
				 */
			} catch (IOException ex) {
				/*
				 * this is also expected if jpeg has unsupported color model.
				 * try again later
				 */
			}
		}
		try {
			WritableRaster raster;
			ColorModel colorModel = getColorModel(pdImage);
			raster = createRaster(pdImage, colorModel);
			return new BufferedImage(colorModel, raster, false, null);
		} catch (ImageException ex) {
			throw ex;
		} catch (Exception ex) {
			throw new ImageException(ex);
		}
	}

	protected static BufferedImage createBufferedImageFromJPEG(PDImage pdImage,
			COSStream cosStream) throws IOException {
		InputStream stream;
		String colorSpaceType;

		stream = new ByteArrayInputStream(cosStream.getEncodedBytes());
		try {
			colorSpaceType = ImageTools.extractJPEGColorSpaceType(stream);
		} catch (ImageException ex) {
			/*
			 * if color space cannot be extracted assume YCbCr and try decoding;
			 * might still fail later
			 */
			colorSpaceType = "YCbCr"; //$NON-NLS-1$
		}
		if (!"GRAY".equals(colorSpaceType) && !"RGB".equals(colorSpaceType) //$NON-NLS-1$ //$NON-NLS-2$
				&& !"YCbCr".equals(colorSpaceType)) { //$NON-NLS-1$
			// can currently not be decoded directly
			throw new ImageException("AWT non-RGB/GRAY JPEG not supported");
		}
		if (!PlatformImageTools.matchColorSpace(pdImage, colorSpaceType)) {
			throw new ImageException(
					"PD color space does not match JPEG metadata");
		}
		stream.reset();
		return ImageIO.read(stream);
	}

	protected static WritableRaster createDirectRaster(PDImage pdImage,
			ColorModel colorModel, byte[] bytes) {
		int componentSize;
		DataBuffer dataBuffer;

		componentSize = pdImage.getBitsPerComponent();
		dataBuffer = new DataBufferByte(bytes, bytes.length);
		if ((componentSize == 1) || (componentSize == 2)
				|| (componentSize == 4)) {
			return Raster.createPackedRaster(dataBuffer, pdImage.getWidth(),
					pdImage.getHeight(), componentSize, new Point(0, 0));
		}

		if (componentSize == 8) {
			int bands;
			int[] bandOffsets;

			if (colorModel instanceof IndexColorModel) {
				bands = 1;
			} else {
				bands = colorModel.getNumComponents();
			}
			bandOffsets = new int[bands];
			for (int i = 0; i < bandOffsets.length; i++) {
				bandOffsets[i] = i;
			}
			return Raster.createInterleavedRaster(dataBuffer,
					pdImage.getWidth(), pdImage.getHeight(), pdImage.getWidth()
							* bands, bands, bandOffsets, new Point(0, 0));
		}
		throw new ImageException("unsupported component depth " + componentSize);
	}

	/**
	 * the boolean parameter is only for forwarding to jpeg method; see comment
	 * there
	 * 
	 * @param alreadyTried
	 *            when true, indicates that we tried reading non-transparent
	 *            jpeg already.
	 */
	protected static WritableRaster createRaster(PDImage pdImage,
			ColorModel colorModel) throws IOException {

		/*
		 * don't know if reading raster only from JPX works at all - try (if we
		 * got here, all else has failed anyway)
		 */
		if (pdImage.cosGetStream().hasFilter(Filter.CN_Filter_DCTDecode)
				|| pdImage.cosGetStream().hasFilter(Filter.CN_Filter_JPXDecode)) {
			WritableRaster jpegRaster;

			jpegRaster = createRasterFromJPEG(pdImage,
					pdImage.cosExtractJPEGStream());
			return jpegRaster;
		}

		return createDirectRaster(pdImage, colorModel,
				pdImage.getAdjustedBytes(colorModel
						.createCompatibleSampleModel(1, 1).getNumBands()));
	}

	protected static WritableRaster createRasterFromAnyJPEG(PDImage pdImage,
			InputStream stream) throws IOException {
		/*
		 * color conversion for cmyk and ycck images is unsupported in standard
		 * decoder. read raster only. same thing if pdf declares a different
		 * color space than what the JPEG metadata says
		 */

		stream.reset();
		ImageInputStream imageStream = ImageIO.createImageInputStream(stream);
		try {
			ImageReader reader;

			reader = ImageIO.getImageReaders(imageStream).next();
			try {
				reader.setInput(imageStream);
				WritableRaster raster = (WritableRaster) reader.readRaster(0,
						null);
				String colorSpaceType;
				try {
					colorSpaceType = ImageTools
							.extractJPEGColorSpaceType(reader);
				} catch (ImageException ex) {
					/*
					 * expect that color conversion will not be needed if jpeg
					 * stream doesn't contain color metadata; in any case this
					 * result will be better than nothing
					 */
					return raster;
				}
				byte[] dataElements = (byte[]) raster.getDataElements(0, 0,
						pdImage.getWidth(), pdImage.getHeight(), null);
				if (colorSpaceType.equals("YCCK") //$NON-NLS-1$
						&& pdImage.getColorSpace() == PDCSDeviceCMYK.SINGLETON) {
					/*
					 * hope the above restriction catches all cases where we
					 * have to convert; the following yields colors that are at
					 * least similar to what we want
					 */
					int pixelCount = dataElements.length / 4;
					for (int index = 0; index < pixelCount; index++) {
						int pixelBase = index * 4;
						double y = (0xFF & dataElements[pixelBase]);
						double cb = ((0xFF & dataElements[pixelBase + 1]) - 0x80);
						double cr = ((0xFF & dataElements[pixelBase + 2]) - 0x80);
						double cyan = 255 - (y + 1.40200 * cr);
						double magenta = 255 - (y - 0.34414 * cb - 0.71414 * cr);
						double yellow = 255 - (y + 1.77200 * cb);
						if (cyan < 0)
							cyan = 0;
						if (cyan > 255)
							cyan = 255;
						if (magenta < 0)
							magenta = 0;
						if (magenta > 255)
							magenta = 255;
						if (yellow < 0)
							yellow = 0;
						if (yellow > 255)
							yellow = 255;
						dataElements[pixelBase] = (byte) cyan;
						dataElements[pixelBase + 1] = (byte) magenta;
						dataElements[pixelBase + 2] = (byte) yellow;
					}
				}

				AwtColorSpace platformColorSpace = AwtColorSpaceFactory.get()
						.createColorSpace(pdImage.getColorSpace());
				return createDirectRaster(pdImage,
						CwtAwtImageTools.createColorModel(platformColorSpace
								.getColorSpace()), dataElements);
			} finally {
				reader.reset();
				reader.dispose();
			}
		} finally {
			imageStream.close();
		}
	}

	/**
	 * create the raster for a buffered image from the provided jpeg encoded
	 * bytes.
	 */
	protected static WritableRaster createRasterFromJPEG(PDImage pdImage,
			COSStream cosStream) throws IOException {
		InputStream stream;
		String colorSpaceType;

		stream = new ByteArrayInputStream(cosStream.getEncodedBytes());
		try {
			colorSpaceType = ImageTools.extractJPEGColorSpaceType(stream);
		} catch (ImageException ex) {
			/*
			 * if color space cannot be extracted assume YCbCr and try decoding;
			 * might still fail later
			 */
			colorSpaceType = "YCbCr"; //$NON-NLS-1$
		}

		stream.reset();
		/*
		 * YCbCr is the only JPEG color space type where color conversion is
		 * needed *and* the latter can be done by simply reading as buffered
		 * image. In all other cases we have to read the raster explicitly and
		 * perform potential color conversion afterwards.
		 */
		if ("YCbCr".equals(colorSpaceType)) { //$NON-NLS-1$
			try {
				return createRasterFromYCbCrJPEG(stream);
			} catch (ImageException ex) {
				// try the other way
			} catch (IOException ex) {
				// try the other way
			}
		}
		return createRasterFromAnyJPEG(pdImage, stream);
	}

	protected static WritableRaster createRasterFromYCbCrJPEG(InputStream stream)
			throws IOException {
		BufferedImage bufferedImage;

		/* read as buffered image to get color conversion right */
		if ((bufferedImage = ImageIO.read(stream)) == null) {
			// don't know if I need to check this anymore; keep it for now
			throw new ImageException("Couldn't read JPEG");
		}
		return bufferedImage.getRaster();
	}

	protected static ColorModel getColorModel(PDImage pdImage) {
		if (pdImage.isImageMask()) {
			return new IndexColorModel(1, 2, new int[] { 0, -1 }, 0, false, -1,
					DataBuffer.TYPE_BYTE);
		}
		AwtColorSpace colorSpace = AwtColorSpaceFactory.get().createColorSpace(
				pdImage.getColorSpace());
		return colorSpace.getColorModel(pdImage);
	}

	private BufferedImage bufferedImage;

	final private PDImage pdImage;

	public ImageConverterPdf2Awt(PDImage pPdImage) {
		pdImage = pPdImage;
	}

	protected BufferedImage createBufferedImage() {
		BufferedImage image;

		image = createBufferedImage(pdImage);
		if (!pdImage.hasTransparency()) {
			return image;
		}
		return mergeMask(image);
	}

	protected RenderedImage createColorKeyMask(BufferedImage source,
			byte[][] colorKeyMask) {
		/*
		 * couldn't find a JAI operation for chroma key masking; use good old
		 * loop. Not a real problem: this kind of mask is rarely used.
		 */
		int width;
		int height;
		ColorModel maskColorModel;
		SampleModel sourceSampleModel;
		DataBuffer sourceDataBuffer;
		byte[] sourceBytes;
		WritableRaster maskRaster;
		byte[] maskBytes;
		BufferedImage mask;

		width = source.getWidth();
		height = source.getHeight();
		sourceSampleModel = source.getSampleModel();
		sourceDataBuffer = source.getRaster().getDataBuffer();
		sourceBytes = new byte[sourceSampleModel.getNumBands()];
		maskColorModel = CwtAwtImageTools.getGrayColorModel();
		maskRaster = maskColorModel.createCompatibleWritableRaster(width,
				height);
		maskBytes = new byte[] { -1 };
		for (int x = 0; x < width; x++) {
			for (int y = 0; y < height; y++) {
				boolean opaque;

				opaque = false;
				sourceSampleModel.getDataElements(x, y, sourceBytes,
						sourceDataBuffer);
				for (int index = 0; index < sourceBytes.length; index++) {
					if ((sourceBytes[index] & 0xFF) < (colorKeyMask[0][index] & 0xFF)) {
						opaque = true;
						break;
					}
					if ((sourceBytes[index] & 0xFF) > (colorKeyMask[1][index] & 0xFF)) {
						opaque = true;
						break;
					}
				}
				if (opaque) {
					maskRaster.setDataElements(x, y, maskBytes);
				}
			}
		}
		mask = new BufferedImage(maskColorModel, maskRaster, false, null);
		return mask;
	}

	public BufferedImage getBufferedImage() {
		if (bufferedImage == null) {
			bufferedImage = createBufferedImage();
		}
		return bufferedImage;
	}

	public PDImage getPDImage() {
		return pdImage;
	}

	protected BufferedImage mergeMask(BufferedImage image) {
		RenderedImage source;
		RenderedImage mask;
		float sourceScaleX;
		float sourceScaleY;
		float maskScaleX;
		float maskScaleY;
		int width;
		int height;
		ColorModel colorModel;
		ImageLayout imageLayout;

		// TODO optimize
		source = image;
		if (pdImage.getMaskImage() != null) {
			BufferedImage maskImage;

			maskImage = createBufferedImage(pdImage.getMaskImage());
			if (pdImage.getMaskImage().isImageMask()) {
				colorModel = CwtAwtImageTools.getGrayColorModel();
				imageLayout = new ImageLayout(0, 0, maskImage.getWidth(),
						maskImage.getHeight(),
						colorModel.createCompatibleSampleModel(
								maskImage.getWidth(), maskImage.getHeight()),
						colorModel);
				mask = FormatDescriptor.create(maskImage, DataBuffer.TYPE_BYTE,
						new RenderingHints(JAI.KEY_IMAGE_LAYOUT, imageLayout));
				mask = InvertDescriptor.create(mask, null);
			} else {
				mask = maskImage;
			}
		} else if (pdImage.getSMask() != null) {
			mask = createBufferedImage((PDImage) pdImage.getSMask());
		} else {
			byte[][] colorKeyMask;

			colorKeyMask = pdImage.getColorKeyMask(image.getSampleModel()
					.getNumBands());
			mask = createColorKeyMask(image, colorKeyMask);
		}

		sourceScaleX = 1;
		sourceScaleY = 1;
		maskScaleX = 1;
		maskScaleY = 1;
		if (source.getWidth() >= mask.getWidth()) {
			maskScaleX = (float) source.getWidth() / mask.getWidth();
		} else {
			sourceScaleX = (float) mask.getWidth() / source.getWidth();
		}
		if (source.getHeight() >= mask.getHeight()) {
			maskScaleY = (float) source.getHeight() / mask.getHeight();
		} else {
			sourceScaleY = (float) mask.getHeight() / source.getHeight();
		}
		/*
		 * for some reason JAI performance degrades rapidly with increasing
		 * scale factor -> constrain
		 */
		if (sourceScaleX * sourceScaleY > MAX_SCALE) {
			float ratio;

			ratio = MAX_SCALE / (sourceScaleX * sourceScaleY);
			sourceScaleX = sourceScaleX * ratio;
			sourceScaleY = sourceScaleY * ratio;
			maskScaleX = maskScaleX * ratio;
			maskScaleY = maskScaleY * ratio;
		}
		if (maskScaleX * maskScaleY > MAX_SCALE) {
			float ratio;

			ratio = MAX_SCALE / (maskScaleX * maskScaleY);
			sourceScaleX = sourceScaleX * ratio;
			sourceScaleY = sourceScaleY * ratio;
			maskScaleX = maskScaleX * ratio;
			maskScaleY = maskScaleY * ratio;
		}

		if (sourceScaleX != 1 || sourceScaleY != 1) {
			source = ScaleDescriptor.create(source, sourceScaleX, sourceScaleY,
					0f, 0f,
					Interpolation.getInstance(Interpolation.INTERP_NEAREST),
					null);
		}
		if (maskScaleX != 1 || maskScaleY != 1) {
			mask = ScaleDescriptor.create(mask, maskScaleX, maskScaleY, 0f, 0f,
					Interpolation.getInstance(Interpolation.INTERP_NEAREST),
					null);
		}
		width = source.getWidth();
		height = source.getHeight();

		colorModel = mask.getColorModel();
		if (!(colorModel instanceof ComponentColorModel)) {
			colorModel = CwtAwtImageTools.getGrayColorModel();
			imageLayout = new ImageLayout(0, 0, 512, 512,
					colorModel.createCompatibleSampleModel(width, height),
					colorModel);
			mask = FormatDescriptor.create(mask, DataBuffer.TYPE_BYTE,
					new RenderingHints(JAI.KEY_IMAGE_LAYOUT, imageLayout));
		}

		colorModel = source.getColorModel();
		if (!(colorModel instanceof ComponentColorModel)) {
			colorModel = CwtAwtImageTools.getRgbColorModel();
			imageLayout = new ImageLayout(0, 0, width, height,
					colorModel.createCompatibleSampleModel(width, height),
					colorModel);
			source = FormatDescriptor.create(source, DataBuffer.TYPE_BYTE,
					new RenderingHints(JAI.KEY_IMAGE_LAYOUT, imageLayout));
		}

		colorModel = new ComponentColorModel(colorModel.getColorSpace(), true,
				false, Transparency.TRANSLUCENT, colorModel.getTransferType());
		imageLayout = new ImageLayout(0, 0, width, height,
				colorModel.createCompatibleSampleModel(width, height),
				colorModel);
		return BandMergeDescriptor
				.create(source, mask,
						new RenderingHints(JAI.KEY_IMAGE_LAYOUT, imageLayout))
				.getRendering().getAsBufferedImage();
	}
}
