/*
 * Copyright (c) 2008, intarsys consulting GmbH
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Public License as published by the 
 * Free Software Foundation; either version 3 of the License, 
 * or (at your option) any later version.
 * <p/>
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  
 * 
 */
package de.intarsys.pdf.platform.cwt.rendering;

import java.awt.BasicStroke;
import java.awt.Color;
import java.awt.RenderingHints;
import java.awt.Shape;
import java.awt.Stroke;
import java.awt.geom.AffineTransform;
import java.awt.geom.Area;
import java.awt.geom.Line2D;
import java.awt.geom.NoninvertibleTransformException;

import de.intarsys.cwt.common.BlendMode;
import de.intarsys.cwt.common.IPaint;
import de.intarsys.cwt.environment.IGraphicsContext;
import de.intarsys.pdf.cds.CDSMatrix;
import de.intarsys.pdf.content.CSBasicDevice;
import de.intarsys.pdf.content.CSError;
import de.intarsys.pdf.content.CSException;
import de.intarsys.pdf.content.GraphicsState;
import de.intarsys.pdf.cos.COSName;
import de.intarsys.pdf.font.PDFont;
import de.intarsys.pdf.font.PDGlyphs;
import de.intarsys.pdf.pd.PDCSPattern;
import de.intarsys.pdf.pd.PDColorSpace;
import de.intarsys.pdf.pd.PDExtGState;
import de.intarsys.pdf.pd.PDImage;
import de.intarsys.pdf.pd.PDPattern;
import de.intarsys.pdf.pd.PDShading;
import de.intarsys.pdf.platform.cwt.font.IPlatformFont;
import de.intarsys.pdf.platform.cwt.font.IPlatformGlyphs;
import de.intarsys.pdf.platform.cwt.font.PlatformFontException;
import de.intarsys.pdf.platform.cwt.font.PlatformFontFactory;
import de.intarsys.pdf.platform.cwt.image.IPlatformImage;
import de.intarsys.pdf.platform.cwt.image.PlatformImageFactory;
import de.intarsys.pdf.platform.cwt.paint.ColorPaint;
import de.intarsys.pdf.platform.cwt.paint.PatternPaint;
import de.intarsys.pdf.platform.cwt.paint.ShadingPaint;
import de.intarsys.tools.attribute.Attribute;
import de.intarsys.tools.geometry.ShapeTools;

/**
 * A device that bridges between the PDF device operations and a real window
 * toolkit graphics context. The graphics context used is abstracted itself to
 * {@link IGraphicsContext} to allow for easy migration between similar toolkit
 * implementations.
 * 
 */
public class CSPlatformDevice extends CSBasicDevice implements
		ICSPlatformDevice {

	protected static final Attribute ATTR_DEVICECLIP = new Attribute(
			"deviceClip"); //$NON-NLS-1$

	protected static final Attribute ATTR_DEVICETRANSFORM = new Attribute(
			"deviceTransform"); //$NON-NLS-1$

	protected static final Attribute ATTR_NONSTROKEPAINT = new Attribute(
			"nonStrokePaint"); //$NON-NLS-1$

	protected static final Attribute ATTR_STROKEPAINT = new Attribute(
			"strokePaint"); //$NON-NLS-1$

	/**
	 * Flag if we should forward a change in the line attributes
	 */
	private boolean applyLineAttributes = true;

	/**
	 * Flag if we should forward a change for stroke / non stroke paint
	 */
	private boolean applyPaint = true;

	/**
	 * The graphics context of the window toolkit
	 */
	private IGraphicsContext graphicsContext;

	private Shape textClip;

	/**
	 * This is a temporary value storing user space while acting on a text
	 * object. We can safely store this with the device as no
	 * saveState/restoreState is allowed.
	 */
	private AffineTransform userSpace;

	/**
	 * 
	 */
	public CSPlatformDevice(IGraphicsContext graphics) {
		super();
		this.graphicsContext = graphics;
	}

	public void addTextClip(Shape shape) {
		Shape transformed;

		transformed = graphicsContext.getTransform().createTransformedShape(
				shape);
		if (textClip == null) {
			textClip = transformed;
		} else {
			Area area;

			if (textClip instanceof Area) {
				area = (Area) textClip;
			} else {
				area = new Area(textClip);
			}
			area.add(new Area(transformed));
			textClip = area;
		}
	}

	protected void applyLineAttributes() {
		if (applyLineAttributes) {
			Stroke result = null;
			if (graphicsState.dashPattern.length == 0) {
				result = new BasicStroke(graphicsState.lineWidth,
						getGSDeviceCapStyle(graphicsState),
						getGSDeviceJoinStyle(graphicsState),
						graphicsState.miterLimit);
			} else {
				result = new BasicStroke(graphicsState.lineWidth,
						getGSDeviceCapStyle(graphicsState),
						getGSDeviceJoinStyle(graphicsState),
						graphicsState.miterLimit, graphicsState.dashPattern,
						graphicsState.dashPhase);
			}
			graphicsContext.setStroke(result);
			applyLineAttributes = false;
		}
	}

	protected void applyPaint() {
		if (applyPaint) {
			graphicsContext
					.setBackgroundPaint(preparePaint(getGSNonStrokePaint(graphicsState)));
			graphicsContext
					.setForegroundPaint(preparePaint(getGSStrokePaint(graphicsState)));
			applyPaint = false;
		}
	}

	@Override
	protected void basicClip(Shape shape) {
		graphicsContext.clip(shape);
	}

	@Override
	protected void basicDraw(Shape shape) {
		applyLineAttributes();
		applyPaint();
		graphicsContext.draw(shape);
	}

	@Override
	protected void basicFill(Shape shape) {
		IPaint paint;

		applyPaint();
		paint = getGSNonStrokePaint(graphicsState);
		if (paint instanceof PatternPaint) {
			PDPattern pattern;
			CDSMatrix matrix;

			/*
			 * The clipping intersection stuff is omitted with patterns.
			 * Patterns occur rarely enough that we can ignore a possible
			 * combination with small clipping spaces.
			 */
			pattern = ((PatternPaint) paint).getPdPattern();
			matrix = pattern.getMatrix();
			if (matrix == null) {
				graphicsContext.fill(shape);
			} else {
				AffineTransform matrixTransform;

				matrixTransform = matrix.toTransform();
				try {
					AffineTransform inverseTransform;
					AffineTransform originalTransform;

					inverseTransform = matrixTransform.createInverse();
					originalTransform = graphicsContext.getTransform();
					graphicsContext.transform(matrixTransform);
					try {
						Shape inverseShape;
						inverseShape = inverseTransform
								.createTransformedShape(shape);
						graphicsContext.fill(inverseShape);
					} finally {
						graphicsContext.setTransform(originalTransform);
					}
				} catch (NoninvertibleTransformException ex) {
					// probably won't happen anyway
					graphicsContext.fill(shape);
				}
			}
		} else {
			Area clip = ShapeTools.createArea(graphicsContext.getClip(), false);
			if (clip == null || clip.isRectangular()) {
				// if we have well behaved clipping, just fill
				graphicsContext.fill(shape);
			} else {
				// otherwise "clip" manually to force using antialiasing
				graphicsContext.setClip(null);
				try {
					Area intersectArea = ShapeTools.createArea(shape, false);
					intersectArea.intersect(clip);
					graphicsContext.fill(intersectArea);
				} finally {
					graphicsContext.setClip(clip);
				}
			}
		}
	}

	@Override
	protected void basicFillDegenerated(Shape shape) {
		graphicsContext.setStroke(new BasicStroke(0));
		applyLineAttributes = true;
		graphicsContext
				.setForegroundColor(graphicsContext.getBackgroundColor());
		applyPaint = true;
		graphicsContext.draw(shape);
	}

	@Override
	protected void basicSetNonStrokeColorSpace(PDColorSpace pdColorSpace) {
		super.basicSetNonStrokeColorSpace(pdColorSpace);
		applyPaint = true;
		if (pdColorSpace instanceof PDCSPattern) {
			setGSNonStrokePaint(graphicsState, new PatternPaint());
		} else {
			setGSNonStrokePaint(graphicsState, new ColorPaint(pdColorSpace));
		}
	}

	@Override
	protected void basicSetNonStrokeColorValues(float[] values) {
		super.basicSetNonStrokeColorValues(values);
		applyPaint = true;
		((ColorPaint) getGSNonStrokePaint(graphicsState))
				.setColorValues(values);
	}

	@Override
	protected void basicSetStrokeColorSpace(PDColorSpace pdColorSpace) {
		super.basicSetStrokeColorSpace(pdColorSpace);
		applyPaint = true;
		if (pdColorSpace instanceof PDCSPattern) {
			setGSStrokePaint(graphicsState, new PatternPaint());
		} else {
			setGSStrokePaint(graphicsState, new ColorPaint(pdColorSpace));
		}
	}

	@Override
	protected void basicSetStrokeColorValues(float[] values) {
		super.basicSetStrokeColorValues(values);
		applyPaint = true;
		((ColorPaint) getGSStrokePaint(graphicsState)).setColorValues(values);
	}

	@Override
	protected void basicTextShowGlyphs(PDGlyphs glyphs, float advance)
			throws CSException {
		IPlatformGlyphs pg;

		if (textState.fontSize != 0) {
			try {
				IPlatformFont pfFont = PlatformFontFactory.get()
						.createPlatformFont(textState.font);
				pg = pfFont.createPlatformGlyphs(glyphs);
				pg.render(this);
			} catch (PlatformFontException e) {
				// ignore for now
			}
		}
		super.basicTextShowGlyphs(glyphs, advance);
	}

	@Override
	protected void doImage(COSName name, PDImage image) throws CSException {
		//
		double scaleX = image.getWidth();
		double scaleY = image.getHeight();

		/*
		 * check if image will be < 1 pixel after transformation in any
		 * direction. if true, scale up so that image will take at least 1 pixel
		 * This needed in some strange places where lines are emulated by
		 * scaling up a 1 pixel image.
		 */
		AffineTransform tempTransform = graphicsContext.getTransform();
		double[] pts = new double[] { scaleX, scaleY };
		tempTransform.deltaTransform(pts, 0, pts, 0, 1);
		pts[0] = Math.abs(pts[0]);
		pts[1] = Math.abs(pts[1]);
		if (pts[0] < 1) {
			scaleX = pts[0];
		}
		if (pts[1] < 1) {
			scaleY = pts[1];
		}

		AffineTransform transform = new AffineTransform();
		transform.translate(0, 1);
		transform.scale(1.0 / scaleX, -1.0 / scaleY);
		graphicsContext.transform(transform);

		Object interpolation = null;
		if (!image.isInterpolate()) {
			// keep interpolation when scaling the image down, remove otherwise
			if (graphicsContext.getTransform().getDeterminant() >= 1) {
				RenderingHints hints = graphicsContext.getRenderingHints();
				interpolation = hints.remove(RenderingHints.KEY_INTERPOLATION);
				graphicsContext.setRenderingHints(hints);
			}
		}

		try {
			applyPaint();
			IPlatformImage platformImage = PlatformImageFactory.get()
					.createPlatformImage(image);
			platformImage.doImage(this);
		} catch (Exception e) {
			graphicsContext.setForegroundColor(Color.DARK_GRAY);
			graphicsContext.draw(new Line2D.Double(0, 0, image.getWidth(),
					image.getHeight()));
			graphicsContext.draw(new Line2D.Double(image.getWidth(), 0, 0,
					image.getHeight()));
			throw new CSError("Image can't be rendered ("
					+ e.getLocalizedMessage() + ")", e);
		} catch (OutOfMemoryError e) {
			graphicsContext.setForegroundColor(Color.DARK_GRAY);
			graphicsContext.draw(new Line2D.Double(0, 0, image.getWidth(),
					image.getHeight()));
			graphicsContext.draw(new Line2D.Double(image.getWidth(), 0, 0,
					image.getHeight()));
			throw new CSError("Image too large", e);
		} finally {
			graphicsContext.setTransform(tempTransform);
			if (interpolation != null) {
				// restore interpolation settings
				graphicsContext.setRenderingHint(
						RenderingHints.KEY_INTERPOLATION, interpolation);
			}
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see
	 * de.intarsys.pdf.content.CSDeviceAdapter#doShading(de.intarsys.pdf.cos
	 * .COSName, de.intarsys.pdf.pd.PDShading)
	 */
	@Override
	public void doShading(COSName name, PDShading shading) {
		if (shading == null) {
			return;
		}
		ShadingPaint paint = new ShadingPaint(shading);
		graphicsContext.setBackgroundPaint(paint);
		if (shading.getBoundingBox() == null) {
			// use clipping region for want of another shape; is as good as any
			Shape shape = graphicsContext.getClip();
			if (shape != null) {
				graphicsContext.fill(shape);
			}
		} else {
			graphicsContext.fill(shading.getBoundingBox()
					.toNormalizedRectangle());
		}
		// would have to set back to original paint but the other drawing
		// methods always set their color first
		applyPaint = true;
	}

	public IGraphicsContext getGraphicsContext() {
		return graphicsContext;
	}

	final private BlendMode getGSDeviceBlendMode(GraphicsState pState) {
		if (pState.extState == null) {
			return BlendMode.META.getDefault();
		} else if (pState.extState.isBlendModeNormal()) {
			return BlendMode.NORMAL;
		} else if (pState.extState.isBlendModeMultiply()) {
			return BlendMode.MULTIPLY;
		} else {
			return BlendMode.META.getDefault();
		}

	}

	final private int getGSDeviceCapStyle(GraphicsState pState) {
		switch (pState.capStyle) {
		case 0:
			return BasicStroke.CAP_BUTT;
		case 1:
			return BasicStroke.CAP_ROUND;
		case 2:
			return BasicStroke.CAP_SQUARE;
		default:
			return BasicStroke.CAP_BUTT;
		}
	}

	final private Shape getGSDeviceClip(GraphicsState pState) {
		return (Shape) pState.getAttribute(ATTR_DEVICECLIP);
	}

	final private int getGSDeviceJoinStyle(GraphicsState pState) {
		switch (pState.joinStyle) {
		case 0:
			return BasicStroke.JOIN_MITER;
		case 1:
			return BasicStroke.JOIN_ROUND;
		case 2:
			return BasicStroke.JOIN_BEVEL;
		default:
			return BasicStroke.JOIN_MITER;
		}

	}

	final private AffineTransform getGSDeviceTransform(GraphicsState pState) {
		return (AffineTransform) pState.getAttribute(ATTR_DEVICETRANSFORM);
	}

	/**
	 * The paint to use for non stroking operations, translated in AWT jargon.
	 * This value is cached in the GraphicsState directly as the Graphics2D
	 * knows only about one color at a time.
	 * 
	 * <p>
	 * initial value: black color;
	 * </p>
	 */
	final private IPaint getGSNonStrokePaint(GraphicsState pState) {
		return (IPaint) pState.getAttribute(ATTR_NONSTROKEPAINT);
	}

	/**
	 * The paint to use for stroking operations, translated in AWT jargon. This
	 * value is cached in the GraphicsState directly as the Graphics2D knows
	 * only about one color at a time.
	 * 
	 * <p>
	 * initial value: black color
	 * </p>
	 */
	final private IPaint getGSStrokePaint(GraphicsState pState) {
		return (IPaint) pState.getAttribute(ATTR_STROKEPAINT);
	}

	@Override
	protected GraphicsState graphicsStateCopy(GraphicsState oldState) {
		GraphicsState newState = super.graphicsStateCopy(oldState);
		// saving the current state may involve state information delegated to
		// the platform specific implementation, for example cached
		// expensive data structures. So we give a hook here to stateful
		// platform objects
		setGSStrokePaint(newState, getGSStrokePaint(oldState).copy());
		setGSNonStrokePaint(newState, getGSNonStrokePaint(oldState).copy());
		if (textState.font != null) {
			try {
				IPlatformFont pfFont = PlatformFontFactory.get()
						.createPlatformFont(oldState.textState.font);
				pfFont.saveState(this, oldState, newState);
			} catch (PlatformFontException e) {
				// ignore
			}
		}
		return newState;
	}

	@Override
	public void inlineImage(PDImage img) {
		doImage(null, img);
	}

	protected IPaint preparePaint(IPaint paint) {
		BlendMode blendMode = getGSDeviceBlendMode(graphicsState);
		if (blendMode == BlendMode.MULTIPLY) {
			float alphaFactor = .5f;
			if (paint instanceof ColorPaint) {
				ColorPaint cPaint = (ColorPaint) paint;
				float[] colorValues = cPaint.getColorValues();
				if (colorValues != null) {
					float luminance = 0;
					for (int i = 0; i < colorValues.length; i++) {
						luminance += colorValues[i];
					}
					luminance /= colorValues.length;
					alphaFactor = 1 - luminance;
				}
			}
			paint = paint.copy();
			paint.setAlphaValue(paint.getAlphaValue() * alphaFactor);
		}
		return paint;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see de.intarsys.pdf.content.CSDeviceAdapter#restoreState()
	 */
	@Override
	public void restoreState() {
		super.restoreState();
		graphicsContext.setTransform(getGSDeviceTransform(graphicsState));
		graphicsContext.setClip(getGSDeviceClip(graphicsState));
		applyLineAttributes = true;
		applyPaint = true;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see de.intarsys.pdf.content.CSDeviceAdapter#saveState()
	 */
	@Override
	public void saveState() {
		setGSDeviceTransform(graphicsState, graphicsContext.getTransform());
		setGSDeviceClip(graphicsState, graphicsContext.getClip());
		super.saveState();
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see
	 * de.intarsys.pdf.content.CSDeviceAdapter#setExtendedState(de.intarsys.
	 * pdf.cos.COSName, de.intarsys.pdf.pd.PDExtGState)
	 */
	@Override
	public void setExtendedState(COSName name, PDExtGState gstate) {
		float oldStrokeAlpha = graphicsState.strokeAlphaValue;
		float oldNonStrokeAlpha = graphicsState.nonStrokeAlphaValue;
		super.setExtendedState(name, gstate);
		applyLineAttributes = true;
		if (oldStrokeAlpha != graphicsState.strokeAlphaValue
				|| oldNonStrokeAlpha != graphicsState.nonStrokeAlphaValue) {
			getGSNonStrokePaint(graphicsState).setAlphaValue(
					graphicsState.nonStrokeAlphaValue);
			getGSStrokePaint(graphicsState).setAlphaValue(
					graphicsState.strokeAlphaValue);
			applyPaint = true;
		}
	}

	final private void setGSDeviceClip(GraphicsState pState, Shape clip) {
		pState.setAttribute(ATTR_DEVICECLIP, clip);
	}

	final private void setGSDeviceTransform(GraphicsState pState,
			AffineTransform transform) {
		pState.setAttribute(ATTR_DEVICETRANSFORM, transform);
	}

	final private void setGSNonStrokePaint(GraphicsState pState, IPaint paint) {
		paint.setAlphaValue(pState.nonStrokeAlphaValue);
		pState.setAttribute(ATTR_NONSTROKEPAINT, paint);
	}

	final private void setGSStrokePaint(GraphicsState pState, IPaint paint) {
		paint.setAlphaValue(pState.strokeAlphaValue);
		pState.setAttribute(ATTR_STROKEPAINT, paint);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see de.intarsys.pdf.content.CSDeviceAdapter#setLineCap(int)
	 */
	@Override
	public void setLineCap(int capStyle) {
		super.setLineCap(capStyle);
		applyLineAttributes = true;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see de.intarsys.pdf.content.CSDeviceAdapter#setDash(float[], float)
	 */
	@Override
	public void setLineDash(float[] pattern, float phase) {
		super.setLineDash(pattern, phase);
		applyLineAttributes = true;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see de.intarsys.pdf.content.CSDeviceAdapter#setLineJoin(int)
	 */
	@Override
	public void setLineJoin(int joinStyle) {
		super.setLineJoin(joinStyle);
		applyLineAttributes = true;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see de.intarsys.pdf.content.CSDeviceAdapter#setLineWidth(float)
	 */
	@Override
	public void setLineWidth(float lineWidth) {
		super.setLineWidth(lineWidth);
		applyLineAttributes = true;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see de.intarsys.pdf.content.CSDeviceAdapter#setMiterLimit(float)
	 */
	@Override
	public void setMiterLimit(float miterLimit) {
		super.setMiterLimit(miterLimit);
		applyLineAttributes = true;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see
	 * de.intarsys.pdf.content.CSDeviceAdapter#setNonStrokePatternValues(float
	 * [], de.intarsys.pdf.cos.COSName, de.intarsys.pdf.pd.PDPattern)
	 */
	@Override
	public void setNonStrokeColorValues(float[] values, COSName name,
			PDPattern pattern) {
		applyPaint = true;
		((PatternPaint) getGSNonStrokePaint(graphicsState)).setPatternValues(
				values, pattern);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see
	 * de.intarsys.pdf.content.CSDeviceAdapter#setStrokePatternValues(float[],
	 * de.intarsys.pdf.cos.COSName, de.intarsys.pdf.pd.PDPattern)
	 */
	@Override
	public void setStrokeColorValues(float[] values, COSName name,
			PDPattern pattern) {
		applyPaint = true;
		((PatternPaint) getGSStrokePaint(graphicsState)).setPatternValues(
				values, pattern);
	}

	@Override
	public void textBegin() {
		// store user space
		// this is not necessarily equal to the graphicsState transformation
		// as we may have some device specific transformation
		userSpace = graphicsContext.getTransform();
		//
		super.textBegin();
		//
		if (textState.font != null) {
			try {
				IPlatformFont pfFont = PlatformFontFactory.get()
						.createPlatformFont(textState.font);
				pfFont.textStateChanged(this);
			} catch (PlatformFontException e) {
				// ignore
			}
		}
	}

	@Override
	public void textEnd() {
		super.textEnd();
		if (textState.font != null) {
			try {
				IPlatformFont pfFont = PlatformFontFactory.get()
						.createPlatformFont(textState.font);
				pfFont.textStateChanged(this);
			} catch (PlatformFontException e) {
				// ignore
			}
		}
		if (textClip != null) {
			graphicsContext.setTransform(new AffineTransform());
			basicClip(textClip);
			textClip = null;
		}
		// restore user space
		graphicsContext.setTransform(userSpace);
	}

	@Override
	public void textLineMove(float dx, float dy) {
		super.textLineMove(dx, dy);
		// keep graphics context in synch
		graphicsContext.setTransform(userSpace);
		graphicsContext.transform(textState.transform);
	}

	@Override
	public void textLineNew() {
		super.textLineNew();
		// keep graphics context in synch
		graphicsContext.setTransform(userSpace);
		graphicsContext.transform(textState.transform);
	}

	@Override
	public void textMove(float dx, float dy) {
		super.textMove(dx, dy);
		// keep graphics context in synch
		graphicsContext.translate(dx, dy);
	}

	@Override
	public void textSetFont(COSName name, PDFont font, float size) {
		IPlatformFont pfFont = null;
		if (textState.font != null) {
			try {
				pfFont = PlatformFontFactory.get().createPlatformFont(
						textState.font);
			} catch (PlatformFontException e) {
				// ignore
			}
		}
		super.textSetFont(name, font, size);
		if (pfFont != null) {
			pfFont.textStateChanged(this);
		}
		try {
			pfFont = PlatformFontFactory.get().createPlatformFont(font);
			pfFont.textStateChanged(this);
		} catch (PlatformFontException e) {
			// ignore
		}
	}

	@Override
	public void textSetHorizontalScaling(float scaling) {
		IPlatformFont pfFont = null;
		if (textState.font != null) {
			try {
				pfFont = PlatformFontFactory.get().createPlatformFont(
						textState.font);
			} catch (PlatformFontException e) {
				// ignore
			}
		}
		super.textSetHorizontalScaling(scaling);
		if (pfFont != null) {
			pfFont.textStateChanged(this);
		}
	}

	@Override
	public void textSetRenderingMode(int renderingMode) {
		IPlatformFont pfFont = null;
		if (textState.font != null) {
			try {
				pfFont = PlatformFontFactory.get().createPlatformFont(
						textState.font);
			} catch (PlatformFontException e) {
				// ignore
			}
		}
		super.textSetRenderingMode(renderingMode);
		if (pfFont != null) {
			pfFont.textStateChanged(this);
		}
	}

	@Override
	public void textSetRise(float rise) {
		IPlatformFont pfFont = null;
		if (textState.font != null) {
			try {
				pfFont = PlatformFontFactory.get().createPlatformFont(
						textState.font);
			} catch (PlatformFontException e) {
				// ignore
			}
		}
		super.textSetRise(rise);
		if (pfFont != null) {
			pfFont.textStateChanged(this);
		}
	}

	@Override
	public void textSetTransform(float a, float b, float c, float d, float e,
			float f) {
		IPlatformFont pfFont = null;
		if (textState.font != null) {
			try {
				pfFont = PlatformFontFactory.get().createPlatformFont(
						textState.font);
			} catch (PlatformFontException ex) {
				// ignore
			}
		}
		super.textSetTransform(a, b, c, d, e, f);
		// keep graphics context in synch
		graphicsContext.setTransform(userSpace);
		graphicsContext.transform(textState.transform);
		if (pfFont != null) {
			pfFont.textStateChanged(this);
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see de.intarsys.pdf.content.CSDeviceAdapter#textShow(byte[])
	 */
	@Override
	public void textShow(byte[] bytes, int offset, int length) {
		applyLineAttributes();
		applyPaint();
		super.textShow(bytes, offset, length);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see de.intarsys.pdf.content.CSDeviceAdapter#transform(float, float,
	 * float, float, float, float)
	 */
	@Override
	public void transform(float a, float b, float c, float d, float e, float f) {
		super.transform(a, b, c, d, e, f);
		// keep graphics context in synch
		graphicsContext.transform(new AffineTransform(a, b, c, d, e, f));
		userSpace = graphicsContext.getTransform();
	}
}
