/*
 *  Copyright (c) 2011 Leibniz Institute of Plant Genetics and Crop Plant Research (IPK), Gatersleben, Germany.
 *  All rights reserved. This program and the accompanying materials
 *  are made available under the terms of the GNU Lesser Public License v2.1
 *  which accompanies this distribution, and is available at
 *  http://www.gnu.org/licenses/old-licenses/gpl-2.0.html
 *
 *  Contributors:
 *      Leibniz Institute of Plant Genetics and Crop Plant Research (IPK), Gatersleben, Germany - initial API and implementation
 */
package de.ipk_gatersleben.bit.bi.edal.rmi.client;

import java.io.InputStream;
import java.io.OutputStream;
import java.rmi.RemoteException;

import javax.security.auth.Subject;

import com.healthmarketscience.rmiio.RemoteInputStream;
import com.healthmarketscience.rmiio.RemoteInputStreamServer;
import com.healthmarketscience.rmiio.RemoteOutputStream;
import com.healthmarketscience.rmiio.RemoteOutputStreamServer;
import com.healthmarketscience.rmiio.SimpleRemoteInputStream;
import com.healthmarketscience.rmiio.SimpleRemoteOutputStream;

import de.ipk_gatersleben.bit.bi.edal.primary_data.file.PrimaryDataEntityVersion;
import de.ipk_gatersleben.bit.bi.edal.primary_data.file.PrimaryDataEntityVersionException;
import de.ipk_gatersleben.bit.bi.edal.primary_data.file.PrimaryDataFile;
import de.ipk_gatersleben.bit.bi.edal.primary_data.file.PrimaryDataFileException;
import de.ipk_gatersleben.bit.bi.edal.primary_data.metadata.MetaData;
import de.ipk_gatersleben.bit.bi.edal.primary_data.metadata.MetaDataException;
import de.ipk_gatersleben.bit.bi.edal.rmi.interfaces.PrimaryDataFileRmiInterface;

/**
 * SampleClient version of {@link PrimaryDataFile} to use eDAL system like the
 * stand-alone implementation.
 * 
 * @author arendd
 */
public class ClientPrimaryDataFile extends ClientPrimaryDataEntity {

	/**
	 * Constructor for ClientPrimaryDataFile.
	 * 
	 * @param file
	 *            a {@link PrimaryDataFileRmiInterface} object.
	 */
	public ClientPrimaryDataFile(PrimaryDataFileRmiInterface file,
			ClientDataManager client) {
		super(file, client);
	}

	/**
	 * {@link PrimaryDataFile#read(OutputStream)}
	 * 
	 * @param dataOutputStream
	 *            the loaded data.
	 * @throws PrimaryDataFileException
	 *             if no data is stored.
	 * @see PrimaryDataFileRmiInterface#read(RemoteOutputStream, Subject)
	 * @throws RemoteException
	 *             if unable to call remote function.
	 */
	public void read(OutputStream dataOutputStream) throws RemoteException,
			PrimaryDataFileException {

		/** Java 7 version */
		try (RemoteOutputStreamServer ostream = new SimpleRemoteOutputStream(
				dataOutputStream)) {
			((PrimaryDataFileRmiInterface) myEntityRMI).read(ostream.export(),
					this.getCurrentVersion().getRevision(),
					clientDataManager.getSubject());
		}

		/** Java 6 version */
		// RemoteOutputStreamServer ostream = new SimpleRemoteOutputStream(
		// dataOutputStream);
		//
		// ((PrimaryDataFileRmiInterface) myEntityRMI).read(ostream.export(),
		// this
		// .getCurrentVersion().getRevision(), clientDataManager
		// .getSubject());
		//
		// ostream.close();

	}

	/**
	 * {@link PrimaryDataFile#store(InputStream)}
	 * 
	 * @param dataInputStream
	 *            the date to store in this {@link PrimaryDataEntityVersion}.
	 * @throws PrimaryDataFileException
	 *             if storing of data fails.
	 * @throws PrimaryDataEntityVersionException
	 *             if provided version conflicts with existing versions.
	 * @see PrimaryDataFileRmiInterface#store(RemoteInputStream, Subject)
	 * @throws RemoteException
	 *             if unable to call remote function.
	 */
	public void store(InputStream dataInputStream) throws RemoteException,
			PrimaryDataFileException, PrimaryDataEntityVersionException {

		if (dataInputStream == null) {
			throw new PrimaryDataFileException(
					"can not store inputstream: stream is null ");
		}

		/** Java 7 Version */
		try (RemoteInputStreamServer istream = new SimpleRemoteInputStream(
				dataInputStream)) {

			((PrimaryDataFileRmiInterface) myEntityRMI).store(istream.export(),
					clientDataManager.getSubject());
		}

		/** Java 6 Version */
		// RemoteInputStreamServer istream = new SimpleRemoteInputStream(
		// dataInputStream);
		// ((PrimaryDataFileRmiInterface) myEntityRMI).store(istream.export(),
		// clientDataManager.getSubject());
		//
		// istream.close();

		this.currentVersion = this.getVersions().last();

	}

	/** {@inheritDoc} */
	public void setMetaData(MetaData new_metadata) throws RemoteException,
			PrimaryDataEntityVersionException, MetaDataException {
		((PrimaryDataFileRmiInterface) myEntityRMI).setMetaData(new_metadata,
				clientDataManager.getSubject());
		this.currentVersion = this.getVersions().last();
	}
}