/*
 * Copyright (c) 2016 Leibniz Institute of Plant Genetics and Crop Plant Research (IPK), Gatersleben, Germany.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Creative Commons Attribution-NoDerivatives 4.0 International (CC BY-ND 4.0)
 * which accompanies this distribution, and is available at http://creativecommons.org/licenses/by-nd/4.0/
 *
 * Contributors:
 *      Leibniz Institute of Plant Genetics and Crop Plant Research (IPK), Gatersleben, Germany - RMI Client, FileChooser and WebDAV
 */
package de.ipk_gatersleben.bit.bi.edal.publication.attribute.panel;

import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.Dimension;
import java.awt.FlowLayout;
import java.awt.GridBagLayout;
import java.awt.GridLayout;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.util.ArrayList;
import java.util.Objects;

import javax.swing.BorderFactory;
import javax.swing.CellEditor;
import javax.swing.DefaultCellEditor;
import javax.swing.DefaultComboBoxModel;
import javax.swing.JComboBox;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.JTable;
import javax.swing.ListSelectionModel;
import javax.swing.border.EmptyBorder;
import javax.swing.table.DefaultTableModel;
import javax.swing.table.TableColumn;

import de.ipk_gatersleben.bit.bi.edal.primary_data.metadata.LegalPerson;
import de.ipk_gatersleben.bit.bi.edal.primary_data.metadata.NaturalPerson;
import de.ipk_gatersleben.bit.bi.edal.primary_data.metadata.Persons;
import de.ipk_gatersleben.bit.bi.edal.publication.AttributeLabel;
import de.ipk_gatersleben.bit.bi.edal.publication.AttributeTablePanel;
import de.ipk_gatersleben.bit.bi.edal.publication.PropertyLoader;
import de.ipk_gatersleben.bit.bi.edal.publication.PublicationButtonLinePanel;
import de.ipk_gatersleben.bit.bi.edal.publication.PublicationFrame;
import de.ipk_gatersleben.bit.bi.edal.publication.PublicationMainPanel;
import de.ipk_gatersleben.bit.bi.edal.publication.SmallButton;
import de.ipk_gatersleben.bit.bi.edal.publication.Utils;

public class AuthorsPanel extends JPanel implements ActionListener {

	private class AuthorTableModel extends DefaultTableModel {

		private static final long serialVersionUID = 8557354261856948625L;

		public AuthorTableModel() {
			this.setDataVector(this.loadUserValues(), AuthorsPanel.COL_NAMES);
		}

		@Override
		public boolean isCellEditable(final int row, final int column) {
			if (column == 3) {
				if (!this.getValueAt(row, 1).toString().isEmpty() || !this.getValueAt(row, 2).toString().isEmpty()) {
					return false;
				} else {
					return true;
				}
			} else if (column == 1 || column == 2) {
				if (!this.getValueAt(row, 3).toString().isEmpty()) {
					return false;
				} else {
					return true;
				}
			}
			return true;
		}

		private Object[][] loadUserValues() {

			final String string = PropertyLoader.userValues.getProperty("AUTHORS");

			if (string == null || string.isEmpty()) {
				return new Objects[0][0];
			} else {
				final String[] authors = string.split(";");

				final Object[][] data = new Object[authors.length][7];

				for (int i = 0; i < authors.length; i++) {

					if (authors[i] == null || authors[i].isEmpty()) {
						return data;
					} else {
						final String[] author = authors[i].split("@");
						for (int j = 0; j < author.length; j++) {
							data[i][j] = author[j];
						}
					}
				}
				return data;
			}
		}

		@Override
		public void setValueAt(final Object value, final int row, final int column) {
			super.setValueAt(value, row, column);

			AuthorsPanel.this.updateUI();

		}
	}

	private static final long serialVersionUID = 8109825692298261311L;

	private static final AttributeLabel AUTHORS_LABEL = new AttributeLabel(
			PropertyLoader.props.getProperty("AUTHORS_LABEL"), PropertyLoader.props.getProperty("AUTHORS_TOOLTIP"));
	private static final SmallButton OKAY_BUTTON = new SmallButton("OK");
	private static final SmallButton ADD_BUTTON = new SmallButton("ADD AUTHOR");

	private static final SmallButton REMOVE_BUTTON = new SmallButton("REMOVE AUTHOR");
	private static final String CORRESPONDING_AUTOR = "Corresponding author";
	private static final String CONTRIBUTOR = "Contributor";
	private static final String[] roles = { AuthorsPanel.CORRESPONDING_AUTOR, AuthorsPanel.CONTRIBUTOR };

	private static final String[] DEFAULT_AUTHOR_VALUE = new String[] { AuthorsPanel.roles[0], "", "", "",
			"Leibniz Institute of Plant Genetics and Crop Plant Research (IPK), OT Gatersleben, Corrensstraße 3, Stadt Seeland",
			"06466", "Germany" };
	private static final String[] COL_NAMES = { "Role", "Given Name", "Surname", "Group Authors", "Address", "Zip",
			"Country" };

	private DefaultTableModel model;

	private JTable table;

	private final DefaultComboBoxModel<String> comboModel;

	private final JScrollPane scrollPane;

	public AuthorsPanel() {

		this.model = new AuthorTableModel();
		this.table = new JTable(this.model);

		// set the editor as default on every column
		for (int i = 0; i < this.table.getColumnCount(); i++) {
			((DefaultCellEditor) this.table.getDefaultEditor(this.table.getColumnClass(i))).setClickCountToStart(1);
		}

		this.table.setFillsViewportHeight(true);
		this.table.setFont(PropertyLoader.DEFAULT_FONT);
		this.table.setForeground(PropertyLoader.MAIN_FONT_COLOR);
		this.table.getTableHeader().setFont(PropertyLoader.DEFAULT_FONT);
		this.table.getTableHeader().setForeground(PropertyLoader.MAIN_FONT_COLOR);
		this.table.setSelectionMode(ListSelectionModel.SINGLE_SELECTION);
		this.table.getTableHeader().setBorder(BorderFactory.createMatteBorder(1, 1, 1, 1, Color.BLACK));

		this.table.getColumnModel().getColumn(0).setPreferredWidth(PropertyLoader.ATTRIBUTE_PANEL_WIDTH / 100 * 20);
		this.table.getColumnModel().getColumn(1).setPreferredWidth(PropertyLoader.ATTRIBUTE_PANEL_WIDTH / 100 * 10);
		this.table.getColumnModel().getColumn(2).setPreferredWidth(PropertyLoader.ATTRIBUTE_PANEL_WIDTH / 100 * 10);
		this.table.getColumnModel().getColumn(3).setPreferredWidth(PropertyLoader.ATTRIBUTE_PANEL_WIDTH / 100 * 15);
		this.table.getColumnModel().getColumn(4).setPreferredWidth(PropertyLoader.ATTRIBUTE_PANEL_WIDTH / 100 * 50);
		this.table.getColumnModel().getColumn(5).setPreferredWidth(PropertyLoader.ATTRIBUTE_PANEL_WIDTH / 100 * 10);
		this.table.getColumnModel().getColumn(6).setPreferredWidth(PropertyLoader.ATTRIBUTE_PANEL_WIDTH / 100 * 10);

		this.comboModel = new DefaultComboBoxModel<String>(AuthorsPanel.roles);
		final JComboBox<String> comboRow = new JComboBox<String>();
		comboRow.setModel(this.comboModel);
		final TableColumn col = this.table.getColumnModel().getColumn(0);
		col.setCellEditor(new DefaultCellEditor(comboRow));

		this.table.getColumnModel().getColumn(1).setCellRenderer(new AuthorCellRenderer());
		this.table.getColumnModel().getColumn(2).setCellRenderer(new AuthorCellRenderer());
		this.table.getColumnModel().getColumn(3).setCellRenderer(new AuthorCellRenderer());

		final JPanel buttonPanel = new JPanel(new FlowLayout(FlowLayout.CENTER));
		buttonPanel.setBackground(PropertyLoader.MAIN_BACKGROUND_COLOR);

		buttonPanel.add(AuthorsPanel.OKAY_BUTTON);
		buttonPanel.add(AuthorsPanel.ADD_BUTTON);
		buttonPanel.add(AuthorsPanel.REMOVE_BUTTON);

		AuthorsPanel.OKAY_BUTTON.addActionListener(this);
		AuthorsPanel.ADD_BUTTON.addActionListener(this);
		AuthorsPanel.REMOVE_BUTTON.addActionListener(this);

		this.scrollPane = new JScrollPane(this.table);
		this.scrollPane.setPreferredSize(
				new Dimension(PropertyLoader.ATTRIBUTE_PANEL_WIDTH, PropertyLoader.AUTHOR_PANEL_HEIGHT));

		final GridBagLayout grid = new GridBagLayout();
		final JPanel tablePanel = new JPanel(grid);

		tablePanel.setBackground(PropertyLoader.MAIN_BACKGROUND_COLOR);
		tablePanel.setPreferredSize(
				new Dimension(PropertyLoader.ATTRIBUTE_PANEL_WIDTH, PropertyLoader.AUTHOR_PANEL_HEIGHT));

		final EmptyBorder inBorder = new EmptyBorder(5, 5, 0, 5);
		final EmptyBorder outBorder = new EmptyBorder(5, 5, 0, 5);
		tablePanel.setBorder(BorderFactory.createCompoundBorder(outBorder, inBorder));

		Utils.add(tablePanel, grid, this.scrollPane, 0, 0, 1, 1, 1, 0.95, 1, 1);
		Utils.add(tablePanel, grid, buttonPanel, 0, 1, 1, 1, 1, 0.05, 1, 1);

		final JPanel attributePanel = new JPanel(new GridLayout());

		AuthorsPanel.AUTHORS_LABEL.setForeground(PropertyLoader.LABEL_COLOR);

		attributePanel.add(AuthorsPanel.AUTHORS_LABEL);
		attributePanel.setBackground(PropertyLoader.MAIN_BACKGROUND_COLOR);
		attributePanel.setPreferredSize(
				new Dimension(PropertyLoader.ATTRIBUTE_LABEL_WIDTH, PropertyLoader.AUTHOR_PANEL_HEIGHT));

		this.setLayout(new BorderLayout());
		this.add(attributePanel, BorderLayout.WEST);
		this.add(tablePanel, BorderLayout.CENTER);

	}

	@Override
	public void actionPerformed(final ActionEvent actionEvent) {

		if (actionEvent.getSource().equals(AuthorsPanel.OKAY_BUTTON)) {

			final CellEditor cellEditor = this.table.getCellEditor();
			if (cellEditor != null && cellEditor.getCellEditorValue() == null) {
				cellEditor.stopCellEditing();
			}

			/* clear(unselect) selected rows */
			this.table.clearSelection();
			this.comboModel.setSelectedItem(this.comboModel.getSelectedItem());
			/* remove rows without any name */
			final ArrayList<Integer> emptyLines = new ArrayList<Integer>();

			for (int i = 0; i < this.table.getRowCount(); i++) {
				if (Utils.checkIfStringIsEmpty(this.model.getValueAt(i, 1).toString())
						&& Utils.checkIfStringIsEmpty(this.model.getValueAt(i, 2).toString())
						&& Utils.checkIfStringIsEmpty(this.model.getValueAt(i, 3).toString())) {
					emptyLines.add(i);
				}
			}

			for (int i = emptyLines.size() - 1; i >= 0; i--) {
				this.model.removeRow(emptyLines.get(i));
			}

			PublicationMainPanel.authorPanel = this;

			PublicationMainPanel.authorsField.setText(this.getTableContent());

			PublicationMainPanel.releaseAllBlockedFields();

			PublicationMainPanel.titleAuthorSplitPanel.setRightComponent(null);

			PropertyLoader.AUTHORS_LABEL.setForeground(PropertyLoader.LABEL_COLOR);

			PublicationMainPanel.titleAuthorSplitPanel
					.setRightComponent(new AttributeTablePanel(PropertyLoader.AUTHORS_LABEL,
							PublicationMainPanel.authorsField, PropertyLoader.AUTHOR_PANEL_HEIGHT));

			PublicationFrame.updateUI();
			// PublicationModul.getFrame().requestFocusInWindow();
			PublicationButtonLinePanel.getSubmitButton().requestFocus();

			this.saveUserValues();

		} else if (actionEvent.getSource().equals(AuthorsPanel.ADD_BUTTON)) {

			boolean correspondingAuthorSetted = false;

			for (int i = 0; i < this.table.getRowCount(); i++) {
				if (this.table.getValueAt(i, 0).toString().equals(AuthorsPanel.roles[0].toString())) {
					correspondingAuthorSetted = true;
					break;
				}
			}

			if (this.table.getSelectedRow() != -1) {
				if (correspondingAuthorSetted) {
					this.model.insertRow(this.table.getSelectedRow() + 1,
							new String[] { AuthorsPanel.roles[1], "", "", "",
									this.table.getValueAt(this.table.getSelectedRow(), 4).toString(),
									this.table.getValueAt(this.table.getSelectedRow(), 5).toString(),
									this.table.getValueAt(this.table.getSelectedRow(), 6).toString() });
				} else {
					this.model.insertRow(this.table.getSelectedRow() + 1,
							new String[] { AuthorsPanel.roles[0], "", "", "",
									this.table.getValueAt(this.table.getSelectedRow(), 4).toString(),
									this.table.getValueAt(this.table.getSelectedRow(), 5).toString(),
									this.table.getValueAt(this.table.getSelectedRow(), 6).toString() });
				}
			} else {
				if (correspondingAuthorSetted) {
					this.model.addRow(new String[] { AuthorsPanel.roles[1], "", "", "",
							this.table.getValueAt(this.table.getRowCount() - 1, 4).toString(),
							this.table.getValueAt(this.table.getRowCount() - 1, 5).toString(),
							this.table.getValueAt(this.table.getRowCount() - 1, 6).toString() });
				} else {
					if (this.table.getRowCount() != 0) {
						this.model.addRow(new String[] { AuthorsPanel.roles[0], "", "", "",
								this.table.getValueAt(this.table.getRowCount() - 1, 4).toString(),
								this.table.getValueAt(this.table.getRowCount() - 1, 5).toString(),
								this.table.getValueAt(this.table.getRowCount() - 1, 6).toString() });
					} else {
						this.model.addRow(AuthorsPanel.DEFAULT_AUTHOR_VALUE);
					}
				}
				this.scrollPane.getVerticalScrollBar().setValue(this.scrollPane.getVerticalScrollBar().getMaximum());
			}

		}

		else if (actionEvent.getSource().equals(AuthorsPanel.REMOVE_BUTTON)) {

			final CellEditor cellEditor = this.table.getCellEditor();
			if (cellEditor != null && cellEditor.getCellEditorValue() == null) {
				cellEditor.stopCellEditing();
			}

			if (this.model.getRowCount() != 0) {
				if (this.table.getSelectedRow() == -1) {
					this.model.removeRow(this.model.getRowCount() - 1);
					this.scrollPane.getVerticalScrollBar()
							.setValue(this.scrollPane.getVerticalScrollBar().getMaximum());
				} else {
					this.model.removeRow(this.table.getSelectedRow());
				}
			}
		}

	}

	public String getAuthors() {

		final String authorSeperator = new String("; ");
		final String nameSeperator = new String(", ");

		final StringBuffer buffer = new StringBuffer("");

		for (int i = 0; i < this.model.getRowCount(); i++) {
			if (this.model.getValueAt(i, 0).equals(AuthorsPanel.roles[0])) {
				if (!Utils.checkIfStringIsEmpty(this.model.getValueAt(i, 2).toString())
						&& !Utils.checkIfStringIsEmpty(this.model.getValueAt(i, 1).toString())) {
					buffer.append(this.model.getValueAt(i, 2).toString());
					buffer.append(nameSeperator);
					buffer.append(this.model.getValueAt(i, 1).toString());
					buffer.append(authorSeperator);
				} else if (!Utils.checkIfStringIsEmpty(this.model.getValueAt(i, 1).toString())) {
					buffer.append(this.model.getValueAt(i, 1).toString());
					buffer.append(authorSeperator);
				} else if (!Utils.checkIfStringIsEmpty(this.model.getValueAt(i, 2).toString())) {
					buffer.append(this.model.getValueAt(i, 2).toString());
					buffer.append(authorSeperator);
				} else if (!Utils.checkIfStringIsEmpty(this.model.getValueAt(i, 3).toString())) {
					buffer.append(this.model.getValueAt(i, 3).toString());
					buffer.append(authorSeperator);
				}
			}

		}

		if (buffer.toString().lastIndexOf(authorSeperator) != -1) {
			return buffer.toString().substring(0, buffer.toString().lastIndexOf(authorSeperator));
		} else {
			return PropertyLoader.props.getProperty("DEFAULT_AUTHORS_STRING");
		}
	}

	public Persons getContributors() {

		final Persons contributors = new Persons();

		for (int i = 0; i < this.model.getRowCount(); i++) {
			if (this.model.getValueAt(i, 0).toString().equals(AuthorsPanel.CONTRIBUTOR)) {
				if (!Utils.checkIfStringIsEmpty(this.model.getValueAt(i, 2).toString())
						|| !Utils.checkIfStringIsEmpty(this.model.getValueAt(i, 1).toString())) {
					contributors.add(new NaturalPerson(this.model.getValueAt(i, 1).toString(),
							this.model.getValueAt(i, 2).toString(), this.model.getValueAt(i, 4).toString(),
							this.model.getValueAt(i, 5).toString(), this.model.getValueAt(i, 6).toString()));
				} else if (!Utils.checkIfStringIsEmpty(this.model.getValueAt(i, 3).toString())) {
					contributors.add(new LegalPerson(this.model.getValueAt(i, 3).toString(),
							this.model.getValueAt(i, 4).toString(), this.model.getValueAt(i, 5).toString(),
							this.model.getValueAt(i, 6).toString()));
				}
			}
		}

		return contributors;

	}

	public Persons getCreators() {
		final Persons creators = new Persons();

		for (int i = 0; i < this.model.getRowCount(); i++) {
			if (this.model.getValueAt(i, 0).toString().equals(AuthorsPanel.CORRESPONDING_AUTOR)) {
				if (!Utils.checkIfStringIsEmpty(this.model.getValueAt(i, 2).toString())
						|| !Utils.checkIfStringIsEmpty(this.model.getValueAt(i, 1).toString())) {
					creators.add(new NaturalPerson(this.model.getValueAt(i, 1).toString(),
							this.model.getValueAt(i, 2).toString(), this.model.getValueAt(i, 4).toString(),
							this.model.getValueAt(i, 5).toString(), this.model.getValueAt(i, 6).toString()));
				} else if (!Utils.checkIfStringIsEmpty(this.model.getValueAt(i, 3).toString())) {
					creators.add(new LegalPerson(this.model.getValueAt(i, 3).toString(),
							this.model.getValueAt(i, 4).toString(), this.model.getValueAt(i, 5).toString(),
							this.model.getValueAt(i, 6).toString()));
				}

			}
		}

		return creators;

	}

	private String getTableContent() {

		if (this.getCreators().size() == 0) {
			return PropertyLoader.props.getProperty("DEFAULT_AUTHORS_STRING");
		}

		final String authorSeperator = new String("; ");
		final String nameSeperator = new String(", ");

		final StringBuffer buffer = new StringBuffer("");

		for (int i = 0; i < this.model.getRowCount(); i++) {
			if (!Utils.checkIfStringIsEmpty(this.model.getValueAt(i, 2).toString())
					&& !Utils.checkIfStringIsEmpty(this.model.getValueAt(i, 1).toString())) {
				buffer.append(this.model.getValueAt(i, 2).toString());
				buffer.append(nameSeperator);
				buffer.append(this.model.getValueAt(i, 1).toString());
				buffer.append(authorSeperator);
			} else if (!Utils.checkIfStringIsEmpty(this.model.getValueAt(i, 1).toString())) {
				buffer.append(this.model.getValueAt(i, 1).toString());
				buffer.append(authorSeperator);
			} else if (!Utils.checkIfStringIsEmpty(this.model.getValueAt(i, 2).toString())) {
				buffer.append(this.model.getValueAt(i, 2).toString());
				buffer.append(authorSeperator);
			} else if (!Utils.checkIfStringIsEmpty(this.model.getValueAt(i, 3).toString())) {
				buffer.append(this.model.getValueAt(i, 3).toString());
				buffer.append(authorSeperator);
			}
		}

		if (buffer.toString().lastIndexOf(authorSeperator) == -1) {
			return PropertyLoader.props.getProperty("DEFAULT_AUTHORS_STRING");
		} else {
			return buffer.toString().substring(0, buffer.toString().lastIndexOf(authorSeperator));
		}
	}

	public void reset() {
		this.model = new AuthorTableModel();
		this.table = new JTable(this.model);
	}

	private void saveUserValues() {

		final String authorSeperator = new String(";");
		final String nameSeperator = new String("@");

		final StringBuffer buffer = new StringBuffer("");

		for (int i = 0; i < this.model.getRowCount(); i++) {
			if (!Utils.checkIfStringIsEmpty(this.model.getValueAt(i, 2).toString())
					&& !Utils.checkIfStringIsEmpty(this.model.getValueAt(i, 1).toString())
					|| !Utils.checkIfStringIsEmpty(this.model.getValueAt(i, 3).toString())) {
				buffer.append(this.model.getValueAt(i, 0).toString());
				buffer.append(nameSeperator);
				buffer.append(this.model.getValueAt(i, 1).toString());
				buffer.append(nameSeperator);
				buffer.append(this.model.getValueAt(i, 2).toString());
				buffer.append(nameSeperator);
				buffer.append(this.model.getValueAt(i, 3).toString());
				buffer.append(nameSeperator);
				buffer.append(this.model.getValueAt(i, 4).toString());
				buffer.append(nameSeperator);
				buffer.append(this.model.getValueAt(i, 5).toString());
				buffer.append(nameSeperator);
				buffer.append(this.model.getValueAt(i, 6).toString());
				buffer.append(authorSeperator);
			}
		}

		PropertyLoader.setUserValue("AUTHORS", buffer.toString());
	}

}
