/*
 * Copyright (c) 2016 Leibniz Institute of Plant Genetics and Crop Plant Research (IPK), Gatersleben, Germany.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Creative Commons Attribution-NoDerivatives 4.0 International (CC BY-ND 4.0)
 * which accompanies this distribution, and is available at http://creativecommons.org/licenses/by-nd/4.0/
 *
 * Contributors:
 *      Leibniz Institute of Plant Genetics and Crop Plant Research (IPK), Gatersleben, Germany - RMI Client, FileChooser and WebDAV
 */
package de.ipk_gatersleben.bit.bi.edal.webdav.wrapper.primary_data;

import io.milton.http.Auth;
import io.milton.http.Request;
import io.milton.http.Request.Method;
import io.milton.http.exceptions.BadRequestException;
import io.milton.http.exceptions.ConflictException;
import io.milton.http.exceptions.NotAuthorizedException;
import io.milton.resource.CollectionResource;
import io.milton.resource.GetableResource;
import io.milton.resource.PutableResource;
import io.milton.resource.Resource;

import java.io.IOException;
import java.io.InputStream;
import java.rmi.NotBoundException;
import java.rmi.RemoteException;
import java.util.ArrayList;
import java.util.List;
import java.util.SortedSet;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import javax.xml.bind.JAXBException;

import de.ipk_gatersleben.bit.bi.edal.primary_data.file.PrimaryDataDirectoryException;
import de.ipk_gatersleben.bit.bi.edal.primary_data.file.PrimaryDataEntityVersionException;
import de.ipk_gatersleben.bit.bi.edal.primary_data.metadata.MetaDataException;
import de.ipk_gatersleben.bit.bi.edal.primary_data.security.EdalAuthenticateException;
import de.ipk_gatersleben.bit.bi.edal.rmi.client.ClientDataManager;
import de.ipk_gatersleben.bit.bi.edal.rmi.client.ClientPrimaryDataDirectory;
import de.ipk_gatersleben.bit.bi.edal.rmi.client.ClientPrimaryDataEntityVersion;
import de.ipk_gatersleben.bit.bi.edal.webdav.wrapper.FileSystemHandler;

/**
 * A class enabling to browse all versions of a PrimaryDataFile using webDav.
 * 
 * @author benz, arendd
 * 
 */

public class PrimaryDataDirectoryVirtualResource extends
		PrimaryDataEntityResource implements CollectionResource,
		PutableResource {
	private ClientPrimaryDataDirectory dir;

	public PrimaryDataDirectoryVirtualResource(String path,
			FileSystemHandler fileSystemHandler) throws RemoteException,
			PrimaryDataDirectoryException, NotBoundException,
			EdalAuthenticateException {
		super(path, fileSystemHandler);
		if (entity != null) {
			dir = (ClientPrimaryDataDirectory) entity;
		}
	}

	public ClientPrimaryDataDirectory getDir() {
		return this.dir;
	}

	@Override
	public boolean authorise(Request request, Method method, Auth auth) {
		boolean b = super.authorise(request, method, auth);
		if (entity != null) {
			dir = (ClientPrimaryDataDirectory) entity;
		}
		return b;
	}

	@Override
	public Resource child(String childName) {
		if (entity == null) {
			return null;
		}
		Pattern p = Pattern.compile(".*\\/.*\\.v([0-9]+)\\..*$");
		Matcher m = p.matcher(childName);
		if (m.matches()) {
			try {
				long version = Long.valueOf(m.group(1));

				// files inside a directory are either
				Pattern property = Pattern
						.compile(".*\\.v[0-9]+\\.?[^\\.]*\\.properties\\.xml$");
				Matcher pm = property.matcher(childName);

				// virtual files
				if (!pm.matches()) {
					return new PrimaryDataFileVirtualResource(entity.getPath(),
							version, fileSystemHandler);
					// or property files
				} else {
					return new PrimaryDataEntityPropertyVirtualResource(
							entity.getCurrentVersion(), entity
									.getCurrentVersion().getRevision());
				}
			} catch (NumberFormatException e) {
				ClientDataManager.logger.error("Failed to get child: "
						+ childName + "\nReason: " + e.toString());
			} catch (RemoteException e) {
				ClientDataManager.logger.error("Failed to get child: "
						+ childName + "\nReason: " + e.toString());
			} catch (JAXBException e) {
				ClientDataManager.logger.error("Failed to get child: "
						+ childName + "\nReason: " + e.toString());
			} catch (MetaDataException e) {
				ClientDataManager.logger.error("Failed to get child: "
						+ childName + "\nReason: " + e.toString());
			} catch (PrimaryDataDirectoryException e) {
				ClientDataManager.logger.error("Failed to get child: "
						+ childName + "\nReason: " + e.toString());
			} catch (NotBoundException e) {
				ClientDataManager.logger.error("Failed to get child: "
						+ childName + "\nReason: " + e.toString());
			} catch (EdalAuthenticateException e) {
				ClientDataManager.logger.error("Failed to get child: "
						+ childName + "\nReason: " + e.toString());
			} catch (PrimaryDataEntityVersionException e) {
				ClientDataManager.logger.error("Failed to get child: " + dir
						+ "\nReason: " + e.toString());
			}
		}

		return null;
	}

	@Override
	public List<? extends Resource> getChildren() {
		// TODO create base interface
		List<GetableResource> children = new ArrayList<GetableResource>();
		try {
			dir = (ClientPrimaryDataDirectory) (this.fileSystemHandler
					.getEntity(path));
			SortedSet<ClientPrimaryDataEntityVersion> versions = dir
					.getVersions();
			// this directory we only show history versions
			if (versions.size() > 1) {
				// PrimaryDataEntityVersion lastversion = versions.last();
				for (ClientPrimaryDataEntityVersion v : versions) {
					/*
					 * if(v.getRevision()==lastversion.getRevision()) {
					 * continue; }
					 */
					if (v.getRevision() != 0L && !v.isDeleted()) {
						children.add(new PrimaryDataFileVirtualResource(dir
								.getPath(), v.getRevision(), fileSystemHandler));

						children.add(new PrimaryDataEntityPropertyVirtualResource(
								v, v.getRevision()));
					}
				}
			} else {
				/*
				 * if(!file.getCurrentVersion().isDeleted()) { children.add(new
				 * PrimaryDataFileResource(file.getPath(), 0L,
				 * fileSystemHandler));
				 * 
				 * children.add(new PrimaryDataEntityPropertyResource(file
				 * .getCurrentVersion(), file.getCurrentVersion()
				 * .getRevision())); }
				 */
			}

		} catch (RemoteException e) {
			ClientDataManager.logger.error("Failed to get children of: " + dir
					+ "\nReason: " + e.toString());
		} catch (JAXBException e) {
			ClientDataManager.logger
					.error("Failed to create properties.xml for: " + dir
							+ "\nReason: " + e.toString());
		} catch (MetaDataException e) {
			ClientDataManager.logger
					.error("Failed to create properties.xml for: " + dir
							+ "\nReason: " + e.toString());
		} catch (PrimaryDataDirectoryException e) {
			ClientDataManager.logger.error("Failed to get children of: " + dir
					+ "\nReason: " + e.toString());
		} catch (NotBoundException e) {
			ClientDataManager.logger.error("Failed to get children of: " + dir
					+ "\nReason: " + e.toString());
		} catch (EdalAuthenticateException e) {
			ClientDataManager.logger.error("Failed to get children of: " + dir
					+ "\nReason: " + e.toString());
		} catch (PrimaryDataEntityVersionException e) {
			ClientDataManager.logger.error("Failed to get child: " + dir
					+ "\nReason: " + e.toString());
		}

		return children;
	}

	@Override
	public String getName() {
		try {
			return FileSystemHandler.VIRTUALFOLDERPREFIX + entity.getName();
		} catch (RemoteException e) {
			ClientDataManager.logger.error("Failed to get name of: " + dir
					+ "\nReason: " + e.toString());
		}
		return null;
	}

	@Override
	public Resource createNew(String arg0, InputStream arg1, Long arg2,
			String arg3) throws IOException, ConflictException,
			NotAuthorizedException, BadRequestException {
		// we must implement PutableResource interface and return null,
		// otherwise windows webdav client will go to infinite loop
		throw new NotAuthorizedException("don't support this operation!", this);
	}

}
