/*
 * Copyright (c) 2016 Leibniz Institute of Plant Genetics and Crop Plant Research (IPK), Gatersleben, Germany.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Creative Commons Attribution-NoDerivatives 4.0 International (CC BY-ND 4.0)
 * which accompanies this distribution, and is available at http://creativecommons.org/licenses/by-nd/4.0/
 *
 * Contributors:
 *      Leibniz Institute of Plant Genetics and Crop Plant Research (IPK), Gatersleben, Germany - RMI Client, FileChooser and WebDAV
 */
package de.ipk_gatersleben.bit.bi.edal.webdav.wrapper.primary_data;

import java.io.IOException;
import java.io.OutputStream;
import java.rmi.NotBoundException;
import java.rmi.RemoteException;
import java.security.AccessControlException;
import java.util.Map;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import io.milton.http.Auth;
import io.milton.http.Request;
import io.milton.resource.GetableResource;
import io.milton.http.Range;
import io.milton.http.Request.Method;
import io.milton.http.exceptions.BadRequestException;
import io.milton.http.exceptions.NotAuthorizedException;
import io.milton.http.exceptions.NotFoundException;

import de.ipk_gatersleben.bit.bi.edal.primary_data.file.PrimaryDataDirectoryException;
import de.ipk_gatersleben.bit.bi.edal.primary_data.file.PrimaryDataEntityVersionException;
import de.ipk_gatersleben.bit.bi.edal.primary_data.file.PrimaryDataFileException;
import de.ipk_gatersleben.bit.bi.edal.primary_data.metadata.DataFormat;
import de.ipk_gatersleben.bit.bi.edal.primary_data.metadata.DataSize;
import de.ipk_gatersleben.bit.bi.edal.primary_data.metadata.EnumDublinCoreElements;
import de.ipk_gatersleben.bit.bi.edal.primary_data.metadata.MetaDataException;
import de.ipk_gatersleben.bit.bi.edal.primary_data.security.EdalAuthenticateException;
import de.ipk_gatersleben.bit.bi.edal.rmi.client.ClientDataManager;
import de.ipk_gatersleben.bit.bi.edal.rmi.client.ClientPrimaryDataEntityVersion;
import de.ipk_gatersleben.bit.bi.edal.rmi.client.ClientPrimaryDataFile;
import de.ipk_gatersleben.bit.bi.edal.webdav.wrapper.FileSystemHandler;

/**
 * A resource representing a virtual PrimaryDataFile.
 * 
 * @author jinbo
 * 
 */

public class PrimaryDataFileVirtualResource extends PrimaryDataEntityResource
		implements GetableResource {

	// the primary data represented by this resource
	private ClientPrimaryDataFile file;
	// revision of the version represented by this resource
	// this is used for sending the right content of virtual version files
	private long version;
	// the name of this resource
	private String name;

	public PrimaryDataFileVirtualResource(String path, long revision,
			FileSystemHandler fileSystemHandler) throws RemoteException,
			PrimaryDataDirectoryException, NotBoundException,
			EdalAuthenticateException, PrimaryDataEntityVersionException {
		super(path, fileSystemHandler);
		this.version = revision;
		if (entity != null) {
			file = (ClientPrimaryDataFile) entity;
			ClientPrimaryDataEntityVersion v = FileSystemHandler.getVersion(
					file, version);
			try {
				this.name = v.getMetaData()
						.getElementValue(EnumDublinCoreElements.TITLE)
						.getString();
			} catch (MetaDataException e) {
				this.name = file.getName();
				ClientDataManager.logger.error("Failed to get name of: " + file
						+ "\nReason" + e.toString());
			}
		}
	}

	@Override
	public boolean authorise(Request request, Method method, Auth auth) {
		boolean b = super.authorise(request, method, auth);
		if (entity != null) {
			file = (ClientPrimaryDataFile) entity;
			try {
				this.name = this.file.getName();
			} catch (RemoteException e) {
				ClientDataManager.logger
						.fatal("Failed to load/initialize FileSystemHandler: "
								+ e.getMessage());
			}
		}
		return b;
	}

	/**
	 * Returns the file name for a virtual version file.
	 * 
	 * Pattern: filename.v[0-9].file-extension Example: test.v1.xml
	 * 
	 * @return the file name for a virtual version file.
	 */
	@Override
	public String getName() {
		Pattern p = Pattern.compile("(.*)\\.([^\\.]*)$");
		Matcher m = p.matcher(name);

		if (m.matches()) {
			// TODO handle files without extension
			return m.group(1) + ".v" + version + "." + m.group(2);
		}

		return name;
	}

	/**
	 * Sends the content of a virtual version file to a requesting client.
	 */
	@Override
	public void sendContent(final OutputStream out, Range range,
			Map<String, String> params, String contentType) throws IOException,
			NotAuthorizedException, BadRequestException, NotFoundException {
		try {
			ClientPrimaryDataEntityVersion v = FileSystemHandler.getVersion(
					file, version);
			file.switchCurrentVersion(v);
			file.read(out);
		} catch (PrimaryDataFileException e) {
			ClientDataManager.logger.error("Failed to send content of: " + file
					+ "\nReason" + e.toString());
		} catch (AccessControlException e) {
			ClientDataManager.logger.error("Failed to send content of: " + file
					+ "\nReason" + e.toString());
		} catch (PrimaryDataEntityVersionException e) {
			ClientDataManager.logger.error("Failed to send content of: " + file
					+ "\nReason" + e.toString());
		}
	}

	@Override
	public Long getContentLength() {
		DataSize size = null;
		try {
			/*
			 * size = file.getCurrentVersion().getMetaData().getElementValue(
			 * EnumDublinCoreElements.SIZE);
			 */
			ClientPrimaryDataEntityVersion v = FileSystemHandler.getVersion(
					file, version);
			size = v.getMetaData().getElementValue(EnumDublinCoreElements.SIZE);
		} catch (RemoteException e) {
			ClientDataManager.logger.error("Failed to get content-length of: "
					+ file + "\nReason" + e.toString());
		} catch (MetaDataException e) {
			ClientDataManager.logger.error("Failed to get content-length of: "
					+ file + "\nReason" + e.toString());
		} catch (AccessControlException e) {
			ClientDataManager.logger.error("Failed to get content-length of: "
					+ file + "\nReason" + e.toString());
		}

		return size.getFileSize();
	}

	@Override
	public String getContentType(String accepts) {
		String type = "default";

		try {
			DataFormat dataType = (DataFormat) file.getMetaData()
					.getElementValue(EnumDublinCoreElements.FORMAT);
			type = dataType.getMimeType();
			return type;
		} catch (RemoteException e) {
			ClientDataManager.logger.error("Failed to get content-type of: "
					+ file + "\nReason" + e.toString());
		} catch (MetaDataException e) {
			ClientDataManager.logger.error("Failed to get content length of: "
					+ file + "\nReason" + e.toString());
		}

		return type;
	}

	@Override
	public Long getMaxAgeSeconds(Auth auth) {
		return null;
	}

}
