/*
 *  Copyright (c) 2011 Leibniz Institute of Plant Genetics and Crop Plant Research (IPK), Gatersleben, Germany.
 *  All rights reserved. This program and the accompanying materials
 *  are made available under the terms of the GNU Lesser Public License v2.1
 *  which accompanies this distribution, and is available at
 *  http://www.gnu.org/licenses/old-licenses/gpl-2.0.html
 *
 *  Contributors:
 *      Leibniz Institute of Plant Genetics and Crop Plant Research (IPK), Gatersleben, Germany - initial API and implementation
 */
package de.ipk_gatersleben.bit.bi.edal.primary_data.reference;

import java.util.List;
import java.util.Map;

import javax.mail.internet.InternetAddress;

import de.ipk_gatersleben.bit.bi.edal.primary_data.file.EdalException;
import de.ipk_gatersleben.bit.bi.edal.primary_data.file.PublicReference;

/**
 * Provide the implementation for the ApprovalService to get a persistent
 * identifier.
 * 
 * @author arendd
 */
public interface ApprovalServiceProvider {

	/**
	 * Accept a {@link PublicReference} and publish to public.
	 * 
	 * @param ticket
	 *            the ticket number to identify the approved
	 *            {@link PublicReference}.
	 * @param reviewerID
	 *            the ID to identify the reviewer.
	 * @throws EdalApprovalException
	 *             if unable to accept the ticket.
	 */
	void accept(String ticket, int reviewerID) throws EdalApprovalException;

	/**
	 * Initialize a request to approve a {@link PublicReference}.
	 * 
	 * @param reference
	 *            the {@link PublicReference} to approve.
	 * @param emailNotificationAddress
	 *            the eMail address of the requesting user
	 * @throws EdalApprovalException
	 *             if unable to initialize the approval request.
	 */
	void approve(PublicReference reference,
			InternetAddress emailNotificationAddress)
			throws EdalApprovalException;

	/**
	 * Reject a {@link PublicReference}.
	 * 
	 * @param ticket
	 *            the ticket number to identify the rejected
	 *            {@link PublicReference}.
	 * @param reviewerID
	 *            the ID to identify the reviewer.
	 * @throws EdalApprovalException
	 *             if unable to reject the ticket.
	 */
	void reject(String ticket, int reviewerID) throws EdalApprovalException;

	/**
	 * Load all open {@link PublicReference}s that are not yet accepted or
	 * rejected.
	 * 
	 * @return a {@link Map} with all open {@link PublicReference}s and the
	 *         corresponding {@link List} with the {@link ReviewStatus}.
	 */
	Map<PublicReference, List<ReviewStatus>> getAllOpenReviewProcesses();

	/**
	 * Check all {@link ReviewStatus} of the given {@link PublicReference} by
	 * calling {@link ReviewProcess#review(List)}.
	 * 
	 * @param results
	 *            a {@link Map} with all open {@link PublicReference}s and their
	 *            {@link ReviewStatus}.
	 * @throws EdalApprovalException
	 *             if unable to check open review processes.
	 */
	void checkOpenReviewProcesses(
			Map<PublicReference, List<ReviewStatus>> results)
			throws EdalApprovalException;

	String getNewURL(PublicReference reference) throws EdalApprovalException;

	String storeNewDOI(PublicReference reference, String doi, int year)
			throws EdalApprovalException;

	PublicReference getPublicReferenceByInternalId(String internalId)
			throws EdalException;

}