/*
 *  Copyright (c) 2011 Leibniz Institute of Plant Genetics and Crop Plant Research (IPK), Gatersleben, Germany.
 *  All rights reserved. This program and the accompanying materials
 *  are made available under the terms of the GNU Lesser Public License v2.1
 *  which accompanies this distribution, and is available at
 *  http://www.gnu.org/licenses/old-licenses/gpl-2.0.html
 *
 *  Contributors:
 *      Leibniz Institute of Plant Genetics and Crop Plant Research (IPK), Gatersleben, Germany - initial API and implementation
 */
package de.ipk_gatersleben.bit.bi.edal.primary_data.reference;

import java.util.ArrayList;
import java.util.Calendar;
import java.util.List;

/**
 * This class represents the {@link Workgroup}.
 * 
 * @author Denny Hecht
 */
public class Workgroup {
	private final String workgroup;
	private static List<Workgroup> workgroupList = new ArrayList<>();
	private static int timeFormat = Calendar.SECOND;

	/**
	 * The constructor with a specific {@link Workgroup}. Only for internal use.
	 * 
	 * @param workgroup
	 *            the work group
	 */
	protected Workgroup(final String workgroup) {
		this.workgroup = workgroup;
		workgroupList.add(this);
	}

	/**
	 * Getter for the name of the {@link Workgroup}
	 * 
	 * @return the name of the {@link Workgroup}
	 */
	public String getName() {
		return workgroup;
	}

	/**
	 * Returns the work group.
	 * 
	 * @return the work group.
	 */
	@Override
	public String toString() {
		return this.workgroup;
	}

	@Override
	public boolean equals(Object value) {
		return this.workgroup.compareTo((String) value) == 0;
	}

	@Override
	public int hashCode() {
		final int prime = 31;
		int result = 1;
		result = prime * result
				+ ((workgroup == null) ? 0 : workgroup.hashCode());
		return result;
	}

	/**
	 * Returns the time format of the review, e.g. Calendar.Date,
	 * Calendar.SECOND...
	 * 
	 * Returns the time format of the review
	 */
	protected static int getTimeFormat() {
		return timeFormat;
	}

	/**
	 * Set the time format of the review, e.g. Calendar.Date, Calendar.SECOND...
	 * The default value is Calendar.SECOND.
	 * 
	 * @param format
	 *            time format of the review
	 */
	protected static void setTimeFormat(final int format) {
		timeFormat = format;
	}

	/**
	 * 
	 * Returns the {@link Workgroup} with the corresponding name.
	 * 
	 * @param workgroup
	 *            name of the workgroup
	 * @return the {@link Workgroup} with the corresponding name
	 *         <code>null</code> if the {@link Workgroup} doesn't exist
	 * @throws EdalApprovalException
	 *             if there is no {@link Workgroup} with the given name defined.
	 */
	public static Workgroup getWorkgroup(String workgroup)
			throws EdalApprovalException {

		if (workgroup == null) {
			return new Workgroup("default");
		}

		for (Workgroup wg : workgroupList) {
			if (wg.equals(workgroup)) {
				return wg;
			}
		}

		throw new EdalApprovalException(
				"unable to find a defined workgroup with the name '"
						+ workgroup + "'");
	}

	/**
	 * Returns the size of the {@link List}.
	 * 
	 * @return the size of the map
	 */
	protected static int getSize() {
		return workgroupList.size();
	}
}
