/*
 *  Copyright (c) 2011 Leibniz Institute of Plant Genetics and Crop Plant Research (IPK), Gatersleben, Germany.
 *  All rights reserved. This program and the accompanying materials
 *  are made available under the terms of the GNU Lesser Public License v2.1
 *  which accompanies this distribution, and is available at
 *  http://www.gnu.org/licenses/old-licenses/gpl-2.0.html
 *
 *  Contributors:
 *      Leibniz Institute of Plant Genetics and Crop Plant Research (IPK), Gatersleben, Germany - initial API and implementation
 */
package de.ipk_gatersleben.bit.bi.edal.primary_data.reference.datacite;

import org.apache.http.HttpHost;
import org.apache.http.conn.params.ConnRoutePNames;
import org.apache.http.impl.client.DefaultHttpClient;
import org.apache.solr.client.solrj.SolrServerException;
import org.apache.solr.client.solrj.impl.HttpSolrServer;
import org.apache.solr.common.params.CommonParams;
import org.apache.solr.common.params.ModifiableSolrParams;

import de.ipk_gatersleben.bit.bi.edal.primary_data.DataManager;
import de.ipk_gatersleben.bit.bi.edal.primary_data.EdalConfiguration;
import de.ipk_gatersleben.bit.bi.edal.primary_data.EdalConfigurationException;

/**
 * Connector to the Solr-API to search for DataCite objects.
 * 
 * @author arendd
 */
public class DataCiteSearchConnector {

	private static final String IPK_PREFIX = "10.5447";

	private HttpSolrServer dataciteSolrServer = null;

	/**
	 * Constructor to initialize a SolrServer to DateCite.
	 * 
	 * @throws DataCiteException
	 *             if unable to create SolrServer.
	 */
	public DataCiteSearchConnector() throws DataCiteException {

		DefaultHttpClient httpClient = new DefaultHttpClient();

		try {
			HttpHost httpProxy = new HttpHost(DataManager.getConfiguration()
					.getHttpProxyHost(), DataManager.getConfiguration()
					.getHttpProxyPort());

			httpClient.getParams().setParameter(ConnRoutePNames.DEFAULT_PROXY,
					httpProxy);
		} catch (EdalConfigurationException e) {
			DataManager.getImplProv().getLogger()
					.debug("SolrSearchServer use no proxy");
		}

		this.dataciteSolrServer = new HttpSolrServer(
				EdalConfiguration.DATACITE_SEARCH_URL, httpClient);

	}

	/**
	 * Get the number of currently registered DOIs
	 * 
	 * @param year
	 *            the current year.
	 * @return the number of DOIs
	 * @throws SolrServerException
	 *             if unable to query DOIs
	 */
	public int getNumberOfDOIs(int year) throws DataCiteException {

		ModifiableSolrParams solrParams = new ModifiableSolrParams();
		solrParams.set(CommonParams.Q, IPK_PREFIX + " and IPK and " + year);
		solrParams.set(CommonParams.FL, "doi");

		/*
		 * important parameter to remove the "/select" path at the beginning:
		 * http://search.datacite.org/api?q=10.5447 and IPK and 2012&fl=doi
		 */
		solrParams.set(CommonParams.QT, "/");

		int numberOfResult = 0;
		try {
			numberOfResult = this.dataciteSolrServer.query(solrParams)
					.getResults().size();
		} catch (SolrServerException e) {
			e.printStackTrace();
			throw new DataCiteException(
					"unable to query the number of stored DOIs", e);
		}

		return numberOfResult;
	}

	/**
	 * Generate a new DOI by counting current DOI
	 * 
	 * @param year
	 *            the current year.
	 * @return the new DOI
	 * @throws SolrServerException
	 *             if unable to generate new DOI.
	 */
	public String generateNewDOI(int year) throws DataCiteException {

		return IPK_PREFIX + "/IPK/" + year + "/" + getNumberOfDOIs(year);
	}
}
