/*
 *  Copyright (c) 2011 Leibniz Institute of Plant Genetics and Crop Plant Research (IPK), Gatersleben, Germany.
 *  All rights reserved. This program and the accompanying materials
 *  are made available under the terms of the GNU Lesser Public License v2.1
 *  which accompanies this distribution, and is available at
 *  http://www.gnu.org/licenses/old-licenses/gpl-2.0.html
 *
 *  Contributors:
 *      Leibniz Institute of Plant Genetics and Crop Plant Research (IPK), Gatersleben, Germany - initial API and implementation
 */
package de.ipk_gatersleben.bit.bi.edal.primary_data.security;

import java.security.Principal;
import java.util.List;
import java.util.Set;

import de.ipk_gatersleben.bit.bi.edal.primary_data.file.PrimaryDataEntity;
import de.ipk_gatersleben.bit.bi.edal.primary_data.file.PrimaryDataEntityException;

/**
 * Interface that provide all necessary functions for the security system of the
 * eDAL API.
 * 
 * @author lange
 * @author arendd
 */
public interface PermissionProvider {

	/**
	 * Provides all granted {@link EdalPermission} object for a {@link Set} of
	 * {@link Principal} objects.
	 * 
	 * A {@link Set} of {@link Principal} represent a
	 * {@link javax.security.auth.Subject}.
	 * 
	 * @param principalList
	 *            a {@link Set} of {@link Principal} objects.
	 * @return a {@link List} of all granted {@link EdalPermission} objects.
	 */
	List<EdalPermission> findPermissions(final Set<Principal> principalList);

	/**
	 * Grant one {@link EdalPermission} for one {@link Principal}.
	 * <p>
	 * <em>grantPermission(Principal,Method)</em>
	 * 
	 * @param principalType
	 *            the type of the {@link Principal} object.
	 * @param principalName
	 *            the name of the {@link Principal} object.
	 * @param edalPermission
	 *            the {@link EdalPermission} to grant.
	 * @throws PrimaryDataEntityException
	 *             if unable to grant {@link EdalPermission}.
	 */
	void grantPermission(final String principalType,
			final String principalName, final EdalPermission edalPermission)
			throws PrimaryDataEntityException;

	/**
	 * Grant all {@link EdalPermission}s for one {@link Principal}.
	 * <p>
	 * <em>grantPermission(Principal,ALL)</em>
	 * 
	 * @param principalType
	 *            the type of the {@link Principal} object.
	 * @param principalName
	 *            the name of the {@link Principal} object.
	 * @param entity
	 *            the {@link PrimaryDataEntity} to grant.
	 * @throws PrimaryDataEntityException
	 *             if unable to grant {@link EdalPermission}.
	 */
	void grantPermission(final String principalType,
			final String principalName, final PrimaryDataEntity entity)
			throws PrimaryDataEntityException;

	/**
	 * Check if the {@link Principal} is a root user.
	 * 
	 * @param principal
	 *            the {@link Principal} to check.
	 * @return <code>true</code> when the {@link Principal} is a root user;
	 *         <code>false</code> otherwise.
	 */
	boolean isRoot(final Principal principal);

	/**
	 * Revoke one {@link EdalPermission} for one {@link Principal}.
	 * 
	 * @param principalType
	 *            the type of the {@link Principal} object.
	 * @param principalName
	 *            the name of the {@link Principal} object.
	 * @param edalPermission
	 *            the {@link EdalPermission} to revoke.
	 * @throws PrimaryDataEntityException
	 *             if unable to revoke {@link EdalPermission}.
	 */
	void revokePermission(final String principalType,
			final String principalName, final EdalPermission edalPermission)
			throws PrimaryDataEntityException;

	/**
	 * Revoke all {@link EdalPermission}s for one {@link Principal}.
	 * <p>
	 * <em>revokePermission(Principal,ALL)</em>
	 * 
	 * @param principalType
	 *            the type of the {@link Principal} object.
	 * @param principalName
	 *            the name of the {@link Principal} object.
	 * @param entity
	 *            the {@link PrimaryDataEntity} to revoke.
	 * @throws PrimaryDataEntityException
	 *             if unable to revoke {@link EdalPermission}.
	 */
	void revokePermission(final String principalType,
			final String principalName, final PrimaryDataEntity entity)
			throws PrimaryDataEntityException;

	/**
	 * Set the id of the current {@link PrimaryDataEntity} object.
	 * 
	 * @param id
	 *            the id to set.
	 */
	void setPermissionObjectID(String id);

}
