/*
 * Copyright (c) 2016 Leibniz Institute of Plant Genetics and Crop Plant Research (IPK), Gatersleben, Germany.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Creative Commons Attribution-NoDerivatives 4.0 International (CC BY-ND 4.0)
 * which accompanies this distribution, and is available at http://creativecommons.org/licenses/by-nd/4.0/
 *
 * Contributors:
 *      Leibniz Institute of Plant Genetics and Crop Plant Research (IPK), Gatersleben, Germany - initial API and implementation
 */
package de.ipk_gatersleben.bit.bi.edal.primary_data;

import java.io.BufferedOutputStream;
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.io.PipedInputStream;
import java.io.PipedOutputStream;
import java.io.StringWriter;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.HashMap;
import java.util.List;
import java.util.Map.Entry;
import java.util.StringTokenizer;
import java.util.UUID;
import java.util.concurrent.ArrayBlockingQueue;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.zip.ZipOutputStream;

import javax.mail.internet.AddressException;
import javax.mail.internet.InternetAddress;
import javax.servlet.ServletException;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import org.apache.commons.io.output.TeeOutputStream;
import org.eclipse.jetty.http.HttpStatus;
import org.eclipse.jetty.http.HttpStatus.Code;
import org.eclipse.jetty.io.EofException;
import org.eclipse.jetty.server.Request;
import org.eclipse.jetty.server.handler.AbstractHandler;

import de.ipk_gatersleben.bit.bi.edal.primary_data.file.EdalException;
import de.ipk_gatersleben.bit.bi.edal.primary_data.file.PrimaryDataDirectory;
import de.ipk_gatersleben.bit.bi.edal.primary_data.file.PrimaryDataEntity;
import de.ipk_gatersleben.bit.bi.edal.primary_data.file.PrimaryDataEntityVersionException;
import de.ipk_gatersleben.bit.bi.edal.primary_data.file.PrimaryDataFile;
import de.ipk_gatersleben.bit.bi.edal.primary_data.file.PrimaryDataFileException;
import de.ipk_gatersleben.bit.bi.edal.primary_data.file.PublicReference;
import de.ipk_gatersleben.bit.bi.edal.primary_data.metadata.DataFormat;
import de.ipk_gatersleben.bit.bi.edal.primary_data.metadata.DataSize;
import de.ipk_gatersleben.bit.bi.edal.primary_data.metadata.EnumDublinCoreElements;
import de.ipk_gatersleben.bit.bi.edal.primary_data.metadata.MetaData;
import de.ipk_gatersleben.bit.bi.edal.primary_data.metadata.MetaDataException;
import de.ipk_gatersleben.bit.bi.edal.primary_data.reference.EdalApprovalException;
import de.ipk_gatersleben.bit.bi.edal.primary_data.reference.PersistentIdentifier;

public class EdalJettyHandler extends AbstractHandler {

	private static HashMap<Integer, List<String>> hashmap = new HashMap<Integer, List<String>>();

	private static ExecutorService executor;
	private static EdalThreadPoolExcecutor zipExecutor;

	private static final int CorePoolSizeForZipExecutor = Runtime.getRuntime().availableProcessors();
	private static final int MaximumPoolSizeForZipExecutor = 2 * Runtime.getRuntime().availableProcessors();

	private static final Long LIMIT_INLINE_FILE_SIZE = new Long(10 * 1024 * 1024);

	public static WebPageCache cache = new WebPageCache();

	static VeloCityHtmlGenerator velocityHtmlGenerator;

	static {
		executor = DataManager.getJettyThreadPool();
		zipExecutor = new EdalThreadPoolExcecutor(CorePoolSizeForZipExecutor, MaximumPoolSizeForZipExecutor, 30,
				TimeUnit.SECONDS, new ArrayBlockingQueue<Runnable>(30));
		cache.init();
		velocityHtmlGenerator = new VeloCityHtmlGenerator();
	}

	@Override
	public void handle(String target, Request baseRequest, HttpServletRequest request, HttpServletResponse response)
			throws IOException, ServletException {

		if (request.getMethod().equalsIgnoreCase("GET")) {

			final String url = request.getRequestURI().toString();

			DataManager.getImplProv().getLogger().debug(url);

			final StringTokenizer tokenizer = new StringTokenizer(url, EdalJettyServer.EDAL_PATH_SEPARATOR);

			if (!tokenizer.hasMoreTokens()) {
				/* no method defined -> break */
				this.sendMessage(response, HttpStatus.Code.NOT_FOUND, "no ID and version specified");
			} else {
				final String methodToken = tokenizer.nextToken().toUpperCase();

				try {
					switch (EdalHttpFunctions.valueOf(methodToken)) {

					case DOI:
					case URL:
						if (tokenizer.hasMoreTokens()) {
							final String internalId = tokenizer.nextToken();

							if (tokenizer.hasMoreTokens()) {

								final String uuidToken = tokenizer.nextToken();

								if (tokenizer.hasMoreTokens()) {
									final String versionToken = tokenizer.nextToken();
									try {
										final long versionNumber = Long.parseLong(versionToken);

										PrimaryDataEntity entity = null;

										/** code for reviewer landing page */
										if (tokenizer.hasMoreTokens()) {
											final String reviewerToken = tokenizer.nextToken();

											if (tokenizer.hasMoreElements()) {
												final String methodTokenForReviewer = tokenizer.nextToken();

												/** process for reviewer : */

												if (methodTokenForReviewer
														.equalsIgnoreCase(EdalHttpFunctions.DOWNLOAD.name())) {

													this.responseDownloadRequest(response, entity, uuidToken,
															versionNumber, internalId, methodToken, reviewerToken);
													break;

												} else if (methodTokenForReviewer
														.equalsIgnoreCase(EdalHttpFunctions.ZIP.name())) {

													this.responseZipRequest(response, entity, uuidToken, versionNumber,
															internalId, methodToken, reviewerToken);
													break;

												} else {

													this.sendMessage(response, HttpStatus.Code.FORBIDDEN,
															"unable to process '" + methodTokenForReviewer + "'");
													break;
												}

											} else {
												/**
												 * check if request is for
												 * DOWNLOAD function
												 */
												if (reviewerToken.equalsIgnoreCase(EdalHttpFunctions.DOWNLOAD.name())) {

													this.responseDownloadRequest(response, entity, uuidToken,
															versionNumber, internalId, methodToken, null);

												}
												/**
												 * check if request is for ZIP
												 * function function
												 */
												else if (reviewerToken.equalsIgnoreCase(EdalHttpFunctions.ZIP.name())) {

													this.responseZipRequest(response, entity, uuidToken, versionNumber,
															internalId, methodToken, null);
													break;

												}

												else {
													/**
													 * check if request is a
													 * reviewer code
													 */
													int reviewer = 0;
													try {
														reviewer = Integer.parseInt(reviewerToken);
													} catch (final NumberFormatException e) {
														this.sendMessage(response, HttpStatus.Code.FORBIDDEN,
																"unable to process reviewer ID '" + reviewerToken
																		+ "', please check again");
														break;
													}

													/**
													 * check if reviewer code
													 * exists
													 */
													entity = DataManager.getPrimaryDataEntityForReviewer(uuidToken,
															versionNumber, internalId, reviewer);
													this.sendEntityMetaDataForReviewer(response, entity, versionNumber,
															internalId, PersistentIdentifier.valueOf(methodToken),
															reviewer);
													break;
												}
											}
										}
										/**
										 * end code for reviewer landing page
										 */
										else {

											entity = DataManager.getPrimaryDataEntityForPersistenIdentifier(uuidToken,
													versionNumber, PersistentIdentifier.valueOf(methodToken));
											this.sendEntityMetaDataForPersistentIdentifier(response, entity,
													versionNumber, PersistentIdentifier.valueOf(methodToken),
													internalId);
											break;
										}

									} catch (final EdalException e) {

										this.sendMessage(response, HttpStatus.Code.LOCKED, e.getMessage());
									}
								} else {
									this.sendMessage(response, HttpStatus.Code.FORBIDDEN, "no version number set");
								}
							} else {
								this.sendMessage(response, HttpStatus.Code.FORBIDDEN, "no entity id set");
							}
						} else {
							this.sendMessage(response, HttpStatus.Code.FORBIDDEN, "no internal id set");
						}
						break;

					/* request an object */
					case EDAL:

						final String uuidToken = tokenizer.nextToken();

						if (tokenizer.hasMoreTokens()) {
							final String versionToken = tokenizer.nextToken();
							try {
								final int versionNumber = Integer.parseInt(versionToken);
								PrimaryDataEntity entity = null;

								if (tokenizer.hasMoreElements()) {

									final String downloadToken = tokenizer.nextToken();
									if (EdalHttpFunctions.valueOf(downloadToken.toUpperCase())
											.equals(EdalHttpFunctions.DOWNLOAD)) {

										entity = DataManager.getPrimaryDataEntityByID(uuidToken, versionNumber);

										if (!entity.isDirectory()) {
											this.sendFile((PrimaryDataFile) entity, versionNumber, response);
										} else {
											this.sendEntityMetaData(response, entity);
										}
									}
								} else {
									entity = DataManager.getPrimaryDataEntityByID(uuidToken, versionNumber);

									this.sendEntityMetaData(response, entity);
								}

							} catch (final NumberFormatException | EdalException e) {
								if (e.getClass().equals(NumberFormatException.class)) {
									this.sendMessage(response, HttpStatus.Code.NOT_FOUND,
											"unable to cast '" + versionToken + "' to a version number");
								} else {
									this.sendMessage(response, HttpStatus.Code.NOT_FOUND,
											"unable to send data : " + e.getMessage());
								}
							}
						} else {
							this.sendMessage(response, HttpStatus.Code.NOT_FOUND, "no version number set");
						}
						break;

					/* accept an object */
					case ACCEPT:

						final String ticketAccept = tokenizer.nextToken();

						if (tokenizer.hasMoreElements()) {

							try {
								final int reviewerHashCode = Integer.parseInt(tokenizer.nextToken());

								if (!ticketAlreadyClicked(reviewerHashCode, ticketAccept)) {

									DataManager.getImplProv().getApprovalServiceProvider().newInstance()
											.accept(ticketAccept, reviewerHashCode);
									this.sendMessage(response, HttpStatus.Code.OK, "Thank you");

								} else {
									this.sendMessage(response, HttpStatus.Code.FORBIDDEN, "Already clicked");
								}
							} catch (EdalApprovalException | InstantiationException | NumberFormatException
									| IllegalAccessException e) {

								EdalJettyHandler.deleteTicketFromHashMap(ticketAccept);
								this.sendMessage(response, HttpStatus.Code.NOT_FOUND,
										"Can not to accept ticket it seems to be already accepted: " + e.getMessage());
							}
						} else {
							this.sendMessage(response, HttpStatus.Code.NOT_FOUND, "No ReviewerCode definded");
						}

						break;

					/* reject an object */
					case REJECT:

						final String ticketReject = tokenizer.nextToken();

						if (tokenizer.hasMoreElements()) {

							try {
								final int reviewerHashCode = Integer.parseInt(tokenizer.nextToken());

								if (!ticketAlreadyClicked(reviewerHashCode, ticketReject)) {
									DataManager.getImplProv().getApprovalServiceProvider().newInstance()
											.reject(ticketReject, reviewerHashCode);
									this.sendMessage(response, HttpStatus.Code.OK, "Thank you");
								} else {
									this.sendMessage(response, HttpStatus.Code.FORBIDDEN, "Already clicked");
								}

							} catch (EdalApprovalException | InstantiationException | NumberFormatException
									| IllegalAccessException e) {

								EdalJettyHandler.deleteTicketFromHashMap(ticketReject);
								this.sendMessage(response, HttpStatus.Code.NOT_FOUND,
										"Can not to reject ticket  it seems to be already rejected: " + e.getMessage());
							}
						} else {
							this.sendMessage(response, HttpStatus.Code.NOT_FOUND, "No ReviewerCode definded");
						}

						break;

					case LOGIN:

						final String uuid = tokenizer.nextToken();
						final String emailAddress = tokenizer.nextToken();

						Boolean successful = false;
						try {
							successful = DataManager.getImplProv().getPermissionProvider().newInstance()
									.validateRootUser(new InternetAddress(emailAddress), UUID.fromString(uuid));
						} catch (final AddressException | InstantiationException | IllegalAccessException e) {
							this.sendMessage(response, HttpStatus.Code.NOT_FOUND,
									"eDAL-Server Admin confirmation failed");
						}

						if (successful) {
							try {
								this.sendMessage(response, HttpStatus.Code.OK,
										"Thank you <br/>You have successfully registered as administrator for eDAL-Server on "
												+ EdalJettyServer.getServerURL()
												+ "<br/>The server is now started in working mode.");
							} catch (final EdalException e) {
								this.sendMessage(response, HttpStatus.Code.NOT_FOUND, "unable to load server URL");
							}
						} else {
							this.sendMessage(response, HttpStatus.Code.NOT_FOUND, "root user validation failed");
						}

						break;

					case LOGO:
						if (tokenizer.hasMoreTokens()) {
							final String logoUrl = tokenizer.nextToken();

							if (logoUrl.equalsIgnoreCase("edal_scaled.png")) {
								this.sendEmbeddedFile(response, "edal_scaled.png", "image/png");
							}
						}
						break;

					case CSS:
						if (tokenizer.hasMoreTokens()) {

							final String fileUrl = tokenizer.nextToken();

							this.sendEmbeddedFile(response, "css" + "/" + fileUrl, "text/css");
						}
						break;
					case JS:
						if (tokenizer.hasMoreTokens()) {

							final String fileUrl = tokenizer.nextToken();

							this.sendEmbeddedFile(response, "js" + "/" + fileUrl, "application/javascript");
						}
						break;
					case FONTS:
						if (tokenizer.hasMoreTokens()) {
							final String fileUrl = tokenizer.nextToken();

							if (fileUrl.contains("svg")) {
								this.sendEmbeddedFile(response, "fonts" + "/" + fileUrl, "image/svg+xml");
							}
							if (fileUrl.contains("eot")) {
								this.sendEmbeddedFile(response, "fonts" + "/" + fileUrl,
										"application/vnd.ms-fontobject");
							}
							if (fileUrl.contains("woff")) {
								this.sendEmbeddedFile(response, "fonts" + "/" + fileUrl, "application/x-font-woff");
							}
							if (fileUrl.contains("woff2")) {
								this.sendEmbeddedFile(response, "fonts" + "/" + fileUrl, "application/x-font-woff");
							}
							if (fileUrl.contains("ttf")) {
								this.sendEmbeddedFile(response, "fonts" + "/" + fileUrl, "application/x-font-ttf");
							}
						}
						break;

					case FILEICONS:
						if (tokenizer.hasMoreTokens()) {

							final String fileUrl = tokenizer.nextToken();

							this.sendEmbeddedFile(response, "fileicons" + "/" + fileUrl, "image/png");
						}
						break;

					case USER_ACCEPT:
						final String userAcceptTicket = tokenizer.nextToken();

						if (tokenizer.hasMoreElements()) {

							try {
								final int reviewerHashCode = Integer.parseInt(tokenizer.nextToken());

								if (!ticketAlreadyClicked(reviewerHashCode, userAcceptTicket)) {

									DataManager.getImplProv().getApprovalServiceProvider().newInstance()
											.acceptTicketByUser(userAcceptTicket, reviewerHashCode);

									this.sendMessage(response, HttpStatus.Code.OK, "Thank you");

								} else {
									this.sendMessage(response, HttpStatus.Code.FORBIDDEN, "Already clicked");
								}
							} catch (NumberFormatException | InstantiationException | IllegalAccessException
									| EdalApprovalException e) {

								EdalJettyHandler.deleteTicketFromHashMap(userAcceptTicket);

								this.sendMessage(response, HttpStatus.Code.NOT_FOUND,
										"Can not to accept ticket it seems to be already accepted: " + e.getMessage());
							}
						} else {
							this.sendMessage(response, HttpStatus.Code.NOT_FOUND, "No ReviewerCode definded");
						}

						break;

					case USER_REJECT:

						final String userRejectTicket = tokenizer.nextToken();

						if (tokenizer.hasMoreElements()) {

							try {
								final int reviewerHashCode = Integer.parseInt(tokenizer.nextToken());

								if (!ticketAlreadyClicked(reviewerHashCode, userRejectTicket)) {

									DataManager.getImplProv().getApprovalServiceProvider().newInstance()
											.rejectTicketByUser(userRejectTicket, reviewerHashCode);

									this.sendMessage(response, HttpStatus.Code.OK, "Thank you");

								} else {
									this.sendMessage(response, HttpStatus.Code.FORBIDDEN, "Already clicked");
								}
							} catch (NumberFormatException | InstantiationException | IllegalAccessException
									| EdalApprovalException e) {

								EdalJettyHandler.deleteTicketFromHashMap(userRejectTicket);

								this.sendMessage(response, HttpStatus.Code.NOT_FOUND,
										"Can not to accept ticket it seems to be already rejected: " + e.getMessage());
							}
						} else {
							this.sendMessage(response, HttpStatus.Code.NOT_FOUND, "No ReviewerCode definded");
						}

						break;

					case REPORT:

						if (tokenizer.countTokens() == 0) {
							this.sendReport(response, HttpStatus.Code.OK, null);
						} else if (tokenizer.countTokens() == 1) {
							this.sendReport(response, HttpStatus.Code.OK, tokenizer.nextToken());
						} else if (tokenizer.countTokens() == 2) {
							this.sendCSVReport(response, HttpStatus.Code.OK, tokenizer.nextToken(),
									tokenizer.nextToken());
						}
						break;

					case OAI:

						// String oaiRequest = request.getQueryString();
						//
						// /** cut set **/
						// if (oaiRequest.contains("set=")) {
						//
						// oaiRequest = oaiRequest.substring(0,
						// oaiRequest.indexOf("set=")) + oaiRequest.substring(
						// oaiRequest.indexOf("&", oaiRequest.indexOf("set=")),
						// oaiRequest.length());
						//
						// System.out.println(oaiRequest);
						//
						// }
						//
						// if (oaiRequest.contains("verb=Identify") ||
						// oaiRequest.contains("verb=ListMetadataFormats")
						// || oaiRequest.contains("verb=ListSets")) {
						// response.sendRedirect("http://oai.datacite.org/oai?"
						// + oaiRequest);
						// } else {
						// response.sendRedirect("http://oai.datacite.org/oai?set=TIB.IPK&"
						// + oaiRequest);
						// }
						//
						// break;

						if (!DataManager.getConfiguration().isInTestMode()) {

							String dataCenter = null;
							try {
								dataCenter = DataManager.getConfiguration().getDataCiteUser();
							} catch (EdalConfigurationException e) {
								e.printStackTrace();
							}

							String oaiRequest = request.getQueryString();

							if (oaiRequest != null) {
								if (oaiRequest.contains("set=")) {
									if (oaiRequest.indexOf("set=") > oaiRequest.lastIndexOf("&")) {

										oaiRequest = oaiRequest.substring(0, oaiRequest.indexOf("set=")) + "set="
												+ dataCenter;

										response.sendRedirect("http://oai.datacite.org/oai?" + oaiRequest);

									} else {

										String prefix = oaiRequest.substring(0, oaiRequest.indexOf("set="));
										String suffix = oaiRequest.substring(
												oaiRequest.indexOf("&", oaiRequest.indexOf("set=")),
												oaiRequest.length());

										oaiRequest = prefix + suffix + "&set=" + dataCenter;
										response.sendRedirect("http://oai.datacite.org/oai?" + oaiRequest);
									}

								} else if (oaiRequest.contains("verb=Identify")
										|| oaiRequest.contains("verb=ListMetadataFormats")
										|| oaiRequest.contains("verb=ListRecords&metadataPrefix=oai_dc")
										|| oaiRequest.contains("verb=ListSets")
										|| oaiRequest.contains("verb=ListIdentifiers&metadataPrefix=oai_dc")) {
									response.sendRedirect("http://oai.datacite.org/oai?" + oaiRequest);
								}
							} else {
								response.sendRedirect("http://oai.datacite.org/oai");
							}
						}

						break;

					default:
						this.sendMessage(response, HttpStatus.Code.FORBIDDEN, "path has to start with '"
								+ EdalHttpFunctions.ACCEPT + "'" + " or '" + EdalHttpFunctions.REJECT + "'");
						break;
					}

				} catch (IllegalArgumentException e) {
					this.sendMessage(response, HttpStatus.Code.FORBIDDEN, "path has to start with '"
							+ EdalHttpFunctions.ACCEPT + "'" + " or '" + EdalHttpFunctions.REJECT + "'");
				}
			}
		}
		response.flushBuffer();

	}

	private void sendCSVReport(HttpServletResponse response, Code responseCode, String function, String filetype) {

		try {

			response.setStatus(responseCode.getCode());

			response.setContentType("text/comma-separated-values");

			SimpleDateFormat sdf = new SimpleDateFormat("yyyy-MM-dd_HH:mm:ss");

			String filename = "eDAL_report_" + EdalJettyServer.getServerURL().getHost() + "_" + function + "_"
					+ sdf.format(Calendar.getInstance().getTime()) + ".csv";

			response.setHeader("Content-Disposition", "attachment; filename=\"" + filename + "\"");

			final OutputStream responseBody = response.getOutputStream();

			CountDownLatch latch = new CountDownLatch(1);

			OutputStreamWriter outputStreamWriter = null;

			outputStreamWriter = velocityHtmlGenerator.generateCSVForReport(function, filetype, responseCode,
					responseBody, latch);

			try {
				latch.await();
				outputStreamWriter.flush();
				outputStreamWriter.close();

			} catch (EofException eof) {

				DataManager.getImplProv().getLogger().warn("HTTP Response canceled by user");

				outputStreamWriter.close();

			} catch (InterruptedException e) {
				throw new EdalException(e);
			}

		} catch (Exception e) {
			DataManager.getImplProv().getLogger()
					.error("unable to send " + responseCode + "-message : " + e.getClass());
		}
	}

	private boolean ticketAlreadyClicked(int reviewerHashCode, String ticketAccept) {
		List<String> list = EdalJettyHandler.hashmap.get(reviewerHashCode);

		if (list != null) {

			if (list.contains(ticketAccept)) {
				return true;
			} else {
				list.add(ticketAccept);
				EdalJettyHandler.hashmap.put(reviewerHashCode, list);
				return false;
			}
		} else {
			List<String> newlist = new ArrayList<String>();
			newlist.add(ticketAccept);
			EdalJettyHandler.hashmap.put(reviewerHashCode, newlist);
			return false;
		}
	}

	/**
	 * Function to response a {@link EdalHttpFunctions#DOWNLOAD} request.
	 * 
	 * @param response
	 * @param entity
	 * @param uuid
	 * @param versionNumber
	 * @param internalId
	 * @param identifierType
	 * @param reviewerCode
	 * @throws EdalException
	 */
	private void responseDownloadRequest(HttpServletResponse response, PrimaryDataEntity entity, String uuid,
			Long versionNumber, String internalId, String identifierType, String reviewerCode) throws EdalException {

		if (reviewerCode == null) {
			entity = DataManager.getPrimaryDataEntityForPersistenIdentifier(uuid, versionNumber,
					PersistentIdentifier.valueOf(identifierType));
			if (!entity.isDirectory()) {
				this.sendFile((PrimaryDataFile) entity, versionNumber, response);
			} else {
				this.sendEntityMetaDataForPersistentIdentifier(response, entity, versionNumber,
						PersistentIdentifier.valueOf(identifierType), internalId);
			}

		} else {
			entity = DataManager.getPrimaryDataEntityForReviewer(uuid, versionNumber, internalId,
					Integer.parseInt(reviewerCode));

			if (!entity.isDirectory()) {
				this.sendFile((PrimaryDataFile) entity, versionNumber, response);
			} else {
				this.sendEntityMetaDataForReviewer(response, entity, versionNumber, internalId,
						PersistentIdentifier.valueOf(identifierType), Integer.parseInt(reviewerCode));

			}
		}
	}

	private void responseZipRequest(HttpServletResponse response, PrimaryDataEntity entity, String uuid,
			long versionNumber, String internalId, String identifierType, String reviewerId) throws EdalException {

		if (!zipExecutor.isShutdown() && (zipExecutor.getActiveCount() >= CorePoolSizeForZipExecutor)) {

			sendMessage(response, HttpStatus.Code.INSUFFICIENT_STORAGE,
					"No more free slots for downloading zip archive, please try again later");
		} else {

			if (reviewerId == null) {
				entity = DataManager.getPrimaryDataEntityForPersistenIdentifier(uuid, versionNumber,
						PersistentIdentifier.valueOf(identifierType));

			} else {
				entity = DataManager.getPrimaryDataEntityForReviewer(uuid, versionNumber, internalId,
						Integer.parseInt(reviewerId));
			}

			if (entity.isDirectory()) {

				try {

					response.setHeader("Content-Disposition", "inline; filename=\"" + entity.getName() + ".zip" + "\"");
					response.setStatus(HttpStatus.Code.OK.getCode());

					CountDownLatch countDownLatch = new CountDownLatch(1);

					OutputStream responseBody = response.getOutputStream();
					// // copy entity bytes stream to HTTP stream
					ZipOutputStream zipOutputStream = new ZipOutputStream(new BufferedOutputStream(responseBody));

					ZipThread zipThread = new ZipThread(countDownLatch, zipOutputStream, (PrimaryDataDirectory) entity);

					if (zipExecutor.isShutdown()) {
						zipExecutor = new EdalThreadPoolExcecutor(CorePoolSizeForZipExecutor,
								MaximumPoolSizeForZipExecutor, 30, TimeUnit.SECONDS,
								new ArrayBlockingQueue<Runnable>(30));
					}

					zipExecutor.execute(zipThread);

					try {
						countDownLatch.await();
					} catch (InterruptedException e) {
						throw new EdalException("unable to send zip file '" + entity.getName() + ".zip'", e);
					} finally {
						try {
							zipOutputStream.flush();
							zipOutputStream.close();
						} catch (EofException e) {
							DataManager.getImplProv().getLogger()
									.warn("HTTP Zip Response for '" + entity.getName() + ".zip' canceled by user");

							zipThread.stopListThread();
						}
						responseBody.flush();
						responseBody.close();
					}
				} catch (IOException e) {
					throw new EdalException("unable to send zip file '" + entity.getName() + ".zip'", e);
				}
			}
		}
	}

	private void sendEmbeddedFile(HttpServletResponse response, String fileName, String contentType) {

		try {

			InputStream file = EdalJettyHandler.class.getResourceAsStream(fileName);

			int fileSize = file.available();

			response.setContentType(contentType);

			response.setHeader("Content-Disposition", "attachment; filename=\"" + fileName + "\"");
			response.setContentLength(fileSize);

			response.setStatus(HttpStatus.Code.OK.getCode());

			CountDownLatch countDownLatch = new CountDownLatch(1);

			final OutputStream responseBody = response.getOutputStream();
			// copy entity bytes stream to HTTP stream
			final PipedInputStream httpIn = new PipedInputStream(PipedThread.BUFFER_SIZE);
			final PipedOutputStream pipedOut = new PipedOutputStream(httpIn);

			final PipedThread pipedThread = new PipedThread(httpIn, responseBody, countDownLatch, fileSize);

			executor.execute(pipedThread);

			byte[] buffer = new byte[PipedThread.BUFFER_SIZE];
			int length;
			while ((length = file.read(buffer)) != -1) {
				pipedOut.write(buffer, 0, length);
			}

			pipedOut.flush();

			try {
				countDownLatch.await();
				file.close();
				pipedOut.close();
				httpIn.close();
			} catch (InterruptedException e) {
				DataManager.getImplProv().getLogger().error("unable to wait for sending file : " + e.getMessage());
			}
			responseBody.close();

		} catch (IOException e) {
			DataManager.getImplProv().getLogger().error("unable to send file : " + e.getMessage());
		}

	}

	/**
	 * Generate a HTML output with the
	 * {@link de.ipk_gatersleben.bit.bi.edal.primary_data.metadata.MetaData} of
	 * the given {@link PrimaryDataEntity} and send the output over HTTP.
	 * 
	 * @param response
	 *            the corresponding HTTP exchange.
	 * @param entity
	 *            the the corresponding {@link PrimaryDataEntity}.
	 * @throws EdalException
	 *             if unable to send metadata
	 */
	private void sendEntityMetaData(final HttpServletResponse response, final PrimaryDataEntity entity)
			throws EdalException {

		try {

			StringWriter w = null;
			if (entity.isDirectory()) {
				w = velocityHtmlGenerator.generateHtmlForDirectory((PrimaryDataDirectory) entity);
			} else {
				w = velocityHtmlGenerator.generateHtmlForFile((PrimaryDataFile) entity);
			}

			ByteArrayInputStream bis = new ByteArrayInputStream(String.valueOf(w.getBuffer()).getBytes());

			response.setContentType("text/html");
			response.setStatus(HttpStatus.Code.OK.getCode());
			final OutputStream responseBody = response.getOutputStream();

			CountDownLatch countDownLatch = new CountDownLatch(1);

			final PipedInputStream httpIn = new PipedInputStream(PipedThread.BUFFER_SIZE);
			final PipedOutputStream pipedOut = new PipedOutputStream(httpIn);

			final PipedThread pipedThread = new PipedThread(httpIn, responseBody, countDownLatch, bis.available());

			executor.execute(pipedThread);

			byte[] buffer = new byte[PipedThread.BUFFER_SIZE];
			int length;
			while ((length = bis.read(buffer)) != -1) {
				pipedOut.write(buffer, 0, length);
			}

			try {
				countDownLatch.await();
			} catch (InterruptedException e) {
				throw new EdalException("unable to send html page", e);
			} finally {
				try {
					bis.close();
					pipedOut.close();
					httpIn.close();
					responseBody.close();
				} catch (Exception e) {
					DataManager.getImplProv().getLogger().error("unable to close html streams : " + e.getMessage());
				}
			}
		} catch (IOException e) {
			throw new EdalException("unable to send the html page over HTTP", e);
		} catch (EdalException e) {
			throw e;
		}

	}

	/**
	 * Generate a HTML output with the
	 * {@link de.ipk_gatersleben.bit.bi.edal.primary_data.metadata.MetaData} of
	 * the given {@link PrimaryDataEntity} and send the output over HTTP.
	 * 
	 * @param response
	 *            the corresponding HTTP exchange.
	 * @param entity
	 *            the the corresponding {@link PrimaryDataEntity}.
	 * @param versionNumber
	 *            the version number of this {@link PrimaryDataEntity}
	 * @param internalId
	 *            the id of the {@link PublicReference}
	 * @param identifierType
	 *            the type of the {@link PublicReference}
	 * @throws EdalException
	 *             if unable to send {@link MetaData}
	 */
	private void sendEntityMetaDataForPersistentIdentifier(final HttpServletResponse response,
			final PrimaryDataEntity entity, final long versionNumber, final PersistentIdentifier identifierType,
			final String internalId) throws EdalException {

		String cacheKey = generateCacheKey(identifierType, internalId, entity, versionNumber, 0);

		if (cache.get(cacheKey) == null) {

			DataManager.getImplProv().getLogger().debug("Regenerate Webpage : " + cacheKey);

			try {

				ByteArrayOutputStream cacheFileOutputStream = new ByteArrayOutputStream();

				TeeOutputStream teeOutputStream = new TeeOutputStream(response.getOutputStream(),
						cacheFileOutputStream);

				CountDownLatch latch = new CountDownLatch(1);

				OutputStreamWriter outputStreamWriter = null;

				if (entity.isDirectory()) {
					outputStreamWriter = velocityHtmlGenerator.generateHtmlForDirectoryOfSnapshot(
							(PrimaryDataDirectory) entity, versionNumber, internalId, identifierType, teeOutputStream,
							latch);
				} else {
					outputStreamWriter = velocityHtmlGenerator.generateHtmlForFileOfSnapshot((PrimaryDataFile) entity,
							versionNumber, identifierType, internalId, teeOutputStream, latch);
				}

				response.setContentType("text/html");
				response.setStatus(HttpStatus.Code.OK.getCode());

				try {
					latch.await();
					outputStreamWriter.flush();
					outputStreamWriter.close();
					teeOutputStream.flush();
					teeOutputStream.close();

					cache.put(cacheKey, cacheFileOutputStream);
				} catch (EofException eof) {

					DataManager.getImplProv().getLogger().warn("HTTP Response canceled by user");

					outputStreamWriter.close();
					teeOutputStream.flush();
					teeOutputStream.close();

				} catch (InterruptedException e) {
					throw new EdalException(e);
				}

			} catch (IOException e) {
				throw new EdalException(e);
			}
		}

		else {
			DataManager.getImplProv().getLogger().debug("Reload Webpage from Cache : " + cacheKey);

			response.setContentType("text/html");
			response.setStatus(HttpStatus.Code.OK.getCode());

			try {

				ByteArrayOutputStream cacheFileInputStream = cache.get(cacheKey);

				cacheFileInputStream.writeTo(response.getOutputStream());
				cacheFileInputStream.close();

				response.getOutputStream().flush();
				response.getOutputStream().close();

			} catch (IOException e) {
				e.printStackTrace();
			}

		}

	}

	/**
	 * Generate a HTML output with the
	 * {@link de.ipk_gatersleben.bit.bi.edal.primary_data.metadata.MetaData} of
	 * the given {@link PrimaryDataEntity} and send the output over HTTP. This
	 * function is especially for the Output of the landing page for a reviewer,
	 * who should approve the requested
	 * {@link de.ipk_gatersleben.bit.bi.edal.primary_data.file.PublicReference}.
	 * 
	 * @param response
	 *            the corresponding HTTP exchange.
	 * @param entity
	 *            the the corresponding {@link PrimaryDataEntity}.
	 * @param reviewerCode
	 *            the reviewerCode to identify a reviewer
	 * @param versionNumber
	 *            the version number of this {@link PrimaryDataEntity}
	 * @param internalId
	 *            the id of the {@link PublicReference}
	 * @param identifierType
	 *            the type of the {@link PublicReference}
	 * @throws EdalException
	 *             if unable to send {@link MetaData}
	 */
	private void sendEntityMetaDataForReviewer(final HttpServletResponse response, final PrimaryDataEntity entity,
			final long versionNumber, final String internalId, final PersistentIdentifier identifierType,
			final int reviewerCode) throws EdalException {

		String cacheKey = generateCacheKey(identifierType, internalId, entity, versionNumber, reviewerCode);

		if (cache.get(cacheKey) == null) {

			DataManager.getImplProv().getLogger().debug("Regenerate Webpage : " + cacheKey);

			try {

				ByteArrayOutputStream cacheFileOutputStream = new ByteArrayOutputStream();

				TeeOutputStream teeOutputStream = new TeeOutputStream(response.getOutputStream(),
						cacheFileOutputStream);

				CountDownLatch latch = new CountDownLatch(1);

				OutputStreamWriter outputStreamWriter = null;

				if (entity.isDirectory()) {
					outputStreamWriter = velocityHtmlGenerator.generateHtmlForDirectoryForReviewer(
							(PrimaryDataDirectory) entity, versionNumber, internalId, identifierType, reviewerCode,
							teeOutputStream, latch);
				} else {
					outputStreamWriter = velocityHtmlGenerator.generateHtmlForFileForReviewer((PrimaryDataFile) entity,
							versionNumber, internalId, identifierType, reviewerCode, teeOutputStream, latch);
				}

				response.setContentType("text/html");
				response.setStatus(HttpStatus.Code.OK.getCode());

				try {
					latch.await();
					outputStreamWriter.flush();
					outputStreamWriter.close();
					teeOutputStream.flush();
					teeOutputStream.close();

					// cache.put(cacheKey, cacheFileOutputStream);

				} catch (EofException eof) {

					DataManager.getImplProv().getLogger().warn("HTTP Response canceled by user");

					outputStreamWriter.close();
					teeOutputStream.flush();
					teeOutputStream.close();

				} catch (InterruptedException e) {
					throw new EdalException(e);
				}

			} catch (IOException e) {
				throw new EdalException(e);
			}
		}

		else {
			DataManager.getImplProv().getLogger().debug("Reload Webpage from Cache : " + cacheKey);

			response.setContentType("text/html");
			response.setStatus(HttpStatus.Code.OK.getCode());

			try {

				ByteArrayOutputStream cacheFileInputStream = cache.get(cacheKey);

				cacheFileInputStream.writeTo(response.getOutputStream());
				cacheFileInputStream.close();

				response.getOutputStream().flush();
				response.getOutputStream().close();

			} catch (IOException e) {
				e.printStackTrace();
			}

		}

	}

	private final String generateCacheKey(final PersistentIdentifier identifierType, final String internalId,
			final PrimaryDataEntity entity, final long versionNumber, final int reviewerCode) {
		return new String(identifierType.toString() + internalId + entity.getID() + String.valueOf(versionNumber)
				+ String.valueOf(reviewerCode));
	}

	/**
	 * Send the data of the corresponding {@link PrimaryDataFile} to the HTTP
	 * {@link OutputStream}.
	 * 
	 * @param file
	 *            the corresponding {@link PrimaryDataFile} to send.
	 * @param response
	 * @throws EdalException
	 *             if unable to send {@link PrimaryDataFile}
	 */
	private void sendFile(final PrimaryDataFile file, final long versionNumber, final HttpServletResponse response)
			throws EdalException {

		final PrimaryDataFile currentFile = file;
		try {
			currentFile.switchCurrentVersion(currentFile.getVersionByRevisionNumber(versionNumber));
		} catch (PrimaryDataEntityVersionException e1) {
			this.sendMessage(response, HttpStatus.Code.NOT_FOUND, "no file found");
		}

		try {
			String type = "";
			Long size = null;
			try {
				type = ((DataFormat) currentFile.getCurrentVersion().getMetaData()
						.getElementValue(EnumDublinCoreElements.FORMAT)).getMimeType();
				size = ((DataSize) currentFile.getCurrentVersion().getMetaData()
						.getElementValue(EnumDublinCoreElements.SIZE)).getFileSize();

			} catch (final MetaDataException e) {
				throw new EdalException("unable to load the MIME type/file size", e);
			}

			response.setContentType(type);
			if (size > (LIMIT_INLINE_FILE_SIZE)) {
				response.setHeader("Content-Disposition", "attachment; filename=\"" + currentFile.getName() + "\"");
			} else {
				response.setHeader("Content-Disposition", "inline; filename=\"" + currentFile.getName() + "\"");
			}
			/* Do not use setContentLenght, because int could be too small */
			/* response.setContentLength(size.intValue()); */
			response.setContentLengthLong(size);

			response.setStatus(HttpStatus.Code.OK.getCode());

			CountDownLatch countDownLatch = new CountDownLatch(1);

			final OutputStream responseBody = response.getOutputStream();
			// copy entity bytes stream to HTTP stream
			final PipedInputStream httpIn = new PipedInputStream(PipedThread.BUFFER_SIZE);
			final PipedOutputStream pipedOut = new PipedOutputStream(httpIn);

			final PipedThread pipedThread = new PipedThread(httpIn, responseBody, countDownLatch, size);

			executor.execute(pipedThread);

			currentFile.read(pipedOut);

			try {
				countDownLatch.await();
			} catch (InterruptedException e) {
				throw new EdalException("unable to send file", e);
			} finally {
				try {
					pipedOut.close();
					httpIn.close();
					responseBody.close();
				} catch (Exception e) {
					DataManager.getImplProv().getLogger().error("unable to close download streams : " + e.getMessage());
				}
			}
		} catch (final PrimaryDataFileException | IOException e) {
			throw new EdalException("unable to send file", e);
		}
	}

	/**
	 * Send a response with the given message and responseCode.
	 * 
	 * @param response
	 *            the corresponding HTTP exchange.
	 * @param responseCode
	 *            the response code for this message (e.g. 200,404...).
	 * @param message
	 *            the message to send.
	 */
	private void sendMessage(final HttpServletResponse response, final HttpStatus.Code responseCode,
			final String message) {

		try {

			final String htmlOutput = velocityHtmlGenerator.generateHtmlForErrorMessage(responseCode, message)
					.toString();

			response.setStatus(responseCode.getCode());

			response.setContentType("text/html");

			final OutputStream responseBody = response.getOutputStream();
			responseBody.write(htmlOutput.getBytes());
			responseBody.close();

		} catch (EofException eof) {
			// Do nothing, because response was already send
		} catch (IOException | EdalException e) {
			DataManager.getImplProv().getLogger().error("unable to send " + responseCode + "-message : " + e);
		}
	}

	/**
	 * Send a response with the given message and responseCode.
	 * 
	 * @param response
	 *            the corresponding HTTP exchange.
	 * @param responseCode
	 *            the response code for this message (e.g. 200,404...).
	 */
	private void sendReport(final HttpServletResponse response, final HttpStatus.Code responseCode, String function) {

		try {

			response.setStatus(responseCode.getCode());

			response.setContentType("text/html");

			final OutputStream responseBody = response.getOutputStream();

			CountDownLatch latch = new CountDownLatch(1);

			OutputStreamWriter outputStreamWriter = null;

			outputStreamWriter = velocityHtmlGenerator.generateHtmlForReport(function, responseCode, responseBody,
					latch);

			try {
				latch.await();
				outputStreamWriter.flush();
				outputStreamWriter.close();

			} catch (EofException eof) {

				DataManager.getImplProv().getLogger().warn("HTTP Response canceled by user");

				outputStreamWriter.close();

			} catch (InterruptedException e) {
				throw new EdalException(e);
			}

		} catch (Exception e) {
			DataManager.getImplProv().getLogger()
					.error("unable to send " + responseCode + "-message : " + e.getClass());
		}
	}

	public static void deleteTicketFromHashMap(String ticket) {

		for (Entry<Integer, List<String>> entry : hashmap.entrySet()) {

			List<String> list = entry.getValue();

			if (list.contains(ticket)) {
				list.remove(ticket);
			}
		}
	}
}