/*
 * Copyright (c) 2016 Leibniz Institute of Plant Genetics and Crop Plant Research (IPK), Gatersleben, Germany.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Creative Commons Attribution-NoDerivatives 4.0 International (CC BY-ND 4.0)
 * which accompanies this distribution, and is available at http://creativecommons.org/licenses/by-nd/4.0/
 *
 * Contributors:
 *      Leibniz Institute of Plant Genetics and Crop Plant Research (IPK), Gatersleben, Germany - initial API and implementation
 */
package de.ipk_gatersleben.bit.bi.edal.primary_data.file.implementation;

import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.ObjectOutputStream;
import java.nio.file.FileVisitResult;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.SimpleFileVisitor;
import java.nio.file.attribute.BasicFileAttributes;
import java.util.List;
import java.util.concurrent.atomic.AtomicLong;

import org.hibernate.Session;
import org.hibernate.criterion.Restrictions;

import de.ipk_gatersleben.bit.bi.edal.primary_data.DataManager;
import de.ipk_gatersleben.bit.bi.edal.primary_data.EdalJettyHandler;
import de.ipk_gatersleben.bit.bi.edal.primary_data.ServiceProvider;
import de.ipk_gatersleben.bit.bi.edal.primary_data.file.EdalException;
import de.ipk_gatersleben.bit.bi.edal.primary_data.file.PrimaryDataDirectory;
import de.ipk_gatersleben.bit.bi.edal.primary_data.file.PrimaryDataDirectoryException;
import de.ipk_gatersleben.bit.bi.edal.primary_data.file.PrimaryDataEntity;
import de.ipk_gatersleben.bit.bi.edal.primary_data.file.PrimaryDataEntityVersion;
import de.ipk_gatersleben.bit.bi.edal.primary_data.file.PublicReference;
import de.ipk_gatersleben.bit.bi.edal.primary_data.metadata.DataSize;
import de.ipk_gatersleben.bit.bi.edal.primary_data.metadata.EnumDublinCoreElements;
import de.ipk_gatersleben.bit.bi.edal.primary_data.metadata.MetaDataException;
import de.ipk_gatersleben.bit.bi.edal.primary_data.reference.PublicationStatus;

/**
 * Implementaion of the {@link ServiceProvider} interface
 * 
 * @author arendd
 *
 */

@SuppressWarnings("unchecked")
public class ServiceProviderImplementation implements ServiceProvider {

	public static final Path PATH_FOR_DIRECTORY_SIZE_MAP = Paths
			.get(DataManager.getImplProv().getConfiguration().getMountPath().toString(), "directory_size_map.dat");

	public static final Path PATH_FOR_TOTAL_FILE_NUMBER = Paths
			.get(DataManager.getImplProv().getConfiguration().getMountPath().toString(), "number_of_files.dat");

	public static final Path PATH_FOR_TOTAL_VOLUME = Paths
			.get(DataManager.getImplProv().getConfiguration().getMountPath().toString(), "total_volume.dat");

	public static Long numberOfFiles;

	/** {@inheritDoc} */
	@Override
	public Long getUsedStorageSpace() throws EdalException {

		if (DataManager.getImplProv().getConfiguration().getDataPath() == null || Files
				.notExists(DataManager.getImplProv().getConfiguration().getDataPath(), LinkOption.NOFOLLOW_LINKS)) {
			throw new EdalException("No mount path defined, please run getRootDirectory first");
		}
		try {
			final AtomicLong size = new AtomicLong();
			Files.walkFileTree(DataManager.getImplProv().getConfiguration().getDataPath(),
					new SimpleFileVisitor<Path>() {
						@Override
						public FileVisitResult visitFile(final Path file, final BasicFileAttributes attrs)
								throws IOException {
							size.addAndGet(attrs.size());
							return FileVisitResult.CONTINUE;
						}
					});
			return size.longValue();
		} catch (final IOException e) {
			throw new EdalException("Unable to request used space", e);
		}
	}

	/** {@inheritDoc} */
	@Override
	public Long getAvailableStorageSpace() throws EdalException {

		if (DataManager.getImplProv().getConfiguration().getDataPath() == null || Files
				.notExists(DataManager.getImplProv().getConfiguration().getDataPath(), LinkOption.NOFOLLOW_LINKS)) {
			throw new EdalException("No mount path defined, please run getRootDirectory first");
		}
		try {
			return Files.getFileStore(DataManager.getImplProv().getConfiguration().getDataPath()).getUsableSpace();
		} catch (final IOException e) {
			throw new EdalException("Unable to request available space", e);
		}
	}

	@Override
	public synchronized void cleanUp() {

		((FileSystemImplementationProvider) DataManager.getImplProv()).getLogger().warn("'CLEAN UP' initiated");

		Session session = ((FileSystemImplementationProvider) DataManager.getImplProv()).getSession();
		session.beginTransaction();

		List<PublicReferenceImplementation> references = (List<PublicReferenceImplementation>) session
				.createCriteria(PublicReferenceImplementation.class)
				.add(Restrictions.eq("publicationStatus", PublicationStatus.REJECTED)).list();

		session.getTransaction().commit();
		session.close();

		boolean foundReferenceToDelete = false;

		if (references != null) {
			foundReferenceToDelete = true;
		}

		for (PublicReferenceImplementation reference : references) {

			PrimaryDataEntity entity = reference.getVersion().getEntity();

			if (entity.getVersions().size() == 3 && !entity.isDirectory()) {

				// System.out.println("Delete single file : " +
				// entity.getName());

				deleteFileAndPermissions((PrimaryDataFileImplementation) entity);

			}

			else if (entity.getVersions().size() == 3 && entity.isDirectory()) {

				// System.out.println("Delete complete directory : " +
				// entity.getName());

				deleteRecursiveDirectory((PrimaryDataDirectoryImplementation) entity);

			}
		}

		if (foundReferenceToDelete) {
			((FileSystemImplementationProvider) DataManager.getImplProv()).getIndexThread().resetIndexThread();
		}

		((FileSystemImplementationProvider) DataManager.getImplProv()).getLogger().warn("'CLEAN UP' finished");

	}

	private void deleteRecursiveDirectory(PrimaryDataDirectoryImplementation directory) {

		Session session = ((FileSystemImplementationProvider) DataManager.getImplProv()).getSession();
		session.beginTransaction();

		try {
			deleteFilesRecursively(session, directory);
			deleteDirectoriesRecursively(session, directory);
		} catch (PrimaryDataDirectoryException e) {
			e.printStackTrace();
		}

		deleteDirectoryAndPermissions(directory);

		session.getTransaction().commit();
		session.close();

	}

	public void deleteDirectoriesRecursively(Session session, final PrimaryDataDirectory currentDirectory)
			throws PrimaryDataDirectoryException {

		final List<PrimaryDataEntity> list = currentDirectory.listPrimaryDataEntities();

		if (list != null) {
			for (final PrimaryDataEntity primaryDataEntity : list) {
				if (primaryDataEntity.isDirectory()) {
					if (((PrimaryDataDirectory) primaryDataEntity).listPrimaryDataEntities().size() == 0) {
						deleteDirectoryAndPermissions((PrimaryDataDirectoryImplementation) primaryDataEntity);
					} else {
						deleteDirectoriesRecursively(session, (PrimaryDataDirectory) primaryDataEntity);
						deleteDirectoryAndPermissions((PrimaryDataDirectoryImplementation) primaryDataEntity);
					}
				}
			}
		}
	}

	public void deleteFilesRecursively(Session session, final PrimaryDataDirectory currentDirectory)
			throws PrimaryDataDirectoryException {

		final List<PrimaryDataEntity> list = currentDirectory.listPrimaryDataEntities();

		if (list != null) {
			for (final PrimaryDataEntity primaryDataEntity : list) {
				if (primaryDataEntity.isDirectory()) {
					deleteFilesRecursively(session, (PrimaryDataDirectory) primaryDataEntity);
				} else {
					deleteFileAndPermissions((PrimaryDataFileImplementation) primaryDataEntity);

				}
			}
		}
	}

	private void deleteFileAndPermissions(PrimaryDataFileImplementation file) {

		// System.out.println("Deleting local File");

		for (PrimaryDataEntityVersion version : file.getVersions()) {

			Path path = file.getPathToLocalFile(version);

			try {
				Files.delete(path);
			} catch (IOException e) {
				e.printStackTrace();
			}
		}

		// System.out.println("Deleting PrimaryDataFile");

		Session session = ((FileSystemImplementationProvider) DataManager.getImplProv()).getSession();
		session.beginTransaction();

		session.delete(file);

		List<EdalPermissionImplementation> permissions = (List<EdalPermissionImplementation>) session
				.createCriteria(EdalPermissionImplementation.class).add(Restrictions.eq("internId", file.getID()))
				.list();

		for (EdalPermissionImplementation permission : permissions) {
			session.delete(permission);
		}

		session.getTransaction().commit();
		session.close();

	}

	private void deleteDirectoryAndPermissions(PrimaryDataDirectoryImplementation directory) {

		// System.out.println("Deleting PrimaryDataDirectory");

		Session session = ((FileSystemImplementationProvider) DataManager.getImplProv()).getSession();
		session.beginTransaction();

		session.delete(directory);

		List<EdalPermissionImplementation> permissions = (List<EdalPermissionImplementation>) session
				.createCriteria(EdalPermissionImplementation.class).add(Restrictions.eq("internId", directory.getID()))
				.list();

		for (EdalPermissionImplementation permission : permissions) {
			session.delete(permission);
		}

		session.getTransaction().commit();
		session.close();
	}

	private Long listDir(final PublicReference reference, PrimaryDataDirectory directory)
			throws PrimaryDataDirectoryException, MetaDataException {

		if (CalculateDirectorySizeThread.directorySizes
				.containsKey(reference.getInternalID() + "/" + directory.getID())) {
			return CalculateDirectorySizeThread.directorySizes.get(reference.getInternalID() + "/" + directory.getID());
		}

		final List<PrimaryDataEntity> list = directory.listPrimaryDataEntities();
		Long dirSize = new Long(0);
		if (list != null) {

			for (final PrimaryDataEntity primaryDataEntity : list) {

				DataSize mySize = primaryDataEntity.getMetaData().getElementValue(EnumDublinCoreElements.SIZE);
				dirSize = dirSize + mySize.getFileSize();

				if (primaryDataEntity.isDirectory()) {
					dirSize = dirSize + listDir(reference, (PrimaryDataDirectory) primaryDataEntity);
				} else {
					if (reference.getPublicationStatus().equals(PublicationStatus.ACCEPTED)) {
						numberOfFiles += 1;
					}
				}
			}
			CalculateDirectorySizeThread.directorySizes.put(reference.getInternalID() + "/" + directory.getID(),
					dirSize);

		}
		return dirSize;
	}

	private void storeValuesToDisk() {

		File file = ServiceProviderImplementation.PATH_FOR_DIRECTORY_SIZE_MAP.toFile();

		try {
			ObjectOutputStream oos = new ObjectOutputStream(new FileOutputStream(file));
			oos.writeObject(CalculateDirectorySizeThread.directorySizes);
			oos.close();
		} catch (Exception e) {
			((FileSystemImplementationProvider) DataManager.getImplProv()).getLogger().error(e);
		}

		file = ServiceProviderImplementation.PATH_FOR_TOTAL_FILE_NUMBER.toFile();
		try {
			ObjectOutputStream oos = new ObjectOutputStream(new FileOutputStream(file));
			oos.writeObject(ServiceProviderImplementation.numberOfFiles);
			oos.close();
		} catch (Exception e) {
			((FileSystemImplementationProvider) DataManager.getImplProv()).getLogger().error(e);
		}

		file = ServiceProviderImplementation.PATH_FOR_TOTAL_VOLUME.toFile();
		try {
			ObjectOutputStream oos = new ObjectOutputStream(new FileOutputStream(file));
			oos.writeObject(CalculateDirectorySizeThread.totalVolumeDataStock);
			oos.close();
		} catch (Exception e) {
			((FileSystemImplementationProvider) DataManager.getImplProv()).getLogger().error(e);
		}

	}

	@Override
	public synchronized void calculateDirectorySizes() {

//		System.out.println("START Calculate Thread");

		Session session = ((FileSystemImplementationProvider) DataManager.getImplProv()).getSession();

		List<PublicReferenceImplementation> references = (List<PublicReferenceImplementation>) session
				.createCriteria(PublicReferenceImplementation.class).list();

		session.close();

		boolean updated = false;

		for (PublicReferenceImplementation reference : references) {

			if (!CalculateDirectorySizeThread.directorySizes
					.containsKey(reference.getInternalID() + "/" + reference.getVersion().getEntity().getID())) {

				try {
					Long size = listDir(reference, ((PrimaryDataDirectory) reference.getVersion().getEntity()));

					if (reference.getPublicationStatus().equals(PublicationStatus.ACCEPTED)) {
						CalculateDirectorySizeThread.totalVolumeDataStock += size;
					}
					storeValuesToDisk();
					updated = true;
				} catch (PrimaryDataDirectoryException | MetaDataException e) {
					e.printStackTrace();
				}
			}
		}

		if (updated) {
			// System.out.println("CLEANED WEB_PAGE_CACHE");
			EdalJettyHandler.cache.clean();
		}
	}

	@Override
	public int getNumberOfUsers() {
		Session session = ((FileSystemImplementationProvider) DataManager.getImplProv()).getSession();
		session.beginTransaction();

		List<PrincipalImplementation> numberOfUsers = (List<PrincipalImplementation>) session
				.createCriteria(PrincipalImplementation.class).list();

		session.getTransaction().commit();
		session.close();

		return numberOfUsers.size();

	}

}
