/*
 * Copyright (c) 2017 Leibniz Institute of Plant Genetics and Crop Plant Research (IPK), Gatersleben, Germany.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Creative Commons Attribution-NoDerivatives 4.0 International (CC BY-ND 4.0)
 * which accompanies this distribution, and is available at http://creativecommons.org/licenses/by-nd/4.0/
 *
 * Contributors:
 *      Leibniz Institute of Plant Genetics and Crop Plant Research (IPK), Gatersleben, Germany - initial API and implementation
 */
package de.ipk_gatersleben.bit.bi.edal.primary_data.file.implementation;

import java.util.Calendar;
import java.util.List;
import java.util.SortedSet;

import org.hibernate.Criteria;
import org.hibernate.Session;
import org.hibernate.criterion.Restrictions;

import de.ipk_gatersleben.bit.bi.edal.primary_data.DataManager;
import de.ipk_gatersleben.bit.bi.edal.primary_data.HttpServiceProvider;
import de.ipk_gatersleben.bit.bi.edal.primary_data.file.EdalException;
import de.ipk_gatersleben.bit.bi.edal.primary_data.file.PrimaryDataDirectory;
import de.ipk_gatersleben.bit.bi.edal.primary_data.file.PrimaryDataDirectoryException;
import de.ipk_gatersleben.bit.bi.edal.primary_data.file.PrimaryDataEntity;
import de.ipk_gatersleben.bit.bi.edal.primary_data.file.PrimaryDataEntityVersion;
import de.ipk_gatersleben.bit.bi.edal.primary_data.file.PrimaryDataEntityVersionException;
import de.ipk_gatersleben.bit.bi.edal.primary_data.file.PrimaryDataFile;
import de.ipk_gatersleben.bit.bi.edal.primary_data.file.PublicReference;
import de.ipk_gatersleben.bit.bi.edal.primary_data.reference.PersistentIdentifier;

/**
 * Implementation of {@link HttpServiceProvider} interface.
 * 
 * @author arendd
 *
 */
public class HttpServiceProviderImplementation implements HttpServiceProvider {

	/** {@inheritDoc} */
	@Override
	public PrimaryDataEntity getPrimaryDataEntityByID(final String uuid, final long versionNumber)
			throws EdalException {

		final Session session = ((FileSystemImplementationProvider) DataManager.getImplProv()).getSession();

		final Criteria getFile = session.createCriteria(PrimaryDataFileImplementation.class)
				.add(Restrictions.eq("class", PrimaryDataFileImplementation.class)).add(Restrictions.eq("ID", uuid));

		final PrimaryDataFile file = (PrimaryDataFile) getFile.uniqueResult();

		if (file == null) {
			final Criteria getDirectory = session.createCriteria(PrimaryDataDirectoryImplementation.class)
					.add(Restrictions.eq("class", PrimaryDataDirectoryImplementation.class))
					.add(Restrictions.eq("ID", uuid));

			final PrimaryDataDirectory directory = (PrimaryDataDirectory) getDirectory.uniqueResult();

			if (directory == null) {
				throw new EdalException("found no entity with ID '" + uuid + "'");
			} else {

				PrimaryDataEntityVersion version;
				try {
					version = directory.getVersionByRevisionNumber(versionNumber);
				} catch (PrimaryDataEntityVersionException e) {
					/** found no version with this number */
					throw new EdalException(e.getMessage(), e);
				}

				try {
					directory.switchCurrentVersion(version);
				} catch (PrimaryDataEntityVersionException e) {
					throw new EdalException("unable to switch the version with the number " + versionNumber, e);
				}

			}

			final List<PublicReference> list = directory.getCurrentVersion().getPublicReferences();

			for (final PublicReference publicReference : list) {
				if (publicReference.isPublic()) {
					if (publicReference.getReleaseDate() == null) {
						return directory;
					} else {
						if (publicReference.getReleaseDate().after(Calendar.getInstance())) {
							throw new EdalException("the PublicReference for this version of '" + directory.getName()
									+ "' is locked until " + publicReference.getReleaseDate().getTime());
						} else {
							return directory;
						}
					}
				}
			}
			throw new EdalException("found no PublicReference for this version of " + directory.getName());

		} else {

			PrimaryDataEntityVersion version;
			try {
				version = file.getVersionByRevisionNumber(versionNumber);
			} catch (PrimaryDataEntityVersionException e) {
				/** found no version with this number */
				throw new EdalException(e.getMessage(), e);
			}

			try {
				file.switchCurrentVersion(version);
			} catch (PrimaryDataEntityVersionException e) {
				throw new EdalException("unable to switch the version with the number " + versionNumber, e);
			}

			final List<PublicReference> list = file.getCurrentVersion().getPublicReferences();
			for (final PublicReference publicReference : list) {
				if (publicReference.isPublic()) {

					if (publicReference.getReleaseDate() == null) {
						return file;
					} else {
						if (publicReference.getReleaseDate().after(Calendar.getInstance())) {
							throw new EdalException("the PublicReference for this version of '" + file.getName()
									+ "' is locked until " + publicReference.getReleaseDate().getTime());
						} else {
							return file;
						}
					}
				}
			}
			throw new EdalException("found no PublicReference for this version of " + file.getName());
		}

	}

	@Override
	public PrimaryDataEntity getPrimaryDataEntityForPersistentIdentifier(String uuid, long versionNumber,
			PersistentIdentifier persistentIdentifier) throws EdalException {

		final Session session = ((FileSystemImplementationProvider) DataManager.getImplProv()).getSession();

		PrimaryDataFile file = (PrimaryDataFile) session.createCriteria(PrimaryDataFileImplementation.class)
				.add(Restrictions.eq("class", PrimaryDataFileImplementation.class)).add(Restrictions.eq("ID", uuid))
				.uniqueResult();

		if (file == null) {

			PrimaryDataDirectory directory = (PrimaryDataDirectory) session
					.createCriteria(PrimaryDataDirectoryImplementation.class)
					.add(Restrictions.eq("class", PrimaryDataDirectoryImplementation.class))
					.add(Restrictions.eq("ID", uuid)).uniqueResult();

			if (directory == null) {
				session.close();
				throw new EdalException("no entity with ID '" + uuid + "' found !");
			} else {

				try {
					PublicReference reference = directory.getVersionByRevisionNumber(versionNumber)
							.getPublicReference(persistentIdentifier);
					if (reference.isPublic()) {
						if (reference.getReleaseDate() == null) {
							session.close();
							return directory;
						} else {
							if (reference.getReleaseDate().after(Calendar.getInstance())) {
								session.close();
								throw new EdalException(
										"the PublicReference for this version of '" + directory.getName()
												+ "' is locked until " + reference.getReleaseDate().getTime());
							} else {
								session.close();
								return directory;
							}
						}

					} else {
						session.close();
						return searchRekursiveForPersistentIdentifiers(directory, versionNumber, persistentIdentifier);
					}
				} catch (PrimaryDataEntityVersionException e) {
					session.close();
					return searchRekursiveForPersistentIdentifiers(directory, versionNumber, persistentIdentifier);
				}
			}
		} else {
			try {
				PublicReference reference = file.getVersionByRevisionNumber(versionNumber)
						.getPublicReference(persistentIdentifier);

				if (reference.isPublic()) {
					if (reference.getReleaseDate() == null) {
						session.close();
						return file;
					} else {
						if (reference.getReleaseDate().after(Calendar.getInstance())) {
							session.close();
							throw new EdalException("the PublicReference for this version of '" + file.getName()
									+ "' is locked until " + reference.getReleaseDate().getTime());
						} else {
							session.close();
							return file;
						}
					}
				} else {
					session.close();
					return searchRekursiveForPersistentIdentifiers(file, versionNumber, persistentIdentifier);
				}
			} catch (PrimaryDataEntityVersionException e) {
				session.close();
				return searchRekursiveForPersistentIdentifiers(file, versionNumber, persistentIdentifier);
			}
		}
	}

	/** {@inheritDoc} */
	@Override
	public PrimaryDataEntity getPrimaryDataEntityForReviewer(String uuid, long versionNumber, String internalId,
			int reviewerCode) throws EdalException {

		final Session session = ((FileSystemImplementationProvider) DataManager.getImplProv()).getSession();

		ReviewersImplementation reviewer = (ReviewersImplementation) session
				.createCriteria(ReviewersImplementation.class).add(Restrictions.eq("hashCode", reviewerCode))
				.uniqueResult();

		if (reviewer != null) {

			PrimaryDataFile file = (PrimaryDataFile) session.createCriteria(PrimaryDataFileImplementation.class)
					.add(Restrictions.eq("class", PrimaryDataFileImplementation.class)).add(Restrictions.eq("ID", uuid))
					.uniqueResult();

			if (file != null) {
				try {
					PrimaryDataEntityVersion version = file.getVersionByRevisionNumber(versionNumber);
					file.switchCurrentVersion(version);
					session.close();
					return file;
				} catch (PrimaryDataEntityVersionException e) {
					session.close();
					throw new EdalException(e.getMessage(), e);
				}
			} else {
				PrimaryDataDirectory directory = (PrimaryDataDirectory) session
						.createCriteria(PrimaryDataDirectoryImplementation.class)
						.add(Restrictions.eq("class", PrimaryDataDirectoryImplementation.class))
						.add(Restrictions.eq("ID", uuid)).uniqueResult();
				if (directory == null) {
					session.close();
					throw new EdalException("no entity with ID '" + uuid + "' found !");
				} else {
					try {
						PrimaryDataEntityVersion version = directory.getVersionByRevisionNumber(versionNumber);
						directory.switchCurrentVersion(version);
						session.close();
						return directory;
					} catch (PrimaryDataEntityVersionException e) {
						session.close();
						throw new EdalException(e.getMessage(), e);
					}
				}
			}
		} else {
			session.close();
			throw new EdalException("no reviewer with ID '" + reviewerCode + "' found !");
		}
	}

	private PrimaryDataEntity searchRekursiveForPersistentIdentifiers(PrimaryDataEntity entity, long versionNumber,
			PersistentIdentifier persistentIdentifier) throws EdalException {

		PrimaryDataEntityVersion version = null;
		try {
			version = entity.getVersionByRevisionNumber(versionNumber);
		} catch (PrimaryDataEntityVersionException e) {
			throw new EdalException(e.getMessage());
		}
		boolean ready = false;

		PrimaryDataEntity parent = null;

		try {
			parent = entity.getParentDirectory();
			if (parent == null) {
				/* root directory */
				throw new EdalException("no PublicReference for entity '" + entity.getName() + "' publicated");
			}
		} catch (PrimaryDataDirectoryException e1) {
			throw new EdalException("no PublicReference for entity '" + entity.getName() + "' publicated");
		}
		while (!ready) {

			SortedSet<PrimaryDataEntityVersion> set = parent.getVersions();

			boolean found = false;

			for (PrimaryDataEntityVersion primaryDataEntityVersion : set) {
				try {
					if (primaryDataEntityVersion.getPublicReference(persistentIdentifier).isPublic()) {
						if (primaryDataEntityVersion.getRevisionDate().after(version.getRevisionDate())) {

							/**
							 * check if the public reference of the parent
							 * directory is not locked
							 **/
							if (primaryDataEntityVersion.getPublicReference(persistentIdentifier)
									.getReleaseDate() != null
									&& primaryDataEntityVersion.getPublicReference(persistentIdentifier)
											.getReleaseDate().after(Calendar.getInstance())) {
								throw new EdalException("the PublicReference for this version of '" + parent.getName()
										+ "' is locked until " + primaryDataEntityVersion
												.getPublicReference(persistentIdentifier).getReleaseDate().getTime());
							}

							found = true;
						}
					}
				} catch (PrimaryDataEntityVersionException e) {
					DataManager.getImplProv().getLogger()
							.debug("no public reference found for '" + parent + "', trying next");
				}
			}

			if (!found) {
				try {
					parent = parent.getParentDirectory();
				} catch (PrimaryDataDirectoryException e) {
					throw new EdalException("no PublicReference for entity '" + entity.getName() + "' publicated");
				}
				if (parent == null) {
					throw new EdalException("no PublicReference for entity '" + entity.getName() + "' publicated");
				}
			} else {
				ready = true;
			}
		}
		return entity;
	}

	@Override
	public PrimaryDataEntity getPrimaryDataEntityRekursiveForPersistenIdentifier(PrimaryDataEntity entity,
			long versionNumber, PersistentIdentifier persistentIdentifier) throws EdalException {

		PrimaryDataEntityVersion version = null;
		try {
			version = entity.getVersionByRevisionNumber(versionNumber);
		} catch (PrimaryDataEntityVersionException e) {
			throw new EdalException(e.getMessage());
		}
		boolean ready = false;

		PrimaryDataEntity parent = null;

		try {
			parent = entity.getParentDirectory();
			if (parent == null) {
				/* root directory */
				throw new EdalException("no PublicReference for entity '" + entity.getName() + "' publicated");
			}
		} catch (PrimaryDataDirectoryException e1) {
			throw new EdalException("no PublicReference for entity '" + entity.getName() + "' publicated");
		}
		while (!ready) {

			SortedSet<PrimaryDataEntityVersion> set = parent.getVersions();

			boolean found = false;

			for (PrimaryDataEntityVersion primaryDataEntityVersion : set) {
				try {
					if (primaryDataEntityVersion.getPublicReference(persistentIdentifier).isPublic()) {
						if (primaryDataEntityVersion.getRevisionDate().after(version.getRevisionDate())) {

							/**
							 * check if the public reference of the parent
							 * directory is not locked
							 **/
							if (primaryDataEntityVersion.getPublicReference(persistentIdentifier)
									.getReleaseDate() != null
									&& primaryDataEntityVersion.getPublicReference(persistentIdentifier)
											.getReleaseDate().after(Calendar.getInstance())) {
								throw new EdalException("the PublicReference for this version of '" + parent.getName()
										+ "' is locked until " + primaryDataEntityVersion
												.getPublicReference(persistentIdentifier).getReleaseDate().getTime());
							}

							found = true;
						}
					}
				} catch (PrimaryDataEntityVersionException e) {
					DataManager.getImplProv().getLogger()
							.debug("no public reference found for '" + parent + "', trying next");
				}
			}

			if (!found) {
				try {
					parent = parent.getParentDirectory();
				} catch (PrimaryDataDirectoryException e) {
					throw new EdalException("no PublicReference for entity '" + entity.getName() + "' publicated");
				}
				if (parent == null) {
					throw new EdalException("no PublicReference for entity '" + entity.getName() + "' publicated");
				}
			} else {
				ready = true;
				return parent;
			}
		}
		return entity;
	}

}
