/*
 * Licensed to the Apache Software Foundation (ASF) under one or more contributor license
 * agreements. See the NOTICE file distributed with this work for additional information regarding
 * copyright ownership. The ASF licenses this file to You under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance with the License. You may obtain a
 * copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software distributed under the License
 * is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express
 * or implied. See the License for the specific language governing permissions and limitations under
 * the License.
 */

package de.knightsoftnet.validators.shared.impl;

import de.knightsoftnet.validators.shared.TaxNumber;
import de.knightsoftnet.validators.shared.data.CountryEnum;
import de.knightsoftnet.validators.shared.data.CreateTaxNumberMapConstantsClass;
import de.knightsoftnet.validators.shared.data.TaxNumberMapSharedConstants;
import de.knightsoftnet.validators.shared.util.BeanPropertyReaderUtil;
import de.knightsoftnet.validators.shared.util.TaxNumberUtil;

import org.apache.commons.lang3.StringUtils;

import jakarta.validation.ConstraintValidatorContext;

/**
 * Check if a Tax Number field is valid for the selected country.
 *
 * @author Manfred Tremmel
 *
 */
public class TaxNumberValidator extends AbstractTaxTinNumberValidator<TaxNumber> {

  /**
   * map of vat id regex values for the different countries.
   */
  private static final TaxNumberMapSharedConstants TAX_NUMBER_MAP =
      CreateTaxNumberMapConstantsClass.create();

  /**
   * field name of the postal code field.
   */
  private String fieldPostalCode;

  /**
   * convert old tax numbers before validating (true/false).
   */
  private boolean convertOldNumbers;

  /**
   * field name of the vat id field.
   */
  private String fieldTaxNumber;

  /**
   * {@inheritDoc} initialize the validator.
   *
   * @see jakarta.validation.ConstraintValidator#initialize(java.lang.annotation.Annotation)
   */
  @Override
  public final void initialize(final TaxNumber pconstraintAnnotation) {
    message = pconstraintAnnotation.message();
    fieldCountryCode = pconstraintAnnotation.fieldCountryCode();
    fieldPostalCode = pconstraintAnnotation.fieldPostalCode();
    allowLowerCaseCountryCode = pconstraintAnnotation.allowLowerCaseCountryCode();
    convertOldNumbers = pconstraintAnnotation.convertOldNumbers();
    fieldTaxNumber = pconstraintAnnotation.fieldTaxNumber();
  }

  @Override
  public final boolean isValid(final Object pvalue, final ConstraintValidatorContext pcontext) {
    if (pvalue == null) {
      return true;
    }
    try {
      String countryCode =
          BeanPropertyReaderUtil.getNullSaveStringProperty(pvalue, fieldCountryCode);
      if (StringUtils.isEmpty(countryCode)) {
        return true;
      }

      if (allowLowerCaseCountryCode) {
        countryCode = StringUtils.upperCase(countryCode);
      }

      final String regExCheck = TAX_NUMBER_MAP.taxNumbers().get(countryCode);
      if (regExCheck == null) {
        return true;
      }

      final CountryEnum country = CountryEnum.valueOf(countryCode);

      final String postalCode =
          BeanPropertyReaderUtil.getNullSaveStringProperty(pvalue, fieldPostalCode);

      final String taxNumber = convertOldNumbers
          ? TaxNumberUtil.regionTaxToNational(
              BeanPropertyReaderUtil.getNullSaveStringProperty(pvalue, fieldTaxNumber), country,
              postalCode)
          : BeanPropertyReaderUtil.getNullSaveStringProperty(pvalue, fieldTaxNumber);
      if (StringUtils.isEmpty(taxNumber)) {
        return true;
      }

      if (taxNumber.matches(regExCheck) && checkSumTest(country, taxNumber)) {
        return true;
      }
      switchContext(pcontext);
      return false;
    } catch (final Exception ignore) {
      switchContext(pcontext);
      return false;
    }
  }

  private void switchContext(final ConstraintValidatorContext pcontext) {
    pcontext.disableDefaultConstraintViolation();
    pcontext.buildConstraintViolationWithTemplate(message).addPropertyNode(fieldTaxNumber)
        .addConstraintViolation();
  }

  private boolean checkSumTest(final CountryEnum pcountryCode, final String ptaxNumber) {
    boolean checkSumOk = false;
    switch (pcountryCode) {
      case AT:
        checkSumOk = checkAtNumber(ptaxNumber);
        break;
      case DE:
        checkSumOk = checkDeTaxNumber(ptaxNumber);
        break;
      case HR:
        checkSumOk = checkModulo11(ptaxNumber);
        break;
      case DK:
        checkSumOk = checkDk(ptaxNumber);
        break;
      case EE:
      case LT:
        checkSumOk = checkEe(ptaxNumber);
        break;
      case ES:
        checkSumOk = checkEs(ptaxNumber);
        break;
      case BA:
      case ME:
      case MK:
        checkSumOk = checkUniqueMasterCitizenNumber(ptaxNumber);
        break;
      case NL:
        checkSumOk = checkNl(ptaxNumber);
        break;
      case PL:
        checkSumOk = checkPl(ptaxNumber);
        break;
      default:
        // for other countries, I haven't found checksum rules
        checkSumOk = true;
        break;
    }
    return checkSumOk;
  }

  /**
   * check the Tax Identification Number number, country version for Germany.
   *
   * @param ptaxNumber vat id to check
   * @return true if checksum is ok
   */
  private boolean checkDeTaxNumber(final String ptaxNumber) {
    final int fa = Integer.parseInt(StringUtils.substring(ptaxNumber, 2, 4));
    final int sb = Integer.parseInt(StringUtils.substring(ptaxNumber, 5, 8));
    final int checkSum = ptaxNumber.charAt(12) - '0';
    if (StringUtils.startsWith(ptaxNumber, "11")) {
      // Berlin
      final int calculatedCheckSum;
      if (fa >= 27 && fa <= 30 //
          || fa < 31 && (sb < 201 || sb > 693) //
          || fa == 19 && (sb < 200 || sb > 639 && sb < 680 || sb > 680 && sb < 684 || sb > 684) //
          || fa == 37) {
        calculatedCheckSum = ((ptaxNumber.charAt(5) - '0') * 7 //
            + (ptaxNumber.charAt(6) - '0') * 6 //
            + (ptaxNumber.charAt(7) - '0') * 5 //
            + (ptaxNumber.charAt(8) - '0') * 8 //
            + (ptaxNumber.charAt(9) - '0') * 4 //
            + (ptaxNumber.charAt(10) - '0') * 3 //
            + (ptaxNumber.charAt(11) - '0') * 2) //
            % MODULO_11;
      } else {
        calculatedCheckSum = ((ptaxNumber.charAt(2) - '0') * 3 //
            + (ptaxNumber.charAt(3) - '0') * 2 //
            + (ptaxNumber.charAt(4) - '0') * 9 //
            + (ptaxNumber.charAt(5) - '0') * 8 //
            + (ptaxNumber.charAt(6) - '0') * 7 //
            + (ptaxNumber.charAt(7) - '0') * 6 //
            + (ptaxNumber.charAt(8) - '0') * 5 //
            + (ptaxNumber.charAt(9) - '0') * 4 //
            + (ptaxNumber.charAt(10) - '0') * 3 //
            + (ptaxNumber.charAt(11) - '0') * 2) //
            % MODULO_11;
      }
      return checkSum == calculatedCheckSum;
    }
    // TODO find checksum calculation routines for the rest
    return true;
  }
}
