/*
 * Licensed to the Apache Software Foundation (ASF) under one or more contributor license
 * agreements. See the NOTICE file distributed with this work for additional information regarding
 * copyright ownership. The ASF licenses this file to You under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance with the License. You may obtain a
 * copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software distributed under the License
 * is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express
 * or implied. See the License for the specific language governing permissions and limitations under
 * the License.
 */

package de.knightsoftnet.validators.shared;

import de.knightsoftnet.validators.shared.impl.PasswordValidator;

import java.lang.annotation.Documented;
import java.lang.annotation.ElementType;
import java.lang.annotation.Retention;
import java.lang.annotation.RetentionPolicy;
import java.lang.annotation.Target;

import jakarta.validation.Constraint;
import jakarta.validation.Payload;

/**
 * The annotated element must be a valid password.<br>
 * Supported types are Strings, other Objects are transfered to Strings, <code>null</code> elements
 * are considered valid.<br>
 * There are the following rules checked (with <code>minRules</code> parameter the number rules can
 * be defined that have to be fulfilled):
 * <ul>
 * <li>lowercase</li>
 * <li>upercase</li>
 * <li>digits</li>
 * <li>special character</li>
 * </ul>
 * Using <code>blacklist</code> you can give a comma separated list of words which are not allowed
 * to be part of the password. Default is no entry.<br>
 * Using <code>disalowedStartChars</code> you can define characters which are not allowed as first
 * character in the password. Default is no entry.<br>
 * With <code>maxRepeatChar</code> you can limit the repeat of a single character, default is 0
 * which means no limitation.<br>
 * size limits should be checked by separate size annotation.
 *
 * @author Manfred Tremmel
 *
 */
@Documented
@Constraint(validatedBy = PasswordValidator.class)
@Target({ElementType.METHOD, ElementType.FIELD, ElementType.ANNOTATION_TYPE,
    ElementType.CONSTRUCTOR, ElementType.PARAMETER})
@Retention(RetentionPolicy.RUNTIME)
public @interface Password {
  /**
   * localized message.
   *
   * @return localized validation message
   */
  String message() default "{de.knightsoftnet.validators.shared.Password.message}";

  /**
   * localized message if blacklisted.
   *
   * @return error message for black listed passwords
   */
  String messageBlacklist() //
  default "{de.knightsoftnet.validators.shared.PasswordBlacklist.message}";

  /**
   * localized message if start character is not allowed.
   *
   * @return error message for not allowed start characters
   */
  String messageStartCharacters() //
  default "{de.knightsoftnet.validators.shared.PasswordStartChar.message}";

  /**
   * localized message if maximum repeat of a char is reached in a row.
   *
   * @return error message for max repeat
   */
  String messageMaxRepeat() //
  default "{de.knightsoftnet.validators.shared.PasswordMaxRepeat.message}";

  /**
   * groups to use.
   *
   * @return array of validation groups
   */
  Class<?>[] groups() default {};

  /**
   * minimum number rules that have to be fulfilled.
   *
   * @return number minimum required rules
   */
  int minRules();

  /**
   * Array of words which are not allowed as part of the password.
   *
   * @return comma separated list of not allowed password
   */
  String[] blacklist() default {};

  /**
   * Characters which are not allowed at the beginning of a password.
   *
   * @return characters which are not allowed
   */
  String[] disalowedStartChars() default {};

  /**
   * maximum repeats of a single character.
   *
   * @return maximum allowed repeated characters
   */
  int maxRepeatChar() default 0;

  /**
   * payload whatever.
   *
   * @return payload class
   */
  Class<? extends Payload>[] payload() default {};

  /**
   * Defines several {@code Password} annotations on the same element.
   */
  @Target({ElementType.METHOD, ElementType.FIELD, ElementType.ANNOTATION_TYPE,
      ElementType.CONSTRUCTOR, ElementType.PARAMETER})
  @Retention(RetentionPolicy.RUNTIME)
  @Documented
  @interface List {
    /**
     * password value.
     *
     * @return value
     */
    Password[] value();
  }
}
