/*
 * Decompiled with CFR 0.152.
 */
package de.mirkosertic.flightrecorderstarter.core;

import de.mirkosertic.flightrecorderstarter.actuator.model.FlightRecorderPublicSession;
import de.mirkosertic.flightrecorderstarter.configuration.FlightRecorderDynamicConfiguration;
import de.mirkosertic.flightrecorderstarter.core.RecordingSession;
import de.mirkosertic.flightrecorderstarter.core.StartRecordingCommand;
import java.io.File;
import java.io.IOException;
import java.nio.file.Path;
import java.time.Duration;
import java.time.Instant;
import java.time.LocalDateTime;
import java.time.ZoneId;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.logging.Level;
import java.util.logging.Logger;
import java.util.stream.Collectors;
import jdk.jfr.Configuration;
import jdk.jfr.Recording;
import jdk.jfr.RecordingState;
import org.springframework.scheduling.annotation.Scheduled;

public class FlightRecorder {
    private static final Logger LOGGER = Logger.getLogger(FlightRecorder.class.getCanonicalName());
    private final Map<Long, RecordingSession> recordings;
    private final FlightRecorderDynamicConfiguration configuration;

    public FlightRecorder(FlightRecorderDynamicConfiguration configuration) {
        this(configuration, new HashMap<Long, RecordingSession>());
    }

    FlightRecorder(FlightRecorderDynamicConfiguration configuration, Map<Long, RecordingSession> recordings) {
        this.configuration = configuration;
        this.recordings = recordings;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public long newRecording(StartRecordingCommand command) {
        Recording recording = new Recording(this.getConfigurationSettings(Configuration.getConfigurations(), command.getCustomSettings()));
        recording.setName("Spring Boot Starter Flight Recording");
        Map<Long, RecordingSession> map = this.recordings;
        synchronized (map) {
            this.recordings.put(recording.getId(), new RecordingSession(recording, command.getDescription()));
        }
        return recording.getId();
    }

    Map<String, String> getConfigurationSettings(List<Configuration> configs, Map<String, String> customSettings) {
        HashMap<String, String> settings = new HashMap<String, String>();
        String chosenConfiguration = this.configuration.getJfrCustomConfig() != null ? this.configuration.getJfrCustomConfig() : "profile";
        for (Configuration config : configs) {
            LOGGER.log(Level.INFO, "Found configuration {0}", config.getName());
            if (!config.getName().contains(chosenConfiguration)) continue;
            LOGGER.log(Level.INFO, "Using configuration {0}", config.getName());
            settings.putAll(config.getSettings());
        }
        if (customSettings != null) {
            settings.putAll(customSettings);
        }
        return settings;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void startRecording(long recordingId, Duration delayDuration) {
        Map<Long, RecordingSession> map = this.recordings;
        synchronized (map) {
            RecordingSession recordingSession = this.recordings.get(recordingId);
            if (recordingSession != null) {
                if (delayDuration == null) {
                    recordingSession.getRecording().start();
                } else {
                    recordingSession.getRecording().scheduleStart(delayDuration);
                }
            } else {
                LOGGER.log(Level.WARNING, "No recording with id {0} found", recordingId);
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public File stopRecording(long recordingId) {
        Map<Long, RecordingSession> map = this.recordings;
        synchronized (map) {
            RecordingSession recordingSession = this.recordings.get(recordingId);
            if (recordingSession != null) {
                Recording recording = recordingSession.getRecording();
                if (recording.getState() == RecordingState.RUNNING) {
                    recording.stop();
                }
                return recording.getDestination().toFile();
            }
            LOGGER.log(Level.WARNING, "No recording with id {0} found", recordingId);
            return null;
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void setRecordingOptions(long recordingId, StartRecordingCommand command, File filename) throws IOException {
        Map<Long, RecordingSession> map = this.recordings;
        synchronized (map) {
            RecordingSession recordingSession = this.recordings.get(recordingId);
            if (recordingSession != null) {
                Recording recording = recordingSession.getRecording();
                recording.setDuration(Duration.of(command.getDuration(), command.getTimeUnit()));
                recording.setDestination(filename.toPath());
                recording.setToDisk(true);
                if (command.getMaxAgeDuration() != null && command.getMaxAgeUnit() != null) {
                    recording.setMaxAge(Duration.of(command.getMaxAgeDuration(), command.getMaxAgeUnit()));
                }
                if (command.getMaxSize() != null) {
                    recording.setMaxSize(command.getMaxSize());
                }
            } else {
                LOGGER.log(Level.WARNING, "No recording with id {0} found", recordingId);
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public long startRecordingFor(StartRecordingCommand command) throws IOException {
        Map<Long, RecordingSession> map = this.recordings;
        synchronized (map) {
            long recordingId = this.newRecording(command);
            File basePath = null;
            if (this.configuration.getJfrBasePath() != null) {
                basePath = Path.of(this.configuration.getJfrBasePath(), new String[0]).toFile();
            }
            File tempFile = File.createTempFile("recording", ".jfr", basePath);
            LOGGER.log(Level.INFO, "Recording {0} to temp file {1}", new Object[]{recordingId, tempFile});
            tempFile.deleteOnExit();
            this.setRecordingOptions(recordingId, command, tempFile);
            Duration delay = null;
            if (command.getDelayDuration() != null && command.getDelayUnit() != null) {
                delay = Duration.of(command.getDelayDuration(), command.getDelayUnit());
            }
            this.startRecording(recordingId, delay);
            return recordingId;
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Scheduled(fixedDelayString="${flightrecorder.recording-cleanup-interval}")
    public void cleanupOldRecordings() {
        Map<Long, RecordingSession> map = this.recordings;
        synchronized (map) {
            Set<Long> deletableRecordings;
            if (this.configuration.getRecordingCleanupType() == FlightRecorderDynamicConfiguration.CleanupType.TTL) {
                deletableRecordings = this.getDeletableRecordingsByTTL();
            } else if (this.configuration.getRecordingCleanupType() == FlightRecorderDynamicConfiguration.CleanupType.COUNT) {
                deletableRecordings = this.getDeletableRecordingsByCount();
            } else {
                throw new IllegalArgumentException("Unknown CleanupType '%s'. Deletion failed.".formatted(new Object[]{this.configuration.getRecordingCleanupType()}));
            }
            deletableRecordings.forEach(this::deleteRecording);
        }
    }

    protected Set<Long> getDeletableRecordingsByTTL() {
        HashSet<Long> deletableRecordings = new HashSet<Long>();
        Instant deadline = Instant.now().minus(this.configuration.getOldRecordingsTTL(), this.configuration.getOldRecordingsTTLTimeUnit());
        for (Map.Entry<Long, RecordingSession> entry : this.recordings.entrySet()) {
            Recording recording = entry.getValue().getRecording();
            if (recording.getState() != RecordingState.STOPPED && recording.getState() != RecordingState.CLOSED || !recording.getStartTime().isBefore(deadline)) continue;
            try {
                if (recording.getState() == RecordingState.STOPPED) {
                    recording.close();
                }
            }
            catch (Exception e) {
                LOGGER.log(Level.INFO, "Cannot close recording {0}", new Object[]{recording.getId()});
            }
            deletableRecordings.add(entry.getKey());
        }
        LOGGER.log(Level.FINE, "Found {0} finished recording(s) to be deleted based on TTL ({1} {2}).", new Object[]{deletableRecordings.size(), this.configuration.getOldRecordingsTTL(), this.configuration.getOldRecordingsTTLTimeUnit()});
        return deletableRecordings;
    }

    protected Set<Long> getDeletableRecordingsByCount() {
        int maxRecordings = this.configuration.getOldRecordingsMax();
        if (this.recordings.size() <= maxRecordings) {
            return Collections.emptySet();
        }
        List recordingsAboveThreshold = this.recordings.entrySet().stream().sorted(Comparator.comparing(recs -> ((RecordingSession)recs.getValue()).getRecording().getStartTime())).collect(Collectors.toList()).subList(0, this.recordings.size() - maxRecordings);
        HashSet<Long> deletableRecordings = new HashSet<Long>();
        for (Map.Entry entry : recordingsAboveThreshold) {
            Recording recording = ((RecordingSession)entry.getValue()).getRecording();
            if (recording.getState() != RecordingState.STOPPED && recording.getState() != RecordingState.CLOSED) continue;
            try {
                if (recording.getState() == RecordingState.STOPPED) {
                    recording.close();
                }
            }
            catch (Exception e) {
                LOGGER.log(Level.INFO, "Cannot close recording {0}", new Object[]{recording.getId()});
            }
            deletableRecordings.add((Long)entry.getKey());
        }
        LOGGER.log(Level.FINE, "Found {0} finished recording(s) to be deleted based on COUNT threshold ({1} recordings).", new Object[]{deletableRecordings.size(), maxRecordings});
        return deletableRecordings;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void deleteRecording(long recordingId) {
        Map<Long, RecordingSession> map = this.recordings;
        synchronized (map) {
            RecordingSession recordingSession = this.recordings.remove(recordingId);
            if (recordingSession != null) {
                Recording recording = recordingSession.getRecording();
                if (recording.getState() == RecordingState.RUNNING) {
                    recording.stop();
                    recording.close();
                } else if (recording.getState() == RecordingState.STOPPED) {
                    recording.close();
                }
                recordingSession.getRecording().getDestination().toFile().delete();
            } else {
                LOGGER.log(Level.WARNING, "No recording with id {0} found", recordingId);
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public boolean isRecordingStopped(long recordingId) {
        Map<Long, RecordingSession> map = this.recordings;
        synchronized (map) {
            RecordingSession recordingSession = this.recordings.get(recordingId);
            if (recordingSession == null) {
                return true;
            }
            // MONITOREXIT @DISABLED, blocks:[0, 1] lbl8 : MonitorExitStatement: MONITOREXIT : var3_2
            Recording recording = recordingSession.getRecording();
            return recording.getState() == RecordingState.CLOSED || recording.getState() == RecordingState.STOPPED;
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public List<FlightRecorderPublicSession> sessions() {
        ArrayList<FlightRecorderPublicSession> result = new ArrayList<FlightRecorderPublicSession>();
        Map<Long, RecordingSession> map = this.recordings;
        synchronized (map) {
            for (RecordingSession session : this.recordings.values()) {
                FlightRecorderPublicSession publicSession = this.getFlightRecorderPublicSession(session);
                result.add(publicSession);
            }
        }
        result.sort(Comparator.comparingLong(FlightRecorderPublicSession::getId));
        return result;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public FlightRecorderPublicSession getById(Long recordingId) {
        Map<Long, RecordingSession> map = this.recordings;
        synchronized (map) {
            RecordingSession session = this.recordings.get(recordingId);
            if (session != null) {
                return this.getFlightRecorderPublicSession(session);
            }
            return null;
        }
    }

    FlightRecorderPublicSession getFlightRecorderPublicSession(RecordingSession session) {
        FlightRecorderPublicSession publicSession = new FlightRecorderPublicSession();
        publicSession.setId(session.getRecording().getId());
        publicSession.setStatus(session.getRecording().getState().name());
        publicSession.setStartedAt(LocalDateTime.ofInstant(session.getRecording().getStartTime(), ZoneId.systemDefault()));
        if (session.getRecording().getState() == RecordingState.CLOSED || session.getRecording().getState() == RecordingState.STOPPED) {
            publicSession.setFinishedAt(LocalDateTime.ofInstant(session.getRecording().getStopTime(), ZoneId.systemDefault()));
        }
        publicSession.setDescription(session.getDescription());
        return publicSession;
    }
}

