package de.pco.sdk;

/*-
 * #%L
 * pco-common
 * %%
 * Copyright (C) 2020 PCO
 * %%
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 * #L%
 */

import de.pco.sdk.enums.B16Version;
import de.pco.sdk.enums.CameraSubtype;
import de.pco.sdk.enums.CameraSyncMode;
import de.pco.sdk.enums.CameraType;
import de.pco.sdk.enums.ColorPatternType;
import de.pco.sdk.enums.ImageTimeStatus;
import de.pco.sdk.enums.ImageType;
import de.pco.sdk.enums.SyncStatus;
import de.pco.sdk.enums.Timebase;
import de.pco.sdk.enums.TimestampPosition;
import de.pco.sdk.enums.TriggerMode;

/**
 * Container of the metadata, see pco_filestruct.h.
 * @author PCO
 *
 */
public class BildStruct 
{
    public Systemtime time;               // shows the exact time stamp of the image
    public int ticks;                    // milliseconds gone after start in stime
    public int xRes;                     // X Resolution
    public int yRes;                     // Y Resolution
    public String text;                  // text which should be placed inside the image
    public boolean isDouble;             // shows if pic is Doubleshutter image
    public boolean thisIsT0;             // True: This image is T0 image; False: -
    public boolean weHaveT0;             // True: Image stream has got T0
    public int bwMin;                    // Lut bw min
    public int bwMax;                    // Lut bw max
    public int bwLut;                    // Lut lin log
    public int rMin;                     // red min
    public int rMax;                     // red max
    public int gMin;                     // green min
    public int gMax;                     // green max
    public int bMin;                     // blue min
    public int bMax;                     // blue max
    public int colLut;                   // Lut lin log color
    public boolean isColor;              // image from Color-CCD: 1 otherwise 0 
    public B16Version version;           // Version of b16 extended info
    // V200: END
    public int bwMin2;                   // Lut bw min
    public int bwMax2;                   // Lut bw max
    public int bwLut2;                   // Lut lin log
    public int rMin2;                    // red min
    public int rMax2;                    // red max
    public int gMin2;                    // green min
    public int gMax2;                    // green max
    public int bMin2;                    // blue min
    public int bMax2;                    // blue max
    public int colLut2;                  // Lut lin log color
    public boolean alignUpper;            // Align MSB (0-MSB is bit14, 1-MSB is bit 16)
    // V300: END
    public double gammaLut;               // Gamma value b/w
    public double gammaLutC;              // Gamma value color
    public double gammaLut2;              // Gamma value b/w 2
    public double gammaLutC2;             // Gamma value color 2
    public ColorPatternType colorPatternType;// Demosaicking type for the color pattern
    public int bitRes;                    // Bit resolution of image
    // V301: END
    public double dSaturation;            // Color saturation common for both ds images
    // V302/V303: END
    public int iSaturation;               // new color saturation
    public int vibrance;
    public int colorTemp;
    public int colorTint;
    public int contrast;
    public int gamma;                     // new Gamma
    // V304: END
    public SRGBColorCorrectionCoefficients colorCoeff;       // 9*double
    // V305: END
    public TimestampPosition timestampPosition;

    public boolean hasMetaData;               // 0->no meta data available, 1->has meta data
    public int sensorConvFactor;          // sensor conversions factor in e-/ct, 0 if unknown
    public CameraType cameraType;         // type of pco camera taking the images, see SDK, 0 if unknown
    public int darkOffset;                // nominal dark offset in counts, 0xFFFF if unknown, current dark offset may differ

    public long imageCounter;             // 00000001 to 99999999, (internally BCD where first byte is least sign. byte)
    public int imageTimeUs;               // 000000 to 999999, (internally BCD where first byte is least significant byte)
    public ImageTimeStatus imageTimeStatus;// 0x00 = internal osc, 0x01 = synced by IRIG, 0x02 = synced by master

    public Timebase exposureTimebase;     // timebase ns/us/ms for following exposure time
    public short sensorTemperature;       // current sensor temperature in °C, 0x8000 if not known
    public long exposureTime;             // exposure time in ns/us/ms  according to timebase

    public long framerateMilliHz;          // framerate in mHz, 0 if unknown or not 
    public byte binningX;                 // binning in x direction, 0x00 if unknown or n/a
    public byte binningY;                 // binning in y direction, 0x00 if unknown or n/a
    public TriggerMode triggerMode;        // exposure trigger mode, see PCO SDK
    public CameraSyncMode cameraSyncMode;  // see PCO SDK

    public long sensorReadoutFrequency;   // sensor readout frequency in Hz, 0 if unknown
    public long cameraSerialNo;           // camera serial no, 0 if unknown
    // V306: END
    public SyncStatus syncStatus;         // status of PLL for external synchronisation (100 Hz or 1 kHz): 0 = off, 1 = locked
    public ImageType imageType;           // 0x01 ist b/w, 0x02 is color bayer pattern, 0x10 is RGB 
    public int colorPattern;              // bayer pattern color mask, same as for SDK command "Get Camera Description", 0 if n/a

    public CameraSubtype cameraSubtype;   // sub-type of pco camera taking the images, see SDK, 0 if unknown
    public long eventNumber;              // application dependent event number
    public int imageSizeXoffset;          // actual x offset in case of ROI (X0) in x direction (horizontal)
    public int imageSizeYoffset;          // actual y offset in case of ROI (Y0) in y direction (vertical)

    public long compression;              // File format compression 0: none
    public long compressionVersion;       // 0
    public long predictorStart;           // Compression predictor: 0
    public long randomStart;              // Random lut start value
    public long randomIndex;              // Random lut index for this image
    public int diffToT0;                  // Difference to T0 image if applicable
    
    /**
     * Standard constructor
     */
    public BildStruct() 
    {
        this.time = new Systemtime();
        this.text = "";
        this.version = B16Version.CURRENT;
        this.colorCoeff = new SRGBColorCorrectionCoefficients();
        this.colorPatternType = ColorPatternType.values()[0];
        this.cameraType = CameraType.UNDEFINED;
        this.imageTimeStatus = ImageTimeStatus.INTERNAL_OSC;
        this.triggerMode = TriggerMode.AUTOTRIGGER;
        this.cameraSyncMode = CameraSyncMode.STANDALONE;
        this.syncStatus = SyncStatus.OFF;
        this.gammaLut = 40.0;
        this.gammaLut2 = 40.0;
        this.gammaLutC = 40.0;
        this.gammaLutC2 = 40.0;
        this.binningX = 1;
        this.binningY = 1;
        this.bitRes = 16;
        this.colorTemp = 3500;
        this.gamma = 40;
        this.imageType = ImageType.UNDEFINED;
        this.cameraSubtype = CameraSubtype.UNDEFINED;
        this.darkOffset = 0xFFFF;
        this.sensorTemperature = Short.MIN_VALUE;
        this.timestampPosition = TimestampPosition.NONE;
        this.exposureTimebase = Timebase.NS;
        this.diffToT0 = 0;
    }
        
    
    ///////////////////////////////////// GENERATED
    
    @Override
    public int hashCode() {
        final int prime = 31;
        int result = 1;
        result = prime * result + (alignUpper ? 1231 : 1237);
        result = prime * result + bMax;
        result = prime * result + bMax2;
        result = prime * result + bMin;
        result = prime * result + bMin2;
        result = prime * result + binningX;
        result = prime * result + binningY;
        result = prime * result + bitRes;
        result = prime * result + bwLut;
        result = prime * result + bwLut2;
        result = prime * result + bwMax;
        result = prime * result + bwMax2;
        result = prime * result + bwMin;
        result = prime * result + bwMin2;
        result = prime * result + (int) (cameraSerialNo ^ (cameraSerialNo >>> 32));
        result = prime * result + ((cameraSubtype == null) ? 0 : cameraSubtype.hashCode());
        result = prime * result + ((cameraSyncMode == null) ? 0 : cameraSyncMode.hashCode());
        result = prime * result + ((cameraType == null) ? 0 : cameraType.hashCode());
        result = prime * result + colLut;
        result = prime * result + colLut2;
        result = prime * result + ((colorCoeff == null) ? 0 : colorCoeff.hashCode());
        result = prime * result + colorPattern;
        result = prime * result + ((colorPatternType == null) ? 0 : colorPatternType.hashCode());
        result = prime * result + colorTemp;
        result = prime * result + colorTint;
        result = prime * result + (int) (compression ^ (compression >>> 32));
        result = prime * result + (int) (compressionVersion ^ (compressionVersion >>> 32));
        result = prime * result + contrast;
        long temp;
        temp = Double.doubleToLongBits(dSaturation);
        result = prime * result + (int) (temp ^ (temp >>> 32));
        result = prime * result + darkOffset;
        result = prime * result + diffToT0;
        result = prime * result + (int) (eventNumber ^ (eventNumber >>> 32));
        result = prime * result + (int) (exposureTime ^ (exposureTime >>> 32));
        result = prime * result + exposureTimebase.ordinal();
        result = prime * result + (int) (framerateMilliHz ^ (framerateMilliHz >>> 32));
        result = prime * result + gMax;
        result = prime * result + gMax2;
        result = prime * result + gMin;
        result = prime * result + gMin2;
        result = prime * result + gamma;
        temp = Double.doubleToLongBits(gammaLut);
        result = prime * result + (int) (temp ^ (temp >>> 32));
        temp = Double.doubleToLongBits(gammaLut2);
        result = prime * result + (int) (temp ^ (temp >>> 32));
        temp = Double.doubleToLongBits(gammaLutC);
        result = prime * result + (int) (temp ^ (temp >>> 32));
        temp = Double.doubleToLongBits(gammaLutC2);
        result = prime * result + (int) (temp ^ (temp >>> 32));
        result = prime * result + (hasMetaData ? 1231 : 1237);
        result = prime * result + ((timestampPosition == null) ? 0 : timestampPosition.hashCode());
        result = prime * result + iSaturation;
        result = prime * result + (int) (imageCounter ^ (imageCounter >>> 32));
        result = prime * result + imageSizeXoffset;
        result = prime * result + imageSizeYoffset;
        result = prime * result + ((imageTimeStatus == null) ? 0 : imageTimeStatus.hashCode());
        result = prime * result + imageTimeUs;
        result = prime * result + ((imageType == null) ? 0 : imageType.hashCode());
        result = prime * result + (isColor ? 1231 : 1237);
        result = prime * result + (isDouble ? 1231 : 1237);
        result = prime * result + (int) (predictorStart ^ (predictorStart >>> 32));
        result = prime * result + rMax;
        result = prime * result + rMax2;
        result = prime * result + rMin;
        result = prime * result + rMin2;
        result = prime * result + (int) (randomIndex ^ (randomIndex >>> 32));
        result = prime * result + (int) (randomStart ^ (randomStart >>> 32));
        result = prime * result + sensorConvFactor;
        result = prime * result + (int) (sensorReadoutFrequency ^ (sensorReadoutFrequency >>> 32));
        result = prime * result + sensorTemperature;
        result = prime * result + ((syncStatus == null) ? 0 : syncStatus.hashCode());
        result = prime * result + ((text == null) ? 0 : text.hashCode());
        result = prime * result + (thisIsT0 ? 1231 : 1237);
        result = prime * result + ticks;
        result = prime * result + ((time == null) ? 0 : time.hashCode());
        result = prime * result + ((triggerMode == null) ? 0 : triggerMode.hashCode());
        result = prime * result + ((version == null) ? 0 : version.hashCode());
        result = prime * result + vibrance;
        result = prime * result + (weHaveT0 ? 1231 : 1237);
        result = prime * result + xRes;
        result = prime * result + yRes;
        return result;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj)
            return true;
        if (obj == null)
            return false;
        if (getClass() != obj.getClass())
            return false;
        BildStruct other = (BildStruct) obj;
        if (alignUpper != other.alignUpper)
            return false;
        if (bMax != other.bMax)
            return false;
        if (bMax2 != other.bMax2)
            return false;
        if (bMin != other.bMin)
            return false;
        if (bMin2 != other.bMin2)
            return false;
        if (binningX != other.binningX)
            return false;
        if (binningY != other.binningY)
            return false;
        if (bitRes != other.bitRes)
            return false;
        if (bwLut != other.bwLut)
            return false;
        if (bwLut2 != other.bwLut2)
            return false;
        if (bwMax != other.bwMax)
            return false;
        if (bwMax2 != other.bwMax2)
            return false;
        if (bwMin != other.bwMin)
            return false;
        if (bwMin2 != other.bwMin2)
            return false;
        if (cameraSerialNo != other.cameraSerialNo)
            return false;
        if (cameraSubtype != other.cameraSubtype)
            return false;
        if (cameraSyncMode != other.cameraSyncMode)
            return false;
        if (cameraType != other.cameraType)
            return false;
        if (colLut != other.colLut)
            return false;
        if (colLut2 != other.colLut2)
            return false;
        if (colorCoeff == null) {
            if (other.colorCoeff != null)
                return false;
        } else if (!colorCoeff.equals(other.colorCoeff))
            return false;
        if (colorPattern != other.colorPattern)
            return false;
        if (colorPatternType != other.colorPatternType)
            return false;
        if (colorTemp != other.colorTemp)
            return false;
        if (colorTint != other.colorTint)
            return false;
        if (compression != other.compression)
            return false;
        if (compressionVersion != other.compressionVersion)
            return false;
        if (contrast != other.contrast)
            return false;
        final double EPSILON = 0.001;
        if (Math.abs(Double.doubleToLongBits(dSaturation) - Double.doubleToLongBits(other.dSaturation)) >= EPSILON)
            return false;
        if (darkOffset != other.darkOffset)
            return false;
        if (diffToT0 != other.diffToT0)
            return false;
        if (eventNumber != other.eventNumber)
            return false;
        if (exposureTime != other.exposureTime)
            return false;
        if (exposureTimebase != other.exposureTimebase)
            return false;
        if (framerateMilliHz != other.framerateMilliHz)
            return false;
        if (gMax != other.gMax)
            return false;
        if (gMax2 != other.gMax2)
            return false;
        if (gMin != other.gMin)
            return false;
        if (gMin2 != other.gMin2)
            return false;
        if (gamma != other.gamma)
            return false;
        if (Math.abs(Double.doubleToLongBits(gammaLut) - Double.doubleToLongBits(other.gammaLut)) >= EPSILON)
            return false;
        if (Math.abs(Double.doubleToLongBits(gammaLut2) - Double.doubleToLongBits(other.gammaLut2)) >= EPSILON)
            return false;
        if (Math.abs(Double.doubleToLongBits(gammaLutC) - Double.doubleToLongBits(other.gammaLutC)) >= EPSILON)
            return false;
        if (Math.abs(Double.doubleToLongBits(gammaLutC2) - Double.doubleToLongBits(other.gammaLutC2)) >= EPSILON)
            return false;
        if (hasMetaData != other.hasMetaData)
            return false;
        if (timestampPosition != other.timestampPosition)
            return false;
        if (iSaturation != other.iSaturation)
            return false;
        if (imageCounter != other.imageCounter)
            return false;
        if (imageSizeXoffset != other.imageSizeXoffset)
            return false;
        if (imageSizeYoffset != other.imageSizeYoffset)
            return false;
        if (imageTimeStatus != other.imageTimeStatus)
            return false;
        if (imageTimeUs != other.imageTimeUs)
            return false;
        if (imageType != other.imageType)
            return false;
        if (isColor != other.isColor)
            return false;
        if (isDouble != other.isDouble)
            return false;
        if (predictorStart != other.predictorStart)
            return false;
        if (rMax != other.rMax)
            return false;
        if (rMax2 != other.rMax2)
            return false;
        if (rMin != other.rMin)
            return false;
        if (rMin2 != other.rMin2)
            return false;
        if (randomIndex != other.randomIndex)
            return false;
        if (randomStart != other.randomStart)
            return false;
        if (sensorConvFactor != other.sensorConvFactor)
            return false;
        if (sensorReadoutFrequency != other.sensorReadoutFrequency)
            return false;
        if (sensorTemperature != other.sensorTemperature)
            return false;
        if (syncStatus != other.syncStatus)
            return false;
        if (text == null) {
            if (other.text != null)
                return false;
        } else if (!text.equals(other.text))
            return false;
        if (thisIsT0 != other.thisIsT0)
            return false;
        if (ticks != other.ticks)
            return false;
        if (time == null) {
            if (other.time != null)
                return false;
        } else if (!time.equals(other.time))
            return false;
        if (triggerMode != other.triggerMode)
            return false;
        if ((version == B16Version.CURRENT && 
            !(other.version == B16Version.CURRENT || other.version == B16Version.V405))) {
            return false;
        }
        if ((version == B16Version.V405 && 
                !(other.version == B16Version.CURRENT || other.version == B16Version.V405))) {
            return false;
        }
        if (version != B16Version.V405 && version != B16Version.CURRENT && version != other.version)
            return false;
        if (vibrance != other.vibrance)
            return false;
        if (weHaveT0 != other.weHaveT0)
            return false;
        if (xRes != other.xRes)
            return false;
        if (yRes != other.yRes)
            return false;
        return true;
    }

    @Override
    public String toString() {
        return "MetadataBean [time=" + time + ", ticks=" + ticks + ", xRes=" + xRes + ", yRes=" + yRes + ", text="
                + text + ", isDouble=" + isDouble + ", thisIsT0=" + thisIsT0 + ", weHaveT0=" + weHaveT0 + ", bwMin="
                + bwMin + ", bwMax=" + bwMax + ", bwLut=" + bwLut + ", rMin=" + rMin + ", rMax=" + rMax + ", gMin="
                + gMin + ", gMax=" + gMax + ", bMin=" + bMin + ", bMax=" + bMax + ", colLut=" + colLut + ", isColor="
                + isColor + ", version=" + version + ", bwMin2=" + bwMin2 + ", bwMax2=" + bwMax2 + ", bwLut2=" + bwLut2
                + ", rMin2=" + rMin2 + ", rMax2=" + rMax2 + ", gMin2=" + gMin2 + ", gMax2=" + gMax2 + ", bMin2=" + bMin2
                + ", bMax2=" + bMax2 + ", colLut2=" + colLut2 + ", alignUpper=" + alignUpper + ", gammaLut=" + gammaLut
                + ", gammaLutC=" + gammaLutC + ", gammaLut2=" + gammaLut2 + ", gammaLutC2=" + gammaLutC2
                + ", colorPatternType=" + colorPatternType + ", bitRes=" + bitRes + ", dSaturation=" + dSaturation
                + ", iSaturation=" + iSaturation + ", vibrance=" + vibrance + ", colorTemp=" + colorTemp
                + ", colorTint=" + colorTint + ", contrast=" + contrast + ", gamma=" + gamma + ", colorCoeff="
                + colorCoeff + ", timestampPosition=" + timestampPosition + ", hasMetaData=" + hasMetaData + ", sensorConvFactor="
                + sensorConvFactor + ", cameraType=" + cameraType + ", darkOffset=" + darkOffset + ", imageCounter="
                + imageCounter + ", imageTimeUs=" + imageTimeUs + ", imageTimeStatus=" + imageTimeStatus
                + ", exposureTimeBase=" + exposureTimebase + ", sensorTemperature=" + sensorTemperature
                + ", exposureTime=" + exposureTime + ", framerateMilliHz=" + framerateMilliHz + ", binningX=" + binningX
                + ", binningY=" + binningY + ", triggerMode=" + triggerMode + ", cameraSyncMode=" + cameraSyncMode
                + ", sensorReadoutFrequency=" + sensorReadoutFrequency + ", cameraSerialNo=" + cameraSerialNo
                + ", syncStatus=" + syncStatus + ", imageType=" + imageType + ", colorPattern=" + colorPattern
                + ", cameraSubtype=" + cameraSubtype + ", eventNumber=" + eventNumber + ", imageSizeXoffset="
                + imageSizeXoffset + ", imageSizeYoffset=" + imageSizeYoffset + ", compression=" + compression
                + ", compressionVersion=" + compressionVersion + ", predictorStart=" + predictorStart + ", randomStart="
                + randomStart + ", randomIndex=" + randomIndex + ", diffToT0=" + diffToT0 + "]";
    }
    
    
}
