package de.retest.recheck.ui.descriptors;

import static java.util.Objects.requireNonNull;

import java.beans.Transient;
import java.io.Serializable;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Objects;
import java.util.Set;
import java.util.stream.Collectors;

import org.apache.commons.lang3.ObjectUtils;
import org.eclipse.persistence.oxm.annotations.XmlInverseReference;

import de.retest.recheck.ui.Path;
import de.retest.recheck.ui.diff.AttributeDifference;
import de.retest.recheck.ui.image.Screenshot;
import de.retest.recheck.ui.review.ActionChangeSet;
import de.retest.recheck.util.RetestIdUtil;
import jakarta.xml.bind.annotation.XmlAccessType;
import jakarta.xml.bind.annotation.XmlAccessorType;
import jakarta.xml.bind.annotation.XmlAttribute;
import jakarta.xml.bind.annotation.XmlElement;
import jakarta.xml.bind.annotation.XmlRootElement;
import jakarta.xml.bind.annotation.XmlTransient;
import jakarta.xml.bind.annotation.adapters.XmlJavaTypeAdapter;

@XmlRootElement
@XmlAccessorType( XmlAccessType.FIELD )
public class Element implements Serializable, Comparable<Element> {

	protected static final long serialVersionUID = 2L;

	@XmlAttribute
	protected final String retestId;

	@XmlInverseReference( mappedBy = "containedElements" )
	private final Element parent;

	@XmlJavaTypeAdapter( IdentifyingAttributesAdapter.class )
	@XmlElement
	protected final IdentifyingAttributes identifyingAttributes;

	@XmlJavaTypeAdapter( StateAttributesAdapter.class )
	@XmlElement
	protected final Attributes attributes;

	@XmlElement
	@XmlJavaTypeAdapter( RenderContainedElementsAdapter.class )
	protected final List<Element> containedElements;

	@XmlElement
	protected Screenshot screenshot;

	@XmlTransient
	private transient Integer hashCodeCache;

	// Warning: Only to be used by JAXB!
	protected Element() {
		retestId = "";
		parent = null;
		identifyingAttributes = null;
		attributes = null;
		containedElements = new ArrayList<>();
	}

	Element( final String retestId, final Element parent, final IdentifyingAttributes identifyingAttributes,
			final Attributes attributes, final Screenshot screenshot ) {
		RetestIdUtil.validate( retestId, identifyingAttributes );
		this.retestId = retestId;
		this.parent = parent;
		this.identifyingAttributes = requireNonNull( identifyingAttributes, "IdentifyingAttributes must not be null" );
		this.attributes = requireNonNull( attributes, "Attributes must not be null" );
		this.screenshot = screenshot;
		containedElements = new ArrayList<>();
	}

	public static Element create( final String retestId, final Element parent,
			final IdentifyingAttributes identifyingAttributes, final Attributes attributes ) {
		return create( retestId, parent, identifyingAttributes, attributes, null );
	}

	public static Element create( final String retestId, final Element parent,
			final IdentifyingAttributes identifyingAttributes, final Attributes attributes,
			final Screenshot screenshot ) {
		requireNonNull( parent, "Parent must not be null" );
		return new Element( retestId, parent, identifyingAttributes, attributes, screenshot );
	}

	public Element applyChanges( final ActionChangeSet actionChangeSet ) {
		if ( actionChangeSet == null ) {
			return this;
		}

		final IdentifyingAttributes newIdentAttributes = identifyingAttributes
				.applyChanges( actionChangeSet.getIdentAttributeChanges().getAll( identifyingAttributes ) );

		final Attributes newAttributes =
				attributes.applyChanges( actionChangeSet.getAttributesChanges().getAll( identifyingAttributes ) );
		final List<Element> newContainedElements = createNewElementList( actionChangeSet, newIdentAttributes );

		final Element element = Element.create( retestId, parent, newIdentAttributes, newAttributes );
		element.addChildren( newContainedElements );
		return element;
	}

	/**
	 * Note that the retest ID is immutable, so this method returns a new element that is a copy of the old one with the
	 * new retest ID.
	 *
	 * @param retestId
	 *            The new retest ID to be used.
	 * @return A copy of the element using the new retest ID.
	 */
	public Element applyRetestId( final String retestId ) {
		return Element.create( retestId, parent, identifyingAttributes, attributes );
	}

	protected List<Element> createNewElementList( final ActionChangeSet actionChangeSet,
			final IdentifyingAttributes newIdentAttributes ) {
		List<Element> newContainedElements = containedElements;
		newContainedElements = removeDeleted( actionChangeSet, newContainedElements );
		newContainedElements =
				applyChangesToContainedElements( actionChangeSet, newIdentAttributes, newContainedElements );
		return addInserted( actionChangeSet, newIdentAttributes, newContainedElements );
	}

	private List<Element> removeDeleted( final ActionChangeSet actionChangeSet,
			final List<Element> oldContainedElements ) {
		final Set<IdentifyingAttributes> deletedChanges = actionChangeSet.getDeletedChanges();
		// IdentifyingAttributes#equals cannot be used here, due to volatile attributes (e.g. outline)
		final List<String> deletedIdentifiers = deletedChanges.stream() //
				.map( IdentifyingAttributes::identifier ) //
				.collect( Collectors.toList() );
		return oldContainedElements.stream() //
				.filter( element -> {
					final IdentifyingAttributes attributes = element.getIdentifyingAttributes();
					return !deletedIdentifiers.contains( attributes.identifier() );
				} ) //
				.collect( Collectors.toList() );
	}

	private List<Element> applyChangesToContainedElements( final ActionChangeSet actionChangeSet,
			final IdentifyingAttributes newIdentAttributes, final List<Element> oldContainedElements ) {
		final List<Element> newContainedElements = new ArrayList<>( oldContainedElements.size() );

		for ( final Element oldElement : oldContainedElements ) {
			addPathChangeToChangeSet( actionChangeSet, newIdentAttributes, oldElement );
			newContainedElements.add( oldElement.applyChanges( actionChangeSet ) );
		}

		return newContainedElements;
	}

	private void addPathChangeToChangeSet( final ActionChangeSet actionChangeSet,
			final IdentifyingAttributes newIdentAttributes, final Element oldElement ) {
		if ( ObjectUtils.notEqual( identifyingAttributes.getPathTyped(), newIdentAttributes.getPathTyped() ) ) {
			final Path oldPath = oldElement.identifyingAttributes.getPathTyped();
			final Path newPath = Path.fromString( newIdentAttributes.getPath() + Path.PATH_SEPARATOR
					+ oldElement.identifyingAttributes.getPathElement().toString() );

			actionChangeSet.getIdentAttributeChanges().add( oldElement.identifyingAttributes,
					new AttributeDifference( "path", oldPath, newPath ) );
		}
	}

	private List<Element> addInserted( final ActionChangeSet actionChangeSet,
			final IdentifyingAttributes newIdentAttributes, final List<Element> newContainedElements ) {
		for ( final Element insertedElement : actionChangeSet.getInsertedChanges() ) {
			if ( isParent( newIdentAttributes, insertedElement.identifyingAttributes ) ) {
				newContainedElements.add( insertedElement );
			}
		}
		return newContainedElements;
	}

	public int countAllContainedElements() {
		// count current elements!
		int result = 1;
		for ( final Element element : containedElements ) {
			result += element.countAllContainedElements();
		}
		return result;
	}

	public Element getElement( final Path path ) {
		final Path thisPath = getIdentifyingAttributes().getPathTyped();
		if ( thisPath.equals( path ) ) {
			return this;
		}
		if ( thisPath.isParent( path ) ) {
			for ( final Element element : containedElements ) {
				final Element contained = element.getElement( path );
				if ( contained != null ) {
					return contained;
				}
			}
		}
		return null;
	}

	public IdentifyingAttributes getIdentifyingAttributes() {
		return identifyingAttributes;
	}

	public List<Element> getContainedElements() {
		return containedElements;
	}

	public Attributes getAttributes() {
		return attributes;
	}

	/**
	 * Returns the value of the given attribute, whether this is an identifying attribute or not.
	 *
	 * @return The {@link java.io.Serializable} value of the attribute.
	 */
	public Object getAttributeValue( final String attributeName ) {
		final Attribute identifyingAttribute = getIdentifyingAttributes().getAttribute( attributeName );
		if ( identifyingAttribute != null ) {
			return identifyingAttribute.getValue();
		}

		return getAttributes().get( attributeName );
	}

	public String getRetestId() {
		return retestId;
	}

	public Screenshot getScreenshot() {
		return screenshot;
	}

	public void setScreenshot( final Screenshot screenshot ) {
		if ( screenshot == null && this.screenshot != null ) {
			throw new RuntimeException( "Screenshot can only be replaced, not deleted." );
		}
		this.screenshot = screenshot;
	}

	public boolean hasContainedElements() {
		return !containedElements.isEmpty();
	}

	private static boolean isParent( final IdentifyingAttributes parentIdentAttributes,
			final IdentifyingAttributes containedIdentAttributes ) {
		return parentIdentAttributes.getPathTyped().equals( containedIdentAttributes.getParentPathTyped() );
	}

	@Transient
	public Element getParent() {
		return parent;
	}

	public void addChildren( final Element... children ) {
		containedElements.addAll( Arrays.asList( children ) );
	}

	public void addChildren( final List<Element> children ) {
		containedElements.addAll( children );
	}

	@Override
	public int compareTo( final Element other ) {
		final int result = identifyingAttributes.compareTo( other.getIdentifyingAttributes() );
		if ( result != 0 ) {
			return result;
		}
		return attributes.compareTo( other.getAttributes() );
	}

	@Override
	public boolean equals( final Object obj ) {
		if ( this == obj ) {
			return true;
		}
		if ( obj == null || getClass() != obj.getClass() || hashCode() != obj.hashCode() ) {
			return false;
		}
		final Element other = (Element) obj;
		if ( !identifyingAttributes.equals( other.identifyingAttributes ) || !attributes.equals( other.attributes ) ) {
			return false;
		}
		return containedElements.equals( other.containedElements );
	}

	@Override
	public int hashCode() {
		if ( hashCodeCache == null ) {
			hashCodeCache = Objects.hash( identifyingAttributes, attributes, containedElements );
		}
		return hashCodeCache;
	}

	@Override
	public String toString() {
		return retestId;
	}

	public RootElement getRootElement() {
		if ( parent != null ) {
			return parent.getRootElement();
		}
		if ( !(this instanceof RootElement) ) {
			throw new IllegalStateException(
					"No parent defined, but element " + toString() + " is no instance of root!" );
		}
		return (RootElement) this;
	}
}
