package streams.esper;

/*
 * #%L
 * Esper implementation of Streams Nodes
 * %%
 * Copyright (C) 2013 University of Zurich
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 2 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-2.0.html>.
 * #L%
 */

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.espertech.esper.client.ConfigurationOperations;
import com.espertech.esper.client.soda.FilterStream;
import com.espertech.esper.client.soda.MethodInvocationStream;
import com.espertech.esper.client.soda.PatternStream;
import com.espertech.esper.client.soda.SQLStream;
import com.espertech.esper.client.soda.Stream;

/**
 * Visit the different kind of streams and check whether their event types are
 * defined in the engine or not.
 * 
 * @author Thomas Scharrenbach
 * @version 0.3.0
 * @since 0.3.0
 * 
 */
public class EsperStreamEventTypeVisitor implements EsperStreamVisitor {

	private static final Logger _log = LoggerFactory
			.getLogger(EsperStreamEventTypeVisitor.class);

	//
	//
	//

	private final ConfigurationOperations _esperConfiguration;

	//
	//
	//

	public EsperStreamEventTypeVisitor(ConfigurationOperations configurationOperations) {
		_esperConfiguration = configurationOperations;
	}

	//
	//
	//

	@Override
	public void visitStream(Stream stream) {
		if (stream instanceof MethodInvocationStream) {
			this.visit((MethodInvocationStream) stream);
		} else if (stream instanceof FilterStream) {
			this.visit((FilterStream) stream);
		} else if (stream instanceof PatternStream) {
			this.visit((PatternStream) stream);
		} else if (stream instanceof SQLStream) {
			this.visit((SQLStream) stream);
		} else {
			throw new IllegalArgumentException(String.format(
					"Cannot visit Esper stream of type %s", stream.getClass()));
		}
	}

	@Override
	public void visit(MethodInvocationStream stream) {
		throw new UnsupportedOperationException("Not yet implemented!");
	}

	@Override
	public void visit(FilterStream stream) {
		final String eventTypeName = stream.getFilter().getEventTypeName();
		if (_esperConfiguration.getEventType(eventTypeName) == null) {
			_log.info("Event type {} not yet included in Esper configuration, "
					+ "declaring new streams-esper EsperData", eventTypeName);
			_esperConfiguration.addEventType(eventTypeName, EsperData.class);
		}
	}

	@Override
	public void visit(PatternStream stream) {
		//throw new UnsupportedOperationException("Not yet implemented!");
	}

	@Override
	public void visit(SQLStream stream) {
		throw new UnsupportedOperationException("Not yet implemented!");
	}

}
