package streams.esper;

/*
 * #%L
 * Esper implementation of Streams Nodes
 * %%
 * Copyright (C) 2013 University of Zurich, Department of Informatics
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 2 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-2.0.html>.
 * #L%
 */

import java.io.Serializable;
import java.util.List;
import java.util.Map.Entry;

import stream.Data;
import stream.data.DataFactory;
import stream.io.Sink;

/**
 * <p>
 * Removes the '`' characters from the keys as being output by Esper. This class
 * hence allows an efficient default implementation and applied this expensive
 * String operation only when explicitly required for a
 * {@link streams.esper.Query}.
 * </p>
 * 
 * @author Thomas Scharrenbach
 * @version 0.3.0
 * @since 0.3.0
 * 
 */
public class EsperTrimmedStatementSubscriber extends EsperStatementSubscriber {

	/**
	 * <p>
	 * Create a new subscriber with the specified keys writing to the specified
	 * {@link Sink}.
	 * </p>
	 * 
	 * @param sinksList
	 * @param keys
	 * 
	 * @version 0.3.0
	 * @since 0.0.1
	 */
	public EsperTrimmedStatementSubscriber(List<Sink> sinksList, String[] keys) {
		super(sinksList, keys);
	}

	/**
	 * <p>
	 * For including special characters, Esper requires names of keys to be
	 * surrounded by '`'. For example, the key myDataItem:timestamp would have
	 * to be expressed as `myDataItem:timestamp`. Since the streams framework
	 * requires no such action, we have to remove these characters before
	 * sending the output of the Esper engine to the Streams framework.
	 * </p>
	 * 
	 * @param values
	 *            the values of an Esper event.
	 * @throws Exception
	 * 
	 * @since 0.3.0
	 * @version 0.3.0
	 */
	public void update(java.util.Map<String, Serializable> values)
			throws Exception {
		final Data item = DataFactory.create();
		for (Entry<String, Serializable> entry : values.entrySet()) {
			item.put(entry.getKey().replace('`', ' ').trim(), entry.getValue());
		}
		write(item);
	}
}
