package streams.esper;

/*
 * #%L
 * Esper implementation of Streams Nodes
 * %%
 * Copyright (C) 2013 University of Zurich
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 2 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-2.0.html>.
 * #L%
 */

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.w3c.dom.Element;

import stream.Configurable;
import stream.annotations.BodyContent;
import stream.annotations.Parameter;
import stream.annotations.Service;

/**
 * <p>
 * Streams-Esper Query objects represent Esper statements. They are Streams
 * {@link stream.Processor} objects and refer to an
 * {@link streams.esper.EsperEngine}.
 * </p>
 * 
 * <p>
 * The {@link stream.util.BodyText} of an XML configuration element represents
 * the actual Esper statement. References to the Esper engine are provided via
 * an attribute.
 * </p>
 * 
 * <p>
 * Note that the Esper types for this query must be delcared to the Esper engine
 * this Query object references.
 * </p>
 * 
 * @author Thomas Scharrenbach
 * @version 0.3.0
 * @since 0.3.0
 * 
 * @see streams.esper.EsperEngine
 * 
 */
public class Query extends stream.AbstractProcessor implements Configurable {

    final static Logger log = LoggerFactory.getLogger(Query.class);

    //
    //
    //

    private stream.io.Sink[] _output;

    private String _engineId;

    private String _esperStatement;

    @Service(required = false)
    private EsperEngine esperEngine;

    private String _name;

    private boolean _removeBackticks;

    private String _inputType;

    private boolean _forward = false;

    //
    //
    //

    /**
     * <p>
     * Hand over the {@link stream.Data} item to the referenced
     * {@link streams.esper.EsperEngine}.
     * </p>
     * 
     * @version 0.3.0
     * @since 0.3.0
     */
    @Override
    public stream.Data process(stream.Data input) {
        try {
            if (input != null) {
                esperEngine.write(input);
            }
        } catch (Exception e) {
            throw new RuntimeException(e);
        }
        return _forward ? input : stream.data.DataFactory.create();
    }

    /**
     * <p>
     * Fetches the Esper engine from the EsperEngine's registry and adds this
     * query to the engine.
     * </p>
     * 
     * @version 0.3.0
     * @since 0.3.0
     * 
     * @throws NullPointerException
     *             if no Esper engine was configured.
     * 
     */
    public void init(stream.ProcessContext ctx) throws Exception {
        super.init(ctx);

        // TODO fast ugly hack: streams default value has not yet been set.
        if (_engineId == null) {
            _engineId = EsperEngine.DEFAULT_ID;
        }

        // Extract the reference to the Esper engine from the registry of
        // EsperEngine.
        esperEngine = EsperEngine.getEsperEngine(_engineId);
        if (esperEngine == null) {
            final String errorMessage = "No Esper engine configured!";
            throw new NullPointerException(errorMessage);
        }

        // Turn this query into a statement bean and add the bean to the engine.
        final EsperStatementBean epStatement = new EsperStatementBean();
        epStatement.setName(getName());
        epStatement.setOutput(getOutput());
        epStatement.setStatement(getQuery());
        epStatement.setRemoveBackticks(isRemoveBackticks());
        esperEngine.addEsperQuery(epStatement, true);
    }

    @Override
    public void finish() throws Exception {
        log.info("Esper query '{}' finished, notifying Esper engine...", this._name);
        esperEngine.notifyShutdown();
        super.finish();
    }

    //
    // Getters and setters
    //

    /**
     * 
     * @version 0.3.0
     * @since 0.3.0
     */
    public String getQuery() {
        log.debug("Returning esper query statement: '{}'", _esperStatement);
        return _esperStatement;
    }

    /**
     * @return the esperEngine
     */
    protected EsperEngine getEsperEngine() {
        return esperEngine;
    }

    /**
     * @param esperEngine
     *            the esperEngine to set
     */
    protected void setEsperEngine(EsperEngine esperEngine) {
        this.esperEngine = esperEngine;
    }

    /**
     * @throws IllegalArgumentException
     *             if the query has already been set.
     * 
     * @version 0.3.0
     * @since 0.3.0
     */
    public void setQuery(BodyContent esperQuery) {
        if (_esperStatement != null) {
            final String errorMessage = String.format("Parameter %s already defined.", "query");
            throw new IllegalArgumentException(errorMessage);
        }
        _esperStatement = esperQuery.getContent();
        log.debug("Esper query is: '{}'", _esperStatement);
    }

    /**
     * @throws IllegalArgumentException
     *             if the sinks have already been set.
     * 
     * @version 0.3.0
     * @since 0.3.0
     */
    @Parameter
    public void setOutput(stream.io.Sink[] output) {
        if (_output != null) {
            final String errorMessage = String.format("Parameter %s already defined.", "output");
            throw new IllegalArgumentException(errorMessage);
        }
        _output = output;
    }

    /**
     * 
     * @version 0.3.0
     * @since 0.3.0
     */
    public stream.io.Sink[] getOutput() {
        return _output;
    }

    /**
     * @throws IllegalArgumentException
     *             if the engine id has already been set.
     * @version 0.3.0
     * @since 0.3.0
     */
    @Parameter(defaultValue = "default", required = false)
    public void setEngine(String engine) {
        if (_engineId != null) {
            final String errorMessage = String.format("Parameter %s already defined.", "engine");
            throw new IllegalArgumentException(errorMessage);
        }
        _engineId = engine;
    }

    /**
     * 
     * @version 0.3.0
     * @since 0.3.0
     */
    public String getEngine() {
        return _engineId;
    }

    /**
     * 
     * @version 0.3.0
     * @since 0.3.0
     */
    @Parameter(required = false, description = "The optional name for the Esper statement.")
    public void setName(String name) {
        if (_name != null) {
            final String errorMessage = String.format("Parameter %s already defined.", "name");
            throw new IllegalArgumentException(errorMessage);
        }
        _name = name;
    }

    /**
     * 
     * @version 0.3.0
     * @since 0.3.0
     */
    public String getName() {
        return _name;
    }

    /**
     * 
     * @version 0.3.0
     * @since 0.3.0
     */
    @Parameter(required = false, defaultValue = "false", description = "Flag whether to remove backticks from the keys.")
    public void setRemoveBackticks(Boolean removeBackticks) {
        _removeBackticks = removeBackticks;
    }

    /**
     * 
     * @version 0.3.0
     * @since 0.3.0
     */
    public Boolean isRemoveBackticks() {
        return _removeBackticks;
    }

    /**
     * 
     * @version 0.3.0
     * @since 0.3.0
     */
    @Parameter(required = false, defaultValue = "", description = "If this parameter is set we assume that events input to this query are of a Map type.")
    public void setInputType(String inputType) {
        _inputType = (inputType == null || inputType.trim().isEmpty()) ? null : inputType.trim();
    }

    /**
     * 
     * @version 0.3.0
     * @since 0.3.0
     */
    public String getInputType() {
        return _inputType;
    }

    @Parameter(defaultValue = "false", name = "forward-input", description = "Flag that inicates whether a data item should be forwarded to later processors. ")
    public void setForward(Boolean forward) {
        _forward = forward;
    }

    public boolean isForward() {
        return _forward;
    }

    /**
     * @see stream.Configurable#configure(org.w3c.dom.Element)
     */
    @Override
    public void configure(Element document) {
        String body = document.getTextContent();

        log.info("Configuring 'streams.esper.Query' element. Body content is:\n{}", body);
        if (body != null) {
            this.setQuery(new BodyContent(body));
        }
    }

}