/*
 * Decompiled with CFR 0.152.
 */
package de.softwareforge.testing.postgres.embedded;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Joiner;
import com.google.common.base.Preconditions;
import com.google.common.base.Stopwatch;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableMap;
import com.google.common.io.CharStreams;
import com.google.common.io.Closeables;
import de.softwareforge.testing.postgres.embedded.DatabaseInfo;
import de.softwareforge.testing.postgres.embedded.EmbeddedUtil;
import de.softwareforge.testing.postgres.embedded.NativeBinaryManager;
import de.softwareforge.testing.postgres.embedded.ProcessOutputLogger;
import de.softwareforge.testing.postgres.embedded.TarXzCompressedBinaryManager;
import de.softwareforge.testing.postgres.embedded.ZonkyIOPostgresLocator;
import edu.umd.cs.findbugs.annotations.NonNull;
import edu.umd.cs.findbugs.annotations.Nullable;
import java.io.Closeable;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStreamReader;
import java.nio.channels.FileLock;
import java.nio.channels.OverlappingFileLockException;
import java.nio.charset.StandardCharsets;
import java.nio.file.Path;
import java.sql.SQLException;
import java.time.Duration;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import javax.sql.DataSource;
import org.postgresql.ds.PGSimpleDataSource;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public final class EmbeddedPostgres
implements AutoCloseable {
    public static final String DEFAULT_POSTGRES_VERSION = "13";
    static final String[] LOCALHOST_SERVER_NAMES = new String[]{"localhost"};
    private static final String PG_TEMPLATE_DB = "template1";
    @VisibleForTesting
    static final Duration DEFAULT_PG_STARTUP_WAIT = Duration.ofSeconds(10L);
    private static final long MINIMUM_AGE_IN_MS = Duration.ofMinutes(10L).toMillis();
    private static final String DATA_DIRECTORY_PREFIX = "data-";
    private static final String PG_STOP_MODE = "fast";
    private static final String PG_STOP_WAIT_SECONDS = "5";
    static final String LOCK_FILE_NAME = "epg-lock";
    private final Logger logger;
    private final String instanceId;
    private final File postgresInstallDirectory;
    private final File dataDirectory;
    private final Duration serverStartupWait;
    private final int port;
    private final AtomicBoolean started = new AtomicBoolean();
    private final AtomicBoolean closed = new AtomicBoolean();
    private final ImmutableMap<String, String> serverConfiguration;
    private final ImmutableMap<String, String> localeConfiguration;
    private final ImmutableMap<String, String> connectionProperties;
    private final File lockFile;
    private volatile FileOutputStream lockStream;
    private volatile FileLock lock;
    private final boolean removeDataOnShutdown;
    private final ProcessBuilder.Redirect errorRedirector;
    private final ProcessBuilder.Redirect outputRedirector;
    private final ProcessOutputLogger pgServerLogger;

    @NonNull
    public static EmbeddedPostgres defaultInstance() throws IOException {
        return EmbeddedPostgres.builderWithDefaults().build();
    }

    @NonNull
    public static Builder builderWithDefaults() {
        return new Builder().withDefaults();
    }

    public static EmbeddedPostgres forVersionCheck() throws IOException {
        return new Builder(false).build();
    }

    @NonNull
    public static Builder builder() {
        return new Builder();
    }

    private EmbeddedPostgres(String instanceId, File postgresInstallDirectory, File dataDirectory, boolean removeDataOnShutdown, Map<String, String> serverConfiguration, Map<String, String> localeConfiguration, Map<String, String> connectionProperties, int port, ProcessBuilder.Redirect errorRedirector, ProcessBuilder.Redirect outputRedirector, Duration serverStartupWait) {
        this.instanceId = (String)Preconditions.checkNotNull((Object)instanceId, (Object)"instanceId is null");
        this.logger = LoggerFactory.getLogger((String)this.toString());
        this.pgServerLogger = new ProcessOutputLogger(this.logger);
        this.postgresInstallDirectory = (File)Preconditions.checkNotNull((Object)postgresInstallDirectory, (Object)"postgresInstallDirectory is null");
        this.dataDirectory = (File)Preconditions.checkNotNull((Object)dataDirectory, (Object)"dataDirectory is null");
        this.removeDataOnShutdown = removeDataOnShutdown;
        this.serverConfiguration = ImmutableMap.copyOf((Map)((Map)Preconditions.checkNotNull(serverConfiguration, (Object)"serverConfiguration is null")));
        this.localeConfiguration = ImmutableMap.copyOf((Map)((Map)Preconditions.checkNotNull(localeConfiguration, (Object)"localeConfiguration is null")));
        this.connectionProperties = ImmutableMap.copyOf((Map)((Map)Preconditions.checkNotNull(connectionProperties, (Object)"connectionProperties is null")));
        this.port = port;
        this.errorRedirector = (ProcessBuilder.Redirect)Preconditions.checkNotNull((Object)errorRedirector, (Object)"errorRedirector is null");
        this.outputRedirector = (ProcessBuilder.Redirect)Preconditions.checkNotNull((Object)outputRedirector, (Object)"outputRedirector is null");
        this.serverStartupWait = (Duration)Preconditions.checkNotNull((Object)serverStartupWait, (Object)"serverStartupWait is null");
        this.lockFile = new File(this.dataDirectory, LOCK_FILE_NAME);
        this.logger.debug(String.format("data dir is %s, install dir is %s", this.dataDirectory, this.postgresInstallDirectory));
    }

    @NonNull
    public DataSource createTemplateDataSource() throws SQLException {
        Preconditions.checkState((boolean)this.started.get(), (Object)"instance has not been started!");
        return EmbeddedPostgres.createDataSource("postgres", PG_TEMPLATE_DB, this.getPort(), this.getConnectionProperties());
    }

    @NonNull
    public DataSource createDefaultDataSource() throws SQLException {
        Preconditions.checkState((boolean)this.started.get(), (Object)"instance has not been started!");
        return EmbeddedPostgres.createDataSource("postgres", "postgres", this.getPort(), this.getConnectionProperties());
    }

    @NonNull
    public DataSource createDataSource(@NonNull String user, @NonNull String databaseName) throws SQLException {
        Preconditions.checkState((boolean)this.started.get(), (Object)"instance has not been started!");
        return EmbeddedPostgres.createDataSource(user, databaseName, this.getPort(), this.getConnectionProperties());
    }

    static DataSource createDataSource(String user, String databaseName, int port, Map<String, String> connectionProperties) throws SQLException {
        Preconditions.checkNotNull((Object)user, (Object)"user is null");
        Preconditions.checkNotNull((Object)databaseName, (Object)"databaseName is null");
        Preconditions.checkNotNull(connectionProperties, (Object)"connectionProperties is null");
        PGSimpleDataSource ds = new PGSimpleDataSource();
        ds.setServerNames(LOCALHOST_SERVER_NAMES);
        ds.setPortNumbers(new int[]{port});
        ds.setDatabaseName(databaseName);
        ds.setUser(user);
        for (Map.Entry<String, String> entry : connectionProperties.entrySet()) {
            ds.setProperty(entry.getKey(), entry.getValue());
        }
        return ds;
    }

    public int getPort() {
        Preconditions.checkState((boolean)this.started.get(), (Object)"instance has not been started!");
        return this.port;
    }

    @NonNull
    ImmutableMap<String, String> getConnectionProperties() {
        Preconditions.checkState((boolean)this.started.get(), (Object)"instance has not been started!");
        return this.connectionProperties;
    }

    @NonNull
    public String instanceId() {
        Preconditions.checkState((boolean)this.started.get(), (Object)"instance has not been started!");
        return this.instanceId;
    }

    public String getPostgresVersion() throws IOException {
        StringBuilder sb = new StringBuilder();
        ProcessOutputLogger.StreamCapture logCapture = this.pgServerLogger.captureStreamAsConsumer(sb::append);
        ImmutableList commandAndArgs = ImmutableList.of((Object)this.pgBin("pg_ctl"), (Object)"--version");
        Stopwatch watch = this.system((List<String>)commandAndArgs, logCapture);
        String version = "unknown";
        try {
            logCapture.getCompletion().get();
            String s = sb.toString();
            Preconditions.checkState((boolean)s.startsWith("pg_ctl "), (String)"Response %s does not match 'pg_ctl'", (Object)sb);
            version = s.substring(s.lastIndexOf(32)).trim();
        }
        catch (ExecutionException e) {
            throw new IOException(String.format("Process '%s' failed%n%s", Joiner.on((String)" ").join((Iterable)commandAndArgs)), e);
        }
        catch (InterruptedException e) {
            Thread.currentThread().interrupt();
        }
        this.logger.debug(String.format("postgres version check completed in %s", EmbeddedUtil.formatDuration(watch.elapsed())));
        return version;
    }

    public String toString() {
        return this.getClass().getName() + "$" + this.instanceId;
    }

    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || this.getClass() != o.getClass()) {
            return false;
        }
        EmbeddedPostgres that = (EmbeddedPostgres)o;
        return this.instanceId.equals(that.instanceId);
    }

    public int hashCode() {
        return Objects.hash(this.instanceId);
    }

    DatabaseInfo createDefaultDatabaseInfo() {
        return DatabaseInfo.builder().port(this.getPort()).connectionProperties(this.getConnectionProperties()).build();
    }

    private void boot() throws IOException {
        EmbeddedUtil.mkdirs(this.dataDirectory);
        if (this.removeDataOnShutdown || !new File(this.dataDirectory, "postgresql.conf").exists()) {
            this.initDatabase();
        }
        this.lock();
        this.startDatabase();
    }

    private synchronized void lock() throws IOException {
        this.lockStream = new FileOutputStream(this.lockFile);
        this.lock = this.lockStream.getChannel().tryLock();
        Preconditions.checkState((this.lock != null ? 1 : 0) != 0, (String)"could not lock %s", (Object)this.lockFile);
    }

    private synchronized void unlock() throws IOException {
        if (this.lock != null) {
            this.lock.release();
        }
        Closeables.close((Closeable)this.lockStream, (boolean)true);
    }

    private void initDatabase() throws IOException {
        ImmutableList.Builder commandBuilder = ImmutableList.builder();
        commandBuilder.add((Object)this.pgBin("initdb")).addAll(this.createInitDbOptions()).add((Object[])new String[]{"-A", "trust", "-U", "postgres", "-D", this.dataDirectory.getPath(), "-E", "UTF-8"});
        Stopwatch watch = this.system((List<String>)commandBuilder.build(), this.pgServerLogger.captureStreamAsLog());
        this.logger.debug(String.format("initdb completed in %s", EmbeddedUtil.formatDuration(watch.elapsed())));
    }

    private void startDatabase() throws IOException {
        Preconditions.checkState((!this.started.getAndSet(true) ? 1 : 0) != 0, (Object)"database already started!");
        ImmutableList.Builder commandBuilder = ImmutableList.builder();
        commandBuilder.add((Object[])new String[]{this.pgBin("pg_ctl"), "-D", this.dataDirectory.getPath(), "-o", String.join((CharSequence)" ", this.createInitOptions()), "start"});
        Stopwatch watch = Stopwatch.createStarted();
        Process postmaster = this.spawn("pg", (List<String>)commandBuilder.build(), this.pgServerLogger.captureStreamAsLog());
        this.logger.info(String.format("started as pid %d on port %d", postmaster.pid(), this.port));
        this.logger.debug(String.format("Waiting up to %s for server startup to finish", EmbeddedUtil.formatDuration(this.serverStartupWait)));
        Runtime.getRuntime().addShutdownHook(this.newCloserThread());
        Preconditions.checkState((boolean)this.waitForServerStartup(), (Object)"Could not start PostgreSQL server, interrupted?");
        this.logger.debug(String.format("startup complete in %s", EmbeddedUtil.formatDuration(watch.elapsed())));
    }

    private void stopDatabase(File dataDirectory) throws IOException {
        if (this.started.get()) {
            ImmutableList.Builder commandBuilder = ImmutableList.builder();
            commandBuilder.add((Object[])new String[]{this.pgBin("pg_ctl"), "-D", dataDirectory.getPath(), "stop", "-m", PG_STOP_MODE, "-t", PG_STOP_WAIT_SECONDS, "-w"});
            Stopwatch watch = this.system((List<String>)commandBuilder.build(), this.pgServerLogger.captureStreamAsLog());
            this.logger.debug(String.format("shutdown complete in %s", EmbeddedUtil.formatDuration(watch.elapsed())));
        }
        this.pgServerLogger.close();
    }

    private List<String> createInitOptions() {
        ImmutableList.Builder initOptions = ImmutableList.builder();
        initOptions.add((Object[])new String[]{"-p", Integer.toString(this.port), "-F"});
        this.serverConfiguration.forEach((key, value) -> {
            initOptions.add((Object)"-c");
            if (value.length() > 0) {
                initOptions.add((Object)(key + "=" + value));
            } else {
                initOptions.add((Object)(key + "=true"));
            }
        });
        return initOptions.build();
    }

    @VisibleForTesting
    List<String> createInitDbOptions() {
        ImmutableList.Builder localeOptions = ImmutableList.builder();
        this.localeConfiguration.forEach((key, value) -> {
            if (value.length() > 0) {
                localeOptions.add((Object)("--" + key + "=" + value));
            } else {
                localeOptions.add((Object)("--" + key));
            }
        });
        return localeOptions.build();
    }

    private boolean waitForServerStartup() throws IOException {
        SQLException lastCause = null;
        long start = System.nanoTime();
        long maxWaitNs = TimeUnit.NANOSECONDS.convert(this.serverStartupWait.toMillis(), TimeUnit.MILLISECONDS);
        while (System.nanoTime() - start < maxWaitNs) {
            try {
                if (this.verifyReady()) {
                    return true;
                }
            }
            catch (SQLException e) {
                lastCause = e;
                this.logger.trace("while waiting for server startup:", (Throwable)e);
            }
            try {
                Thread.sleep(100L);
            }
            catch (InterruptedException e) {
                Thread.currentThread().interrupt();
                return false;
            }
        }
        throw new IOException("Gave up waiting for server to start after " + this.serverStartupWait.toMillis() + "ms", lastCause);
    }

    /*
     * Exception decompiling
     */
    private boolean verifyReady() throws IOException, SQLException {
        /*
         * This method has failed to decompile.  When submitting a bug report, please provide this stack trace, and (if you hold appropriate legal rights) the relevant class file.
         * 
         * org.benf.cfr.reader.util.ConfusedCFRException: Started 2 blocks at once
         *     at org.benf.cfr.reader.bytecode.analysis.opgraph.Op04StructuredStatement.getStartingBlocks(Op04StructuredStatement.java:412)
         *     at org.benf.cfr.reader.bytecode.analysis.opgraph.Op04StructuredStatement.buildNestedBlocks(Op04StructuredStatement.java:487)
         *     at org.benf.cfr.reader.bytecode.analysis.opgraph.Op03SimpleStatement.createInitialStructuredBlock(Op03SimpleStatement.java:736)
         *     at org.benf.cfr.reader.bytecode.CodeAnalyser.getAnalysisInner(CodeAnalyser.java:850)
         *     at org.benf.cfr.reader.bytecode.CodeAnalyser.getAnalysisOrWrapFail(CodeAnalyser.java:278)
         *     at org.benf.cfr.reader.bytecode.CodeAnalyser.getAnalysis(CodeAnalyser.java:201)
         *     at org.benf.cfr.reader.entities.attributes.AttributeCode.analyse(AttributeCode.java:94)
         *     at org.benf.cfr.reader.entities.Method.analyse(Method.java:531)
         *     at org.benf.cfr.reader.entities.ClassFile.analyseMid(ClassFile.java:1055)
         *     at org.benf.cfr.reader.entities.ClassFile.analyseTop(ClassFile.java:942)
         *     at org.benf.cfr.reader.Driver.doJarVersionTypes(Driver.java:257)
         *     at org.benf.cfr.reader.Driver.doJar(Driver.java:139)
         *     at org.benf.cfr.reader.CfrDriverImpl.analyse(CfrDriverImpl.java:76)
         *     at org.benf.cfr.reader.Main.main(Main.java:54)
         */
        throw new IllegalStateException("Decompilation failed");
    }

    private Thread newCloserThread() {
        Thread closeThread = new Thread(() -> {
            try {
                this.close();
            }
            catch (IOException e) {
                this.logger.trace("while closing instance:", (Throwable)e);
            }
        });
        closeThread.setName("pg-closer");
        return closeThread;
    }

    @Override
    public void close() throws IOException {
        if (this.closed.getAndSet(true)) {
            return;
        }
        try {
            this.stopDatabase(this.dataDirectory);
        }
        catch (Exception e) {
            this.logger.error("could not stop pg:", (Throwable)e);
        }
        this.unlock();
        if (this.removeDataOnShutdown) {
            try {
                EmbeddedUtil.rmdirs(this.dataDirectory);
            }
            catch (Exception e) {
                this.logger.error(String.format("Could not clean up directory %s:", this.dataDirectory.getAbsolutePath()), (Throwable)e);
            }
        } else {
            this.logger.debug(String.format("preserved data directory %s", this.dataDirectory.getAbsolutePath()));
        }
    }

    @VisibleForTesting
    File getDataDirectory() {
        return this.dataDirectory;
    }

    @VisibleForTesting
    Map<String, String> getLocaleConfiguration() {
        return this.localeConfiguration;
    }

    private void cleanOldDataDirectories(File parentDirectory) {
        File[] children = parentDirectory.listFiles();
        if (children == null) {
            return;
        }
        for (File dir : children) {
            File lockFile;
            if (!dir.isDirectory() || !dir.getName().startsWith(DATA_DIRECTORY_PREFIX) || !(lockFile = new File(dir, LOCK_FILE_NAME)).exists() || System.currentTimeMillis() - lockFile.lastModified() < MINIMUM_AGE_IN_MS) continue;
            try (FileOutputStream fos = new FileOutputStream(lockFile);
                 FileLock lock = fos.getChannel().tryLock();){
                if (lock == null) continue;
                this.logger.debug(String.format("found stale data directory %s", dir));
                if (new File(dir, "postmaster.pid").exists()) {
                    try {
                        this.stopDatabase(dir);
                        this.logger.debug("shutting down orphaned database!");
                    }
                    catch (Exception e) {
                        this.logger.warn(String.format("failed to orphaned database in %s:", dir), (Throwable)e);
                    }
                }
                EmbeddedUtil.rmdirs(dir);
            }
            catch (OverlappingFileLockException e) {
                this.logger.trace("while cleaning old data directories:", (Throwable)e);
            }
            catch (Exception e) {
                this.logger.warn("while cleaning old data directories:", (Throwable)e);
            }
        }
    }

    private String pgBin(String binaryName) {
        String extension = EmbeddedUtil.IS_OS_WINDOWS ? ".exe" : "";
        return new File(this.postgresInstallDirectory, "bin/" + binaryName + extension).getPath();
    }

    private Process spawn(@Nullable String processName, List<String> commandAndArgs, ProcessOutputLogger.StreamCapture logCapture) throws IOException {
        ProcessBuilder builder = new ProcessBuilder(commandAndArgs);
        builder.redirectErrorStream(true);
        builder.redirectError(this.errorRedirector);
        builder.redirectOutput(this.outputRedirector);
        Process process = builder.start();
        if (this.outputRedirector == ProcessBuilder.Redirect.PIPE) {
            processName = processName != null ? processName : process.info().command().map(EmbeddedUtil::getFileBaseName).orElse("<unknown>");
            String name = String.format("%s (%d)", processName, process.pid());
            logCapture.accept(name, process.getInputStream());
        }
        return process;
    }

    private Stopwatch system(List<String> commandAndArgs, ProcessOutputLogger.StreamCapture logCapture) throws IOException {
        Stopwatch watch;
        block7: {
            Preconditions.checkArgument((commandAndArgs.size() > 0 ? 1 : 0) != 0, (Object)"No commandAndArgs given!");
            String prefix = EmbeddedUtil.getFileBaseName(commandAndArgs.get(0));
            watch = Stopwatch.createStarted();
            try {
                Process process = this.spawn(prefix, commandAndArgs, logCapture);
                if (process.waitFor() == 0) break block7;
                if (this.errorRedirector == ProcessBuilder.Redirect.PIPE) {
                    InputStreamReader errorReader = new InputStreamReader(process.getErrorStream(), StandardCharsets.UTF_8);
                    try {
                        throw new IOException(String.format("Process '%s' failed%n%s", Joiner.on((String)" ").join(commandAndArgs), CharStreams.toString((Readable)errorReader)));
                    }
                    catch (Throwable throwable) {
                        try {
                            errorReader.close();
                        }
                        catch (Throwable throwable2) {
                            throwable.addSuppressed(throwable2);
                        }
                        throw throwable;
                    }
                }
                throw new IOException(String.format("Process '%s' failed", Joiner.on((String)" ").join(commandAndArgs)));
            }
            catch (InterruptedException e) {
                Thread.currentThread().interrupt();
            }
        }
        return watch;
    }

    public static class Builder {
        private File installationBaseDirectory = null;
        private File dataDirectory = null;
        private final Map<String, String> serverConfiguration = new HashMap<String, String>();
        private final Map<String, String> localeConfiguration = new HashMap<String, String>();
        private boolean removeDataOnShutdown = true;
        private int port = 0;
        private String serverVersion = "13";
        private final Map<String, String> connectionProperties = new HashMap<String, String>();
        private NativeBinaryManager nativeBinaryManager = null;
        private Duration serverStartupWait = DEFAULT_PG_STARTUP_WAIT;
        private ProcessBuilder.Redirect errorRedirector = ProcessBuilder.Redirect.PIPE;
        private ProcessBuilder.Redirect outputRedirector = ProcessBuilder.Redirect.PIPE;
        private boolean bootInstance;

        private Builder(boolean bootInstance) {
            this.bootInstance = bootInstance;
        }

        Builder() {
            this(true);
        }

        @NonNull
        public Builder withDefaults() {
            this.serverConfiguration.put("timezone", "UTC");
            this.serverConfiguration.put("synchronous_commit", "off");
            this.serverConfiguration.put("max_connections", "300");
            return this;
        }

        @NonNull
        public Builder setServerStartupWait(@NonNull Duration serverStartupWait) {
            Preconditions.checkNotNull((Object)serverStartupWait, (Object)"serverStartupWait is null");
            Preconditions.checkArgument((!serverStartupWait.isNegative() ? 1 : 0) != 0, (Object)"Negative durations are not permitted.");
            this.serverStartupWait = serverStartupWait;
            return this;
        }

        @NonNull
        public Builder setRemoveDataOnShutdown(boolean removeDataOnShutdown) {
            this.removeDataOnShutdown = removeDataOnShutdown;
            return this;
        }

        @NonNull
        public Builder setDataDirectory(@NonNull Path dataDirectory) {
            Preconditions.checkNotNull((Object)dataDirectory, (Object)"dataDirectory is null");
            return this.setDataDirectory(dataDirectory.toFile());
        }

        @NonNull
        public Builder setDataDirectory(@NonNull String dataDirectory) {
            Preconditions.checkNotNull((Object)dataDirectory, (Object)"dataDirectory is null");
            return this.setDataDirectory(new File(dataDirectory));
        }

        @NonNull
        public Builder setDataDirectory(@NonNull File dataDirectory) {
            this.dataDirectory = (File)Preconditions.checkNotNull((Object)dataDirectory, (Object)"dataDirectory is null");
            return this;
        }

        @NonNull
        public Builder addServerConfiguration(@NonNull String key, @NonNull String value) {
            Preconditions.checkNotNull((Object)key, (Object)"key is null");
            Preconditions.checkNotNull((Object)value, (Object)"value is null");
            this.serverConfiguration.put(key, value);
            return this;
        }

        @Deprecated
        @NonNull
        public Builder addLocaleConfiguration(@NonNull String key, @NonNull String value) {
            Preconditions.checkNotNull((Object)key, (Object)"key is null");
            Preconditions.checkNotNull((Object)value, (Object)"value is null");
            this.localeConfiguration.put(key, value);
            return this;
        }

        @NonNull
        public Builder addInitDbConfiguration(@NonNull String key, @NonNull String value) {
            Preconditions.checkNotNull((Object)key, (Object)"key is null");
            Preconditions.checkNotNull((Object)value, (Object)"value is null");
            this.localeConfiguration.put(key, value);
            return this;
        }

        @NonNull
        public Builder addConnectionProperty(@NonNull String key, @NonNull String value) {
            Preconditions.checkNotNull((Object)key, (Object)"key is null");
            Preconditions.checkNotNull((Object)value, (Object)"value is null");
            this.connectionProperties.put(key, value);
            return this;
        }

        @NonNull
        public Builder setInstallationBaseDirectory(@NonNull File installationBaseDirectory) {
            Preconditions.checkNotNull((Object)installationBaseDirectory, (Object)"installationBaseDirectory is null");
            this.installationBaseDirectory = installationBaseDirectory;
            this.nativeBinaryManager = null;
            return this;
        }

        @NonNull
        public Builder setPort(int port) {
            Preconditions.checkState((port > 1023 && port < 65535 ? 1 : 0) != 0, (String)"Port %s is not within 1024..65535", (int)port);
            this.port = port;
            return this;
        }

        @NonNull
        public Builder setServerVersion(@NonNull String serverVersion) {
            this.serverVersion = (String)Preconditions.checkNotNull((Object)serverVersion, (Object)"serverVersion is null");
            return this;
        }

        @NonNull
        public Builder setErrorRedirector(@NonNull ProcessBuilder.Redirect errorRedirector) {
            this.errorRedirector = (ProcessBuilder.Redirect)Preconditions.checkNotNull((Object)errorRedirector, (Object)"errorRedirector is null");
            return this;
        }

        @NonNull
        public Builder setOutputRedirector(@NonNull ProcessBuilder.Redirect outputRedirector) {
            this.outputRedirector = (ProcessBuilder.Redirect)Preconditions.checkNotNull((Object)outputRedirector, (Object)"outputRedirector is null");
            return this;
        }

        @NonNull
        public Builder setNativeBinaryManager(@NonNull NativeBinaryManager nativeBinaryManager) {
            this.nativeBinaryManager = (NativeBinaryManager)Preconditions.checkNotNull((Object)nativeBinaryManager, (Object)"nativeBinaryManager is null");
            return this;
        }

        @NonNull
        public Builder useLocalPostgresInstallation(@NonNull File directory) {
            Preconditions.checkNotNull((Object)directory, (Object)"directory is null");
            Preconditions.checkState((directory.exists() && directory.isDirectory() ? 1 : 0) != 0, (String)"'%s' either does not exist or is not a directory!", (Object)directory);
            return this.setNativeBinaryManager(() -> directory);
        }

        @NonNull
        public EmbeddedPostgres build() throws IOException {
            String instanceId = EmbeddedUtil.randomAlphaNumeric(16);
            int port = this.port != 0 ? this.port : EmbeddedUtil.allocatePort();
            File parentDirectory = EmbeddedUtil.getWorkingDirectory();
            EmbeddedUtil.mkdirs(parentDirectory);
            NativeBinaryManager nativeBinaryManager = this.nativeBinaryManager;
            if (nativeBinaryManager == null) {
                String serverVersion = System.getProperty("pg-embedded.postgres-version", this.serverVersion);
                nativeBinaryManager = new TarXzCompressedBinaryManager(new ZonkyIOPostgresLocator(serverVersion));
            }
            File installationBaseDirectory = Objects.requireNonNullElse(this.installationBaseDirectory, parentDirectory);
            nativeBinaryManager.setInstallationBaseDirectory(installationBaseDirectory);
            File postgresInstallDirectory = nativeBinaryManager.getLocation();
            File dataDirectory = this.dataDirectory;
            if (dataDirectory == null) {
                dataDirectory = new File(parentDirectory, EmbeddedPostgres.DATA_DIRECTORY_PREFIX + instanceId);
            }
            EmbeddedPostgres embeddedPostgres = new EmbeddedPostgres(instanceId, postgresInstallDirectory, dataDirectory, this.removeDataOnShutdown, this.serverConfiguration, this.localeConfiguration, this.connectionProperties, port, this.errorRedirector, this.outputRedirector, this.serverStartupWait);
            embeddedPostgres.cleanOldDataDirectories(parentDirectory);
            if (this.bootInstance) {
                embeddedPostgres.boot();
            }
            return embeddedPostgres;
        }
    }

    @Deprecated
    @FunctionalInterface
    public static interface BuilderCustomizer {
        public void customize(@NonNull Builder var1) throws IOException, SQLException;
    }
}

