package ninja.mongodb;

import java.net.UnknownHostException;
import java.util.List;

import ninja.utils.NinjaProperties;

import org.bson.types.ObjectId;
import org.mongodb.morphia.Datastore;
import org.mongodb.morphia.Morphia;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.google.common.base.Preconditions;
import com.google.inject.Inject;
import com.google.inject.Singleton;
import com.mongodb.MongoClient;

/**
 * Convenient class for interacting with MongoDB and/or Morphia 
 * in the Ninja Web Framework
 * 
 * @author svenkubiak
 *
 */
@Singleton
public class MongoDB {
    private static final Logger LOG = LoggerFactory.getLogger(MongoDB.class);
    private static final String MONGODB_HOST = "ninja.mongodb.host";
    private static final String MONGODB_PORT = "ninja.mongodb.port";
    private static final String MONGODB_DBNAME = "ninja.mongodb.dbname";
    private static final String MORPHIA_PACKAGE = "ninja.morphia.package";
    private Datastore datastore;
    private Morphia morphia;
    private MongoClient mongoClient;
    private NinjaProperties ninjaProperties;
    
    @Inject
    private MongoDB(NinjaProperties ninjaProperties) {
        this.ninjaProperties = ninjaProperties;
        
        final String host = this.ninjaProperties.get(MONGODB_HOST);
        final int port = this.ninjaProperties.getInteger(MONGODB_PORT);
        try {
            this.mongoClient = new MongoClient(host, port);
            LOG.info("Successfully created MongoClient @ " + host + ":" + port);
        } catch (UnknownHostException e) {
            LOG.error("Failed to created MongoClient @ " + host + ":" + port, e);
        }
    }
    
    /**
     * Creates a Morphia Datastore if none present
     * 
     * @return Morphia Datastore object
     */
    public Datastore getDatastore() {
        Preconditions.checkNotNull(this.mongoClient);
        if (this.datastore == null) {
            initMorphia(this.mongoClient);
        }

        return this.datastore;
    }
    
    /**
     * Creates Morphia instance if none present
     * 
     * @return Morphia instance object
     */
    public Morphia getMorphia() {
        Preconditions.checkNotNull(this.mongoClient);
        if (this.morphia == null) {
            initMorphia(this.mongoClient);
        }
        
        return morphia;
    }

    /**
     * Returns the MongoClient instance
     * 
     * @return MongoClient
     */
    public MongoClient getMongoClient() {
        return mongoClient;
    }
    
    /**
     * Convenient method for overwriting the Morphia
     * object with a given MongoClient
     * 
     * @param MongoClient object
     */
    public void setMongoClient(MongoClient mongoClient) {
        Preconditions.checkNotNull(mongoClient);
        this.mongoClient = mongoClient;
        
        LOG.info("Successfully set MongoClient @ " + mongoClient.getAddress().getHost() + ":" + mongoClient.getAddress().getPort());
        
        initMorphia(mongoClient);
    }

    /**
     * Initializes the Morphia instance by setting the models with
     * packages and creating the datastore
     * 
     * @param mongoClient The MongoClient with the MongoDB connection
     */
    private void initMorphia(MongoClient mongoClient) {
        final String packageName = ninjaProperties.get(MORPHIA_PACKAGE);
        final String dbName = ninjaProperties.get(MONGODB_DBNAME);
        
        this.morphia = new Morphia().mapPackage(packageName);
        this.datastore = this.morphia.createDatastore(mongoClient, dbName);
        
        LOG.info("Mapped Morphia to package '" + packageName + "' and created Morphia Datastore to database '" + dbName + "'");
    }
    
    /**
     * Retrieves a mapped Morphia object from MongoDB. If the id is not of 
     * type ObjectId, it will we converted to ObjectId
     * 
     * @param id The id of the object
     * @param clazz The mapped Morphia class
     * 
     * @return The requested class from MongoDB or null if none found
     */
    public <T extends Object> T findById(Object id, Class<T> clazz) {
        Preconditions.checkNotNull(clazz);
        Preconditions.checkNotNull(id);

        String objectId = null;
        if (!(id instanceof ObjectId)) {
            objectId = String.valueOf(id);
        }
        
        return this.datastore.get(clazz, new ObjectId(objectId));  
    }
    
    /**
     * Retrieves all mapped Morphia objects from MongoDB
     * 
     * @param clazz The mapped Morphia class
     * @return A list of mapped Morphia objects or an empty list of none found
     */
    public <T extends Object> List<T> findAll(Class<T> clazz) {
        Preconditions.checkNotNull(clazz);
        
        return this.datastore.find(clazz).asList();
    }
    
    /**
     * Counts all objected of a mapped Morphia class
     * 
     * @param clazz The mapped Morphia class
     * @return The number of objects in MongoDB
     */
    public <T extends Object> long countAll(Class<T> clazz) {
        Preconditions.checkNotNull(clazz);
        
        return this.datastore.find(clazz).countAll();
    }
    
    /**
     * Saves a mapped Morphia object to MongoDB
     * 
     * @param object The object to save
     */
    public void save(Object object) {
        Preconditions.checkNotNull(object);
        
        this.datastore.save(object);
    }
    
    /**
     * Deletes a mapped Morphia object in MongoDB
     * 
     * @param object The object to delete
     */
    public void delete(Object object) {
        Preconditions.checkNotNull(object);
        
        this.datastore.delete(object);
    }
    
    /**
     * Deletes all mapped Morphia objects of a given class
     * 
     * @param clazz The mapped Morphia class
     */
    public <T extends Object> void deleteAll(Class<T> clazz) {
        this.datastore.delete(this.datastore.createQuery(clazz));
    }
    
    /**
     * Drops all data in MongoDB on the configured database in 
     * Ninja Framework application.conf
     */
    public void dropDatabase() {
        this.datastore.getDB().dropDatabase();
    }
}