package de.svenkubiak.ninja.mongodb;

import java.net.UnknownHostException;
import java.util.Arrays;
import java.util.List;

import ninja.utils.NinjaProperties;

import org.apache.commons.lang3.StringUtils;
import org.bson.types.ObjectId;
import org.mongodb.morphia.Datastore;
import org.mongodb.morphia.Morphia;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.google.common.base.Preconditions;
import com.google.inject.Inject;
import com.google.inject.Singleton;
import com.mongodb.MongoClient;
import com.mongodb.MongoCredential;
import com.mongodb.ServerAddress;

/**
 * Convenient class for interacting with MongoDB and/or Morphia 
 * in the Ninja Web Framework
 * 
 * @author svenkubiak
 *
 */
@Singleton
public class MongoDB {
    private static final Logger LOG = LoggerFactory.getLogger(MongoDB.class);
    private static final String DEFAULT_MORPHIA_PACKAGE = "MyMorphiaPackage";
    private static final String DEFAULT_MONGODB_NAME = "MyMongoDB";
    private static final String DEFAULT_MONGODB_HOST = "localhost";
    private static final String MONGODB_HOST = "ninja.mongodb.host";
    private static final String MONGODB_PORT = "ninja.mongodb.port";
    private static final String MONGODB_USER = "ninja.mongodb.user";
    private static final String MONGODB_PASS = "ninja.mongodb.pass";
    private static final String MONGODB_DBNAME = "ninja.mongodb.dbname";
    private static final String MONGODB_AUTHDB = "ninja.mongodb.authdb";
    private static final String MORPHIA_PACKAGE = "ninja.morphia.package";
    private static final String MORPHIA_INIT = "ninja.morphia.init";
    private static final int DEFAULT_MONGODB_PORT = 27017;
    private Datastore datastore;
    private Morphia morphia;
    private MongoClient mongoClient;
    private NinjaProperties ninjaProperties;
    
    @Inject
    public MongoDB(NinjaProperties ninjaProperties) {
        this.ninjaProperties = ninjaProperties;
    }

    /**
     * Creates a Morphia Datastore if none is present
     * 
     * @return Morphia Datastore object
     */
    public Datastore getDatastore() {
        if (this.mongoClient == null) {
            initMongoClient();
        }
        
        if (this.datastore == null) {
            initMorphia();
        }

        return this.datastore;
    }
    
    /**
     * Creates Morphia instance if none is present
     * 
     * @return Morphia instance object
     */
    public Morphia getMorphia() {
        if (this.mongoClient == null) {
            initMongoClient();
        }
        
        if (this.morphia == null) {
            initMorphia();
        }
        
        return this.morphia;
    }

    /**
     * Returns the MongoClient instance
     * 
     * @return MongoClient object
     */
    public MongoClient getMongoClient() {
        if (this.mongoClient == null) {
            initMongoClient();
        }
        
        return this.mongoClient;
    }
    
    /**
     * Convenient method for overwriting the Morphia
     * object with a given MongoClient
     * 
     * @param mongoClient MongoClient object
     */
    public void setMongoClient(MongoClient mongoClient) {
        Preconditions.checkNotNull(mongoClient);

        this.mongoClient = mongoClient;
        if (this.mongoClient.getAddress() != null) {
            LOG.info("Successfully set MongoClient @ " + this.mongoClient.getAddress().getHost() + ":" + mongoClient.getAddress().getPort());            
        }
        
        if (this.morphia == null) {
            initMorphia();
        }
    }
    
    /**
     * Sets up a Mongo client object with or withour credentals
     *  
     * @param ninjaProperties
     * @return MongoClient object
     * @throws UnknownHostException if the given host is not found
     */
    private MongoClient initMongoClient() {
        String host = this.ninjaProperties.getWithDefault(MONGODB_HOST, DEFAULT_MONGODB_HOST);
        int port = this.ninjaProperties.getIntegerWithDefault(MONGODB_PORT, DEFAULT_MONGODB_PORT);
        
        String username = ninjaProperties.getWithDefault(MONGODB_USER, null);
        String password = ninjaProperties.getWithDefault(MONGODB_PASS, null);

        MongoClient client = null;
        try {
            if (StringUtils.isNotBlank(username) && StringUtils.isNotBlank(password)) {
                final String authdb = ninjaProperties.getWithDefault(MONGODB_AUTHDB, DEFAULT_MONGODB_NAME);
                
                MongoCredential credential = MongoCredential.createCredential(username, authdb, password.toCharArray());
                ServerAddress server = new ServerAddress(host, port);
                
                client = new MongoClient(server, Arrays.asList(credential));
                LOG.info("Successfully created MongoClient @ {}:{} with authentication {}/*********", host, port, username);
            } else {
                client = new MongoClient(host, port);
                LOG.info("Successfully created MongoClient @ {}:{}",host, port);
            }
        } catch (UnknownHostException e) {
            LOG.error("Failed to create MongoClient", e);
        }
        
        return client;
    }
    

    /**
     * Initializes the Morphia instance by setting the models with
     * packages and creating the datastore
     * 
     * @param mongoClient The MongoClient with the MongoDB connection
     */
    private void initMorphia() {
        if (this.ninjaProperties.getBooleanWithDefault(MORPHIA_INIT, false)) {
            String packageName = ninjaProperties.getWithDefault(MORPHIA_PACKAGE, DEFAULT_MORPHIA_PACKAGE);
            String dbName = ninjaProperties.getWithDefault(MONGODB_DBNAME, DEFAULT_MONGODB_NAME);
            
            this.morphia = new Morphia().mapPackage(packageName);
            this.datastore = this.morphia.createDatastore(mongoClient, dbName);
            
            LOG.info("Mapped Morphia to package '" + packageName + "' and created Morphia Datastore to database '" + dbName + "'");  
        }
    }
    
    /**
     * Retrieves a mapped Morphia object from MongoDB. If the id is not of 
     * type ObjectId, it will we converted to ObjectId
     * 
     * @param id The id of the object
     * @param clazz The mapped Morphia class
     * 
     * @return The requested class from MongoDB or null if none found
     */
    public <T extends Object> T findById(Object id, Class<T> clazz) {
        Preconditions.checkNotNull(clazz, "Tryed to find an object by id, but given class is null");
        Preconditions.checkNotNull(id, "Tryed to find an object by id, but given id is null");

        return this.datastore.get(clazz, (id instanceof ObjectId) ? id : new ObjectId(String.valueOf(id)));  
    }
    
    /**
     * Retrieves all mapped Morphia objects from MongoDB
     * 
     * @param clazz The mapped Morphia class
     * @return A list of mapped Morphia objects or an empty list of none found
     */
    public <T extends Object> List<T> findAll(Class<T> clazz) {
        Preconditions.checkNotNull(clazz, "Tryed to get all morphia objects of a given object, but given object is null");
        
        return this.datastore.find(clazz).asList();
    }
    
    /**
     * Counts all objected of a mapped Morphia class
     * 
     * @param clazz The mapped Morphia class
     * @return The number of objects in MongoDB
     */
    public <T extends Object> long countAll(Class<T> clazz) {
        Preconditions.checkNotNull(clazz, "Tryed to count all an morphia objects of a given object, but given object is null");
        
        return this.datastore.find(clazz).countAll();
    }
    
    /**
     * Saves a mapped Morphia object to MongoDB
     * 
     * @param object The object to save
     */
    public void save(Object object) {
        Preconditions.checkNotNull(object, "Tryed to save an morphia object, but a given object is null");
        
        this.datastore.save(object);
    }
    
    /**
     * Deletes a mapped Morphia object in MongoDB
     * 
     * @param object The object to delete
     */
    public void delete(Object object) {
        Preconditions.checkNotNull(object, "Tryed to delete an morphia object, but given object is null");
        
        this.datastore.delete(object);
    }
    
    /**
     * Deletes all mapped Morphia objects of a given class
     * 
     * @param clazz The mapped Morphia class
     */
    public <T extends Object> void deleteAll(Class<T> clazz) {
        this.datastore.delete(this.datastore.createQuery(clazz));
    }
    
    /**
     * Drops all data in MongoDB on the configured database in 
     * Ninja Framework application.conf
     */
    public void dropDatabase() {
        this.datastore.getDB().dropDatabase();
    }
}