/*
 * Decompiled with CFR 0.152.
 */
package de.vinado.spring.mail.javamail.concurrent;

import de.vinado.spring.mail.javamail.JavaMailSenderDecorator;
import de.vinado.spring.mail.javamail.concurrent.Batch;
import de.vinado.spring.mail.javamail.concurrent.BatchConsumer;
import de.vinado.spring.mail.javamail.concurrent.MailQueueException;
import java.time.Instant;
import java.time.ZoneId;
import java.util.Arrays;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.DelayQueue;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.atomic.AtomicLong;
import java.util.stream.IntStream;
import java.util.stream.Stream;
import javax.mail.internet.MimeMessage;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.mail.MailException;
import org.springframework.mail.SimpleMailMessage;
import org.springframework.mail.javamail.JavaMailSender;
import org.springframework.mail.javamail.MimeMessagePreparator;
import org.springframework.util.Assert;

public class ConcurrentJavaMailSender
extends JavaMailSenderDecorator
implements JavaMailSender {
    private static final Logger log = LoggerFactory.getLogger(ConcurrentJavaMailSender.class);
    private int batchSize = 20;
    private int cooldownMillis = 20000;
    private final BlockingQueue<Batch> queue = new DelayQueue<Batch>();
    private final AtomicLong delay = new AtomicLong(System.currentTimeMillis());

    ConcurrentJavaMailSender(JavaMailSender delegate, ExecutorService threadPool) {
        super(delegate);
        this.startConsumer(threadPool, delegate);
    }

    public void setBatchSize(int batchSize) {
        Assert.isTrue((batchSize > 0 ? 1 : 0) != 0, (String)"Batch size must be a positive integer");
        this.batchSize = batchSize;
    }

    private void startConsumer(ExecutorService threadPool, JavaMailSender delegate) {
        BatchConsumer consumer = new BatchConsumer(this.queue, delegate);
        threadPool.execute(consumer);
    }

    @Override
    public void send(MimeMessage mimeMessage) throws MailException {
        this.send(new MimeMessage[]{mimeMessage});
    }

    @Override
    public void send(MimeMessage ... mimeMessages) throws MailException {
        this.doSend(mimeMessages);
    }

    @Override
    public void send(MimeMessagePreparator mimeMessagePreparator) throws MailException {
        this.send(new MimeMessagePreparator[]{mimeMessagePreparator});
    }

    @Override
    public void send(MimeMessagePreparator ... mimeMessagePreparators) throws MailException {
        this.doSend(mimeMessagePreparators);
    }

    @Override
    public void send(SimpleMailMessage simpleMessage) throws MailException {
        this.send(new SimpleMailMessage[]{simpleMessage});
    }

    @Override
    public void send(SimpleMailMessage ... simpleMessages) throws MailException {
        this.doSend(simpleMessages);
    }

    private void doSend(Object[] messages) throws MailException {
        this.resetDelayIfQueueIsEmpty();
        Batch[] batches = this.createBatches(messages);
        this.enqueue(batches);
    }

    private void resetDelayIfQueueIsEmpty() {
        long time = System.currentTimeMillis();
        int queueSize = this.queue.size();
        long delay = this.delay.get();
        if (log.isTraceEnabled()) {
            log.trace("Queue length is: {}", (Object)queueSize);
            log.trace("Delay is at: {}", (Object)Instant.ofEpochMilli(delay).atZone(ZoneId.systemDefault()).toLocalDateTime());
            log.trace("Cooldown expires in: {}", (Object)(delay + (long)this.cooldownMillis - time));
        }
        if (queueSize == 0 && delay + (long)this.cooldownMillis < time) {
            if (log.isTraceEnabled()) {
                log.trace("Resetting delay");
            }
            this.delay.set(time);
        }
    }

    private Batch[] createBatches(Object[] original) {
        return (Batch[])ConcurrentJavaMailSender.partition(original, this.batchSize).map(messages -> {
            long time = this.delay.getAndAdd(this.cooldownMillis);
            return new Batch(time, (Object[])messages);
        }).toArray(Batch[]::new);
    }

    private static <T> Stream<T[]> partition(T[] original, int batchSize) {
        return IntStream.iterate(0, i -> i + batchSize).limit((long)Math.ceil((double)original.length / (double)batchSize)).mapToObj(i -> Arrays.copyOfRange(original, i, Math.min(i + batchSize, original.length)));
    }

    private void enqueue(Batch[] batches) throws MailException {
        for (Batch batch : batches) {
            try {
                if (log.isDebugEnabled()) {
                    log.debug("Enqueueing {}", (Object)batch);
                }
                this.queue.put(batch);
            }
            catch (InterruptedException e) {
                if (log.isErrorEnabled()) {
                    log.error("An error occurred while waiting for {} to be enqueued.", (Object)batch);
                }
                throw new MailQueueException("Could not enqueue email batch", e);
            }
        }
    }

    public int getBatchSize() {
        return this.batchSize;
    }

    public int getCooldownMillis() {
        return this.cooldownMillis;
    }

    public void setCooldownMillis(int cooldownMillis) {
        this.cooldownMillis = cooldownMillis;
    }
}

