/*
 * Copyright 2019 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.gradle.performance

import groovy.transform.CompileStatic
import org.gradle.integtests.fixtures.executer.IntegrationTestBuildContext
import org.gradle.performance.annotations.AllFeaturesShouldBeAnnotated
import org.gradle.performance.fixture.BuildExperimentSpec
import org.gradle.performance.fixture.CrossBuildPerformanceTestRunner
import org.gradle.performance.fixture.GradleBuildExperimentRunner
import org.gradle.performance.fixture.GradleBuildExperimentSpec
import org.gradle.performance.fixture.PerformanceTestDirectoryProvider
import org.gradle.performance.fixture.PerformanceTestIdProvider
import org.gradle.performance.results.CrossBuildPerformanceResults
import org.gradle.performance.results.CrossBuildResultsStore
import org.gradle.performance.results.GradleProfilerReporter
import org.gradle.performance.results.WritableResultsStore
import org.gradle.test.fixtures.file.CleanupTestDirectory
import org.gradle.test.fixtures.file.TestNameTestDirectoryProvider
import org.junit.Rule
import spock.lang.Specification

import static org.gradle.performance.results.ResultsStoreHelper.createResultsStoreWhenDatabaseAvailable

@CompileStatic
@CleanupTestDirectory
@AllFeaturesShouldBeAnnotated
class AbstractCrossBuildPerformanceTest extends Specification {
    private static final WritableResultsStore<CrossBuildPerformanceResults> RESULTS_STORE = createResultsStoreWhenDatabaseAvailable { new CrossBuildResultsStore() }

    protected final IntegrationTestBuildContext buildContext = new IntegrationTestBuildContext()

    @Rule
    TestNameTestDirectoryProvider temporaryFolder = new PerformanceTestDirectoryProvider(getClass())

    @Rule
    PerformanceTestIdProvider performanceTestIdProvider = new PerformanceTestIdProvider()

    CrossBuildPerformanceTestRunner runner

    def setup() {
        def gradleProfilerReporter = new GradleProfilerReporter(temporaryFolder.testDirectory)
        runner = new CrossBuildPerformanceTestRunner(
                new GradleBuildExperimentRunner(gradleProfilerReporter),
                RESULTS_STORE.reportAlso(gradleProfilerReporter),
                buildContext
        ) {
            @Override
            protected void defaultSpec(BuildExperimentSpec.Builder builder) {
                super.defaultSpec(builder)
                builder.workingDirectory = temporaryFolder.testDirectory
            }

            @Override
            protected void configureGradleSpec(GradleBuildExperimentSpec.GradleBuilder builder) {
                super.configureGradleSpec(builder)
                AbstractCrossBuildPerformanceTest.this.defaultSpec(builder)
            }

            @Override
            protected void finalizeGradleSpec(GradleBuildExperimentSpec.GradleBuilder builder) {
                super.finalizeGradleSpec(builder)
                AbstractCrossBuildPerformanceTest.this.finalizeSpec(builder)
            }
        }
        performanceTestIdProvider.setTestSpec(runner)
    }

    protected void defaultSpec(GradleBuildExperimentSpec.GradleBuilder builder) {
    }

    protected void finalizeSpec(GradleBuildExperimentSpec.GradleBuilder builder) {
    }

    static {
        // TODO - find a better way to cleanup
        System.addShutdownHook {
            ((Closeable) RESULTS_STORE).close()
        }
    }
}
