/*
 * Copyright 2020 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.gradle.smoketests

import org.gradle.integtests.fixtures.ToBeFixedForConfigurationCache
import org.gradle.integtests.fixtures.UnsupportedWithConfigurationCache
import org.gradle.testkit.runner.BuildResult
import org.gradle.util.Requires
import org.gradle.util.TestPrecondition
import spock.lang.Unroll

import static org.gradle.testkit.runner.TaskOutcome.FROM_CACHE
import static org.gradle.testkit.runner.TaskOutcome.NO_SOURCE
import static org.gradle.testkit.runner.TaskOutcome.SUCCESS
import static org.gradle.testkit.runner.TaskOutcome.UP_TO_DATE
import static org.hamcrest.CoreMatchers.equalTo
import static org.hamcrest.CoreMatchers.not
import static org.junit.Assume.assumeThat


@Requires(TestPrecondition.JDK11_OR_EARLIER)
class AndroidSantaTrackerKotlinCachingSmokeTest extends AbstractAndroidSantaTrackerSmokeTest {

    @Unroll
    @UnsupportedWithConfigurationCache(iterationMatchers = [AGP_3_ITERATION_MATCHER, AGP_4_0_ITERATION_MATCHER])
    @ToBeFixedForConfigurationCache(iterationMatchers = [AGP_4_1_ITERATION_MATCHER, AGP_4_2_ITERATION_MATCHER])
    def "can cache Santa Tracker Kotlin Android application (agp=#agpVersion)"() {

        // TODO remove after next 4.2 release
        assumeThat(agpVersion, not(equalTo("4.2.0-alpha13")))
        given:
        def originalDir = temporaryFolder.createDir("original")
        def relocatedDir = temporaryFolder.createDir("relocated")

        and:
        setupCopyOfSantaTracker(originalDir, 'Kotlin', agpVersion)
        setupCopyOfSantaTracker(relocatedDir, 'Kotlin', agpVersion)

        when: 'clean build'
        buildLocation(originalDir, agpVersion)

        then:
        assertConfigurationCacheStateStored()

        when: 'up-to-date build, reusing configuration cache when enabled'
        buildLocation(originalDir, agpVersion)

        then:
        assertConfigurationCacheStateLoaded()

        when: 'clean cached build'
        BuildResult relocatedResult = buildLocation(relocatedDir, agpVersion)

        then:
        assertConfigurationCacheStateStored()

        and:
        def expectedResults = agpVersion.startsWith('3.6')
            ? EXPECTED_RESULTS_3_6
            : agpVersion.startsWith('4.0')
            ? EXPECTED_RESULTS_4_0
            : agpVersion.startsWith('4.1')
            ? EXPECTED_RESULTS_4_1
            : EXPECTED_RESULTS_4_2
        verify(relocatedResult, expectedResults)

        when: 'clean cached build, reusing configuration cache when enabled'
        cleanLocation(relocatedDir, agpVersion)
        buildLocation(relocatedDir, agpVersion)

        then:
        assertConfigurationCacheStateLoaded()

        where:
        agpVersion << TESTED_AGP_VERSIONS
    }

    private static final EXPECTED_RESULTS_4_2 = [
        ':cityquiz:assembleDebug': SUCCESS,
        ':cityquiz:checkDebugAarMetadata': FROM_CACHE,
        ':cityquiz:checkDebugDuplicateClasses': FROM_CACHE,
        ':cityquiz:compileDebugAidl': NO_SOURCE,
        ':cityquiz:compileDebugJavaWithJavac': FROM_CACHE,
        ':cityquiz:compileDebugKotlin': FROM_CACHE,
        ':cityquiz:compileDebugRenderscript': NO_SOURCE,
        ':cityquiz:compileDebugShaders': NO_SOURCE,
        ':cityquiz:compileDebugSources': UP_TO_DATE,
        ':cityquiz:compressDebugAssets': FROM_CACHE,
        ':cityquiz:createDebugCompatibleScreenManifests': FROM_CACHE,
        ':cityquiz:desugarDebugFileDependencies': SUCCESS,
        ':cityquiz:dexBuilderDebug': FROM_CACHE,
        ':cityquiz:extractDeepLinksDebug': FROM_CACHE,
        ':cityquiz:featureDebugWriter': SUCCESS,
        ':cityquiz:generateDebugAssets': UP_TO_DATE,
        ':cityquiz:generateDebugBuildConfig': FROM_CACHE,
        ':cityquiz:generateDebugFeatureTransitiveDeps': FROM_CACHE,
        ':cityquiz:generateDebugResValues': FROM_CACHE,
        ':cityquiz:generateDebugResources': UP_TO_DATE,
        ':cityquiz:javaPreCompileDebug': FROM_CACHE,
        ':cityquiz:mergeDebugAssets': FROM_CACHE,
        ':cityquiz:mergeDebugJavaResource': FROM_CACHE,
        ':cityquiz:mergeDebugJniLibFolders': FROM_CACHE,
        ':cityquiz:mergeDebugNativeLibs': NO_SOURCE,
        ':cityquiz:mergeDebugResources': SUCCESS,
        ':cityquiz:mergeDebugShaders': FROM_CACHE,
        ':cityquiz:mergeExtDexDebug': FROM_CACHE,
        ':cityquiz:mergeLibDexDebug': FROM_CACHE,
        ':cityquiz:mergeProjectDexDebug': FROM_CACHE,
        ':cityquiz:packageDebug': SUCCESS,
        ':cityquiz:preBuild': UP_TO_DATE,
        ':cityquiz:preDebugBuild': UP_TO_DATE,
        ':cityquiz:processApplicationManifestDebugForBundle': FROM_CACHE,
        ':cityquiz:processDebugJavaRes': NO_SOURCE,
        ':cityquiz:processDebugMainManifest': FROM_CACHE,
        ':cityquiz:processDebugManifest': FROM_CACHE,
        ':cityquiz:processDebugManifestForPackage': FROM_CACHE,
        ':cityquiz:processDebugResources': SUCCESS,
        ':cityquiz:processManifestDebugForFeature': FROM_CACHE,
        ':cityquiz:stripDebugDebugSymbols': NO_SOURCE,
        ':common:assembleDebug': SUCCESS,
        ':common:bundleDebugAar': SUCCESS,
        ':common:bundleLibCompileToJarDebug': FROM_CACHE,
        ':common:bundleLibResDebug': FROM_CACHE,
        ':common:bundleLibRuntimeToDirDebug': FROM_CACHE,
        ':common:bundleLibRuntimeToJarDebug': FROM_CACHE,
        ':common:compileDebugAidl': NO_SOURCE,
        ':common:compileDebugJavaWithJavac': FROM_CACHE,
        ':common:compileDebugKotlin': FROM_CACHE,
        ':common:compileDebugLibraryResources': SUCCESS,
        ':common:compileDebugRenderscript': NO_SOURCE,
        ':common:compileDebugShaders': NO_SOURCE,
        ':common:compileDebugSources': UP_TO_DATE,
        ':common:copyDebugJniLibsProjectAndLocalJars': FROM_CACHE,
        ':common:copyDebugJniLibsProjectOnly': FROM_CACHE,
        ':common:createFullJarDebug': FROM_CACHE,
        ':common:extractDebugAnnotations': FROM_CACHE,
        ':common:extractDeepLinksDebug': FROM_CACHE,
        ':common:generateDebugAssets': UP_TO_DATE,
        ':common:generateDebugBuildConfig': FROM_CACHE,
        ':common:generateDebugRFile': FROM_CACHE,
        ':common:generateDebugResValues': FROM_CACHE,
        ':common:generateDebugResources': UP_TO_DATE,
        ':common:javaPreCompileDebug': FROM_CACHE,
        ':common:mergeDebugConsumerProguardFiles': FROM_CACHE,
        ':common:mergeDebugGeneratedProguardFiles': FROM_CACHE,
        ':common:mergeDebugJavaResource': FROM_CACHE,
        ':common:mergeDebugJniLibFolders': FROM_CACHE,
        ':common:mergeDebugNativeLibs': NO_SOURCE,
        ':common:mergeDebugShaders': FROM_CACHE,
        ':common:packageDebugAssets': FROM_CACHE,
        ':common:packageDebugRenderscript': NO_SOURCE,
        ':common:packageDebugResources': FROM_CACHE,
        ':common:parseDebugLocalResources': FROM_CACHE,
        ':common:preBuild': UP_TO_DATE,
        ':common:preDebugBuild': UP_TO_DATE,
        ':common:prepareLintJarForPublish': SUCCESS,
        ':common:processDebugJavaRes': NO_SOURCE,
        ':common:processDebugManifest': FROM_CACHE,
        ':common:stripDebugDebugSymbols': NO_SOURCE,
        ':common:syncDebugLibJars': FROM_CACHE,
        ':common:writeDebugAarMetadata': FROM_CACHE,
        ':dasherdancer:assembleDebug': SUCCESS,
        ':dasherdancer:checkDebugAarMetadata': FROM_CACHE,
        ':dasherdancer:checkDebugDuplicateClasses': FROM_CACHE,
        ':dasherdancer:compileDebugAidl': NO_SOURCE,
        ':dasherdancer:compileDebugJavaWithJavac': FROM_CACHE,
        ':dasherdancer:compileDebugKotlin': FROM_CACHE,
        ':dasherdancer:compileDebugRenderscript': NO_SOURCE,
        ':dasherdancer:compileDebugShaders': NO_SOURCE,
        ':dasherdancer:compileDebugSources': UP_TO_DATE,
        ':dasherdancer:compressDebugAssets': FROM_CACHE,
        ':dasherdancer:createDebugCompatibleScreenManifests': FROM_CACHE,
        ':dasherdancer:desugarDebugFileDependencies': SUCCESS,
        ':dasherdancer:dexBuilderDebug': FROM_CACHE,
        ':dasherdancer:extractDeepLinksDebug': FROM_CACHE,
        ':dasherdancer:featureDebugWriter': SUCCESS,
        ':dasherdancer:generateDebugAssets': UP_TO_DATE,
        ':dasherdancer:generateDebugBuildConfig': FROM_CACHE,
        ':dasherdancer:generateDebugFeatureTransitiveDeps': FROM_CACHE,
        ':dasherdancer:generateDebugResValues': FROM_CACHE,
        ':dasherdancer:generateDebugResources': UP_TO_DATE,
        ':dasherdancer:javaPreCompileDebug': FROM_CACHE,
        ':dasherdancer:mergeDebugAssets': FROM_CACHE,
        ':dasherdancer:mergeDebugJavaResource': FROM_CACHE,
        ':dasherdancer:mergeDebugJniLibFolders': FROM_CACHE,
        ':dasherdancer:mergeDebugNativeLibs': NO_SOURCE,
        ':dasherdancer:mergeDebugResources': SUCCESS,
        ':dasherdancer:mergeDebugShaders': FROM_CACHE,
        ':dasherdancer:mergeExtDexDebug': FROM_CACHE,
        ':dasherdancer:mergeLibDexDebug': FROM_CACHE,
        ':dasherdancer:mergeProjectDexDebug': FROM_CACHE,
        ':dasherdancer:packageDebug': SUCCESS,
        ':dasherdancer:preBuild': UP_TO_DATE,
        ':dasherdancer:preDebugBuild': UP_TO_DATE,
        ':dasherdancer:processApplicationManifestDebugForBundle': FROM_CACHE,
        ':dasherdancer:processDebugJavaRes': NO_SOURCE,
        ':dasherdancer:processDebugMainManifest': FROM_CACHE,
        ':dasherdancer:processDebugManifest': FROM_CACHE,
        ':dasherdancer:processDebugManifestForPackage': FROM_CACHE,
        ':dasherdancer:processDebugResources': SUCCESS,
        ':dasherdancer:processManifestDebugForFeature': FROM_CACHE,
        ':dasherdancer:stripDebugDebugSymbols': NO_SOURCE,
        ':doodles-lib:assembleDebug': SUCCESS,
        ':doodles-lib:bundleDebugAar': SUCCESS,
        ':doodles-lib:bundleLibCompileToJarDebug': FROM_CACHE,
        ':doodles-lib:bundleLibResDebug': NO_SOURCE,
        ':doodles-lib:bundleLibRuntimeToDirDebug': FROM_CACHE,
        ':doodles-lib:bundleLibRuntimeToJarDebug': FROM_CACHE,
        ':doodles-lib:compileDebugAidl': NO_SOURCE,
        ':doodles-lib:compileDebugJavaWithJavac': FROM_CACHE,
        ':doodles-lib:compileDebugLibraryResources': SUCCESS,
        ':doodles-lib:compileDebugRenderscript': NO_SOURCE,
        ':doodles-lib:compileDebugShaders': NO_SOURCE,
        ':doodles-lib:compileDebugSources': UP_TO_DATE,
        ':doodles-lib:copyDebugJniLibsProjectAndLocalJars': FROM_CACHE,
        ':doodles-lib:copyDebugJniLibsProjectOnly': FROM_CACHE,
        ':doodles-lib:createFullJarDebug': FROM_CACHE,
        ':doodles-lib:extractDebugAnnotations': FROM_CACHE,
        ':doodles-lib:extractDeepLinksDebug': FROM_CACHE,
        ':doodles-lib:generateDebugAssets': UP_TO_DATE,
        ':doodles-lib:generateDebugBuildConfig': FROM_CACHE,
        ':doodles-lib:generateDebugRFile': FROM_CACHE,
        ':doodles-lib:generateDebugResValues': FROM_CACHE,
        ':doodles-lib:generateDebugResources': UP_TO_DATE,
        ':doodles-lib:javaPreCompileDebug': FROM_CACHE,
        ':doodles-lib:mergeDebugConsumerProguardFiles': FROM_CACHE,
        ':doodles-lib:mergeDebugGeneratedProguardFiles': FROM_CACHE,
        ':doodles-lib:mergeDebugJavaResource': FROM_CACHE,
        ':doodles-lib:mergeDebugJniLibFolders': FROM_CACHE,
        ':doodles-lib:mergeDebugNativeLibs': NO_SOURCE,
        ':doodles-lib:mergeDebugShaders': FROM_CACHE,
        ':doodles-lib:packageDebugAssets': FROM_CACHE,
        ':doodles-lib:packageDebugRenderscript': NO_SOURCE,
        ':doodles-lib:packageDebugResources': FROM_CACHE,
        ':doodles-lib:parseDebugLocalResources': FROM_CACHE,
        ':doodles-lib:preBuild': UP_TO_DATE,
        ':doodles-lib:preDebugBuild': UP_TO_DATE,
        ':doodles-lib:prepareLintJarForPublish': SUCCESS,
        ':doodles-lib:processDebugJavaRes': NO_SOURCE,
        ':doodles-lib:processDebugManifest': FROM_CACHE,
        ':doodles-lib:stripDebugDebugSymbols': NO_SOURCE,
        ':doodles-lib:syncDebugLibJars': FROM_CACHE,
        ':doodles-lib:writeDebugAarMetadata': FROM_CACHE,
        ':gumball:assembleDebug': SUCCESS,
        ':gumball:checkDebugAarMetadata': FROM_CACHE,
        ':gumball:checkDebugDuplicateClasses': FROM_CACHE,
        ':gumball:compileDebugAidl': NO_SOURCE,
        ':gumball:compileDebugJavaWithJavac': FROM_CACHE,
        ':gumball:compileDebugRenderscript': NO_SOURCE,
        ':gumball:compileDebugShaders': NO_SOURCE,
        ':gumball:compileDebugSources': UP_TO_DATE,
        ':gumball:compressDebugAssets': FROM_CACHE,
        ':gumball:createDebugCompatibleScreenManifests': FROM_CACHE,
        ':gumball:desugarDebugFileDependencies': SUCCESS,
        ':gumball:dexBuilderDebug': FROM_CACHE,
        ':gumball:extractDeepLinksDebug': FROM_CACHE,
        ':gumball:featureDebugWriter': SUCCESS,
        ':gumball:generateDebugAssets': UP_TO_DATE,
        ':gumball:generateDebugBuildConfig': FROM_CACHE,
        ':gumball:generateDebugFeatureTransitiveDeps': FROM_CACHE,
        ':gumball:generateDebugResValues': FROM_CACHE,
        ':gumball:generateDebugResources': UP_TO_DATE,
        ':gumball:javaPreCompileDebug': FROM_CACHE,
        ':gumball:mergeDebugAssets': FROM_CACHE,
        ':gumball:mergeDebugJavaResource': FROM_CACHE,
        ':gumball:mergeDebugJniLibFolders': FROM_CACHE,
        ':gumball:mergeDebugNativeLibs': FROM_CACHE,
        ':gumball:mergeDebugResources': SUCCESS,
        ':gumball:mergeDebugShaders': FROM_CACHE,
        ':gumball:mergeExtDexDebug': FROM_CACHE,
        ':gumball:mergeLibDexDebug': FROM_CACHE,
        ':gumball:mergeProjectDexDebug': FROM_CACHE,
        ':gumball:packageDebug': SUCCESS,
        ':gumball:preBuild': UP_TO_DATE,
        ':gumball:preDebugBuild': UP_TO_DATE,
        ':gumball:processApplicationManifestDebugForBundle': FROM_CACHE,
        ':gumball:processDebugJavaRes': NO_SOURCE,
        ':gumball:processDebugMainManifest': FROM_CACHE,
        ':gumball:processDebugManifest': FROM_CACHE,
        ':gumball:processDebugManifestForPackage': FROM_CACHE,
        ':gumball:processDebugResources': SUCCESS,
        ':gumball:processManifestDebugForFeature': FROM_CACHE,
        ':gumball:stripDebugDebugSymbols': NO_SOURCE,
        ':jetpack:assembleDebug': SUCCESS,
        ':jetpack:checkDebugAarMetadata': FROM_CACHE,
        ':jetpack:checkDebugDuplicateClasses': FROM_CACHE,
        ':jetpack:compileDebugAidl': NO_SOURCE,
        ':jetpack:compileDebugJavaWithJavac': FROM_CACHE,
        ':jetpack:compileDebugKotlin': FROM_CACHE,
        ':jetpack:compileDebugRenderscript': NO_SOURCE,
        ':jetpack:compileDebugShaders': NO_SOURCE,
        ':jetpack:compileDebugSources': UP_TO_DATE,
        ':jetpack:compressDebugAssets': FROM_CACHE,
        ':jetpack:createDebugCompatibleScreenManifests': FROM_CACHE,
        ':jetpack:desugarDebugFileDependencies': SUCCESS,
        ':jetpack:dexBuilderDebug': FROM_CACHE,
        ':jetpack:extractDeepLinksDebug': FROM_CACHE,
        ':jetpack:featureDebugWriter': SUCCESS,
        ':jetpack:generateDebugAssets': UP_TO_DATE,
        ':jetpack:generateDebugBuildConfig': FROM_CACHE,
        ':jetpack:generateDebugFeatureTransitiveDeps': FROM_CACHE,
        ':jetpack:generateDebugResValues': FROM_CACHE,
        ':jetpack:generateDebugResources': UP_TO_DATE,
        ':jetpack:javaPreCompileDebug': FROM_CACHE,
        ':jetpack:mergeDebugAssets': FROM_CACHE,
        ':jetpack:mergeDebugJavaResource': FROM_CACHE,
        ':jetpack:mergeDebugJniLibFolders': FROM_CACHE,
        ':jetpack:mergeDebugNativeLibs': NO_SOURCE,
        ':jetpack:mergeDebugResources': SUCCESS,
        ':jetpack:mergeDebugShaders': FROM_CACHE,
        ':jetpack:mergeExtDexDebug': FROM_CACHE,
        ':jetpack:mergeLibDexDebug': FROM_CACHE,
        ':jetpack:mergeProjectDexDebug': FROM_CACHE,
        ':jetpack:packageDebug': SUCCESS,
        ':jetpack:preBuild': UP_TO_DATE,
        ':jetpack:preDebugBuild': UP_TO_DATE,
        ':jetpack:processApplicationManifestDebugForBundle': FROM_CACHE,
        ':jetpack:processDebugJavaRes': NO_SOURCE,
        ':jetpack:processDebugMainManifest': FROM_CACHE,
        ':jetpack:processDebugManifest': FROM_CACHE,
        ':jetpack:processDebugManifestForPackage': FROM_CACHE,
        ':jetpack:processDebugResources': SUCCESS,
        ':jetpack:processManifestDebugForFeature': FROM_CACHE,
        ':jetpack:stripDebugDebugSymbols': NO_SOURCE,
        ':memory:assembleDebug': SUCCESS,
        ':memory:checkDebugAarMetadata': FROM_CACHE,
        ':memory:checkDebugDuplicateClasses': FROM_CACHE,
        ':memory:compileDebugAidl': NO_SOURCE,
        ':memory:compileDebugJavaWithJavac': FROM_CACHE,
        ':memory:compileDebugRenderscript': NO_SOURCE,
        ':memory:compileDebugShaders': NO_SOURCE,
        ':memory:compileDebugSources': UP_TO_DATE,
        ':memory:compressDebugAssets': FROM_CACHE,
        ':memory:createDebugCompatibleScreenManifests': FROM_CACHE,
        ':memory:desugarDebugFileDependencies': SUCCESS,
        ':memory:dexBuilderDebug': FROM_CACHE,
        ':memory:extractDeepLinksDebug': FROM_CACHE,
        ':memory:featureDebugWriter': SUCCESS,
        ':memory:generateDebugAssets': UP_TO_DATE,
        ':memory:generateDebugBuildConfig': FROM_CACHE,
        ':memory:generateDebugFeatureTransitiveDeps': FROM_CACHE,
        ':memory:generateDebugResValues': FROM_CACHE,
        ':memory:generateDebugResources': UP_TO_DATE,
        ':memory:javaPreCompileDebug': FROM_CACHE,
        ':memory:mergeDebugAssets': FROM_CACHE,
        ':memory:mergeDebugJavaResource': FROM_CACHE,
        ':memory:mergeDebugJniLibFolders': FROM_CACHE,
        ':memory:mergeDebugNativeLibs': NO_SOURCE,
        ':memory:mergeDebugResources': SUCCESS,
        ':memory:mergeDebugShaders': FROM_CACHE,
        ':memory:mergeExtDexDebug': FROM_CACHE,
        ':memory:mergeLibDexDebug': FROM_CACHE,
        ':memory:mergeProjectDexDebug': FROM_CACHE,
        ':memory:packageDebug': SUCCESS,
        ':memory:preBuild': UP_TO_DATE,
        ':memory:preDebugBuild': UP_TO_DATE,
        ':memory:processApplicationManifestDebugForBundle': FROM_CACHE,
        ':memory:processDebugJavaRes': NO_SOURCE,
        ':memory:processDebugMainManifest': FROM_CACHE,
        ':memory:processDebugManifest': FROM_CACHE,
        ':memory:processDebugManifestForPackage': FROM_CACHE,
        ':memory:processDebugResources': SUCCESS,
        ':memory:processManifestDebugForFeature': FROM_CACHE,
        ':memory:stripDebugDebugSymbols': NO_SOURCE,
        ':penguinswim:assembleDebug': SUCCESS,
        ':penguinswim:checkDebugAarMetadata': FROM_CACHE,
        ':penguinswim:checkDebugDuplicateClasses': FROM_CACHE,
        ':penguinswim:compileDebugAidl': NO_SOURCE,
        ':penguinswim:compileDebugJavaWithJavac': FROM_CACHE,
        ':penguinswim:compileDebugRenderscript': NO_SOURCE,
        ':penguinswim:compileDebugShaders': NO_SOURCE,
        ':penguinswim:compileDebugSources': UP_TO_DATE,
        ':penguinswim:compressDebugAssets': FROM_CACHE,
        ':penguinswim:createDebugCompatibleScreenManifests': FROM_CACHE,
        ':penguinswim:desugarDebugFileDependencies': SUCCESS,
        ':penguinswim:dexBuilderDebug': FROM_CACHE,
        ':penguinswim:extractDeepLinksDebug': FROM_CACHE,
        ':penguinswim:featureDebugWriter': SUCCESS,
        ':penguinswim:generateDebugAssets': UP_TO_DATE,
        ':penguinswim:generateDebugBuildConfig': FROM_CACHE,
        ':penguinswim:generateDebugFeatureTransitiveDeps': FROM_CACHE,
        ':penguinswim:generateDebugResValues': FROM_CACHE,
        ':penguinswim:generateDebugResources': UP_TO_DATE,
        ':penguinswim:javaPreCompileDebug': FROM_CACHE,
        ':penguinswim:mergeDebugAssets': FROM_CACHE,
        ':penguinswim:mergeDebugJavaResource': FROM_CACHE,
        ':penguinswim:mergeDebugJniLibFolders': FROM_CACHE,
        ':penguinswim:mergeDebugNativeLibs': NO_SOURCE,
        ':penguinswim:mergeDebugResources': SUCCESS,
        ':penguinswim:mergeDebugShaders': FROM_CACHE,
        ':penguinswim:mergeExtDexDebug': FROM_CACHE,
        ':penguinswim:mergeLibDexDebug': FROM_CACHE,
        ':penguinswim:mergeProjectDexDebug': FROM_CACHE,
        ':penguinswim:packageDebug': SUCCESS,
        ':penguinswim:preBuild': UP_TO_DATE,
        ':penguinswim:preDebugBuild': UP_TO_DATE,
        ':penguinswim:processApplicationManifestDebugForBundle': FROM_CACHE,
        ':penguinswim:processDebugJavaRes': NO_SOURCE,
        ':penguinswim:processDebugMainManifest': FROM_CACHE,
        ':penguinswim:processDebugManifest': FROM_CACHE,
        ':penguinswim:processDebugManifestForPackage': FROM_CACHE,
        ':penguinswim:processDebugResources': SUCCESS,
        ':penguinswim:processManifestDebugForFeature': FROM_CACHE,
        ':penguinswim:stripDebugDebugSymbols': NO_SOURCE,
        ':playgames:assembleDebug': SUCCESS,
        ':playgames:bundleDebugAar': SUCCESS,
        ':playgames:bundleLibCompileToJarDebug': FROM_CACHE,
        ':playgames:bundleLibResDebug': NO_SOURCE,
        ':playgames:bundleLibRuntimeToDirDebug': FROM_CACHE,
        ':playgames:bundleLibRuntimeToJarDebug': FROM_CACHE,
        ':playgames:compileDebugAidl': NO_SOURCE,
        ':playgames:compileDebugJavaWithJavac': FROM_CACHE,
        ':playgames:compileDebugLibraryResources': SUCCESS,
        ':playgames:compileDebugRenderscript': NO_SOURCE,
        ':playgames:compileDebugShaders': NO_SOURCE,
        ':playgames:compileDebugSources': UP_TO_DATE,
        ':playgames:copyDebugJniLibsProjectAndLocalJars': FROM_CACHE,
        ':playgames:copyDebugJniLibsProjectOnly': FROM_CACHE,
        ':playgames:createFullJarDebug': FROM_CACHE,
        ':playgames:extractDebugAnnotations': FROM_CACHE,
        ':playgames:extractDeepLinksDebug': FROM_CACHE,
        ':playgames:generateDebugAssets': UP_TO_DATE,
        ':playgames:generateDebugBuildConfig': FROM_CACHE,
        ':playgames:generateDebugRFile': FROM_CACHE,
        ':playgames:generateDebugResValues': FROM_CACHE,
        ':playgames:generateDebugResources': UP_TO_DATE,
        ':playgames:javaPreCompileDebug': FROM_CACHE,
        ':playgames:mergeDebugConsumerProguardFiles': FROM_CACHE,
        ':playgames:mergeDebugGeneratedProguardFiles': FROM_CACHE,
        ':playgames:mergeDebugJavaResource': FROM_CACHE,
        ':playgames:mergeDebugJniLibFolders': FROM_CACHE,
        ':playgames:mergeDebugNativeLibs': NO_SOURCE,
        ':playgames:mergeDebugShaders': FROM_CACHE,
        ':playgames:packageDebugAssets': FROM_CACHE,
        ':playgames:packageDebugRenderscript': NO_SOURCE,
        ':playgames:packageDebugResources': FROM_CACHE,
        ':playgames:parseDebugLocalResources': FROM_CACHE,
        ':playgames:preBuild': UP_TO_DATE,
        ':playgames:preDebugBuild': UP_TO_DATE,
        ':playgames:prepareLintJarForPublish': SUCCESS,
        ':playgames:processDebugJavaRes': NO_SOURCE,
        ':playgames:processDebugManifest': FROM_CACHE,
        ':playgames:stripDebugDebugSymbols': NO_SOURCE,
        ':playgames:syncDebugLibJars': FROM_CACHE,
        ':playgames:writeDebugAarMetadata': FROM_CACHE,
        ':presenttoss:assembleDebug': SUCCESS,
        ':presenttoss:checkDebugAarMetadata': FROM_CACHE,
        ':presenttoss:checkDebugDuplicateClasses': FROM_CACHE,
        ':presenttoss:compileDebugAidl': NO_SOURCE,
        ':presenttoss:compileDebugJavaWithJavac': FROM_CACHE,
        ':presenttoss:compileDebugRenderscript': NO_SOURCE,
        ':presenttoss:compileDebugShaders': NO_SOURCE,
        ':presenttoss:compileDebugSources': UP_TO_DATE,
        ':presenttoss:compressDebugAssets': FROM_CACHE,
        ':presenttoss:createDebugCompatibleScreenManifests': FROM_CACHE,
        ':presenttoss:desugarDebugFileDependencies': SUCCESS,
        ':presenttoss:dexBuilderDebug': FROM_CACHE,
        ':presenttoss:extractDeepLinksDebug': FROM_CACHE,
        ':presenttoss:featureDebugWriter': SUCCESS,
        ':presenttoss:generateDebugAssets': UP_TO_DATE,
        ':presenttoss:generateDebugBuildConfig': FROM_CACHE,
        ':presenttoss:generateDebugFeatureTransitiveDeps': FROM_CACHE,
        ':presenttoss:generateDebugResValues': FROM_CACHE,
        ':presenttoss:generateDebugResources': UP_TO_DATE,
        ':presenttoss:javaPreCompileDebug': FROM_CACHE,
        ':presenttoss:mergeDebugAssets': FROM_CACHE,
        ':presenttoss:mergeDebugJavaResource': FROM_CACHE,
        ':presenttoss:mergeDebugJniLibFolders': FROM_CACHE,
        ':presenttoss:mergeDebugNativeLibs': NO_SOURCE,
        ':presenttoss:mergeDebugResources': SUCCESS,
        ':presenttoss:mergeDebugShaders': FROM_CACHE,
        ':presenttoss:mergeExtDexDebug': FROM_CACHE,
        ':presenttoss:mergeLibDexDebug': FROM_CACHE,
        ':presenttoss:mergeProjectDexDebug': FROM_CACHE,
        ':presenttoss:packageDebug': SUCCESS,
        ':presenttoss:preBuild': UP_TO_DATE,
        ':presenttoss:preDebugBuild': UP_TO_DATE,
        ':presenttoss:processApplicationManifestDebugForBundle': FROM_CACHE,
        ':presenttoss:processDebugJavaRes': NO_SOURCE,
        ':presenttoss:processDebugMainManifest': FROM_CACHE,
        ':presenttoss:processDebugManifest': FROM_CACHE,
        ':presenttoss:processDebugManifestForPackage': FROM_CACHE,
        ':presenttoss:processDebugResources': SUCCESS,
        ':presenttoss:processManifestDebugForFeature': FROM_CACHE,
        ':presenttoss:stripDebugDebugSymbols': NO_SOURCE,
        ':rocketsleigh:assembleDebug': SUCCESS,
        ':rocketsleigh:checkDebugAarMetadata': FROM_CACHE,
        ':rocketsleigh:checkDebugDuplicateClasses': FROM_CACHE,
        ':rocketsleigh:compileDebugAidl': NO_SOURCE,
        ':rocketsleigh:compileDebugJavaWithJavac': FROM_CACHE,
        ':rocketsleigh:compileDebugKotlin': FROM_CACHE,
        ':rocketsleigh:compileDebugRenderscript': NO_SOURCE,
        ':rocketsleigh:compileDebugShaders': NO_SOURCE,
        ':rocketsleigh:compileDebugSources': UP_TO_DATE,
        ':rocketsleigh:compressDebugAssets': FROM_CACHE,
        ':rocketsleigh:createDebugCompatibleScreenManifests': FROM_CACHE,
        ':rocketsleigh:desugarDebugFileDependencies': SUCCESS,
        ':rocketsleigh:dexBuilderDebug': FROM_CACHE,
        ':rocketsleigh:extractDeepLinksDebug': FROM_CACHE,
        ':rocketsleigh:featureDebugWriter': SUCCESS,
        ':rocketsleigh:generateDebugAssets': UP_TO_DATE,
        ':rocketsleigh:generateDebugBuildConfig': FROM_CACHE,
        ':rocketsleigh:generateDebugFeatureTransitiveDeps': FROM_CACHE,
        ':rocketsleigh:generateDebugResValues': FROM_CACHE,
        ':rocketsleigh:generateDebugResources': UP_TO_DATE,
        ':rocketsleigh:javaPreCompileDebug': FROM_CACHE,
        ':rocketsleigh:mergeDebugAssets': FROM_CACHE,
        ':rocketsleigh:mergeDebugJavaResource': FROM_CACHE,
        ':rocketsleigh:mergeDebugJniLibFolders': FROM_CACHE,
        ':rocketsleigh:mergeDebugNativeLibs': NO_SOURCE,
        ':rocketsleigh:mergeDebugResources': SUCCESS,
        ':rocketsleigh:mergeDebugShaders': FROM_CACHE,
        ':rocketsleigh:mergeExtDexDebug': FROM_CACHE,
        ':rocketsleigh:mergeLibDexDebug': FROM_CACHE,
        ':rocketsleigh:mergeProjectDexDebug': FROM_CACHE,
        ':rocketsleigh:packageDebug': SUCCESS,
        ':rocketsleigh:preBuild': UP_TO_DATE,
        ':rocketsleigh:preDebugBuild': UP_TO_DATE,
        ':rocketsleigh:processApplicationManifestDebugForBundle': FROM_CACHE,
        ':rocketsleigh:processDebugJavaRes': NO_SOURCE,
        ':rocketsleigh:processDebugMainManifest': FROM_CACHE,
        ':rocketsleigh:processDebugManifest': FROM_CACHE,
        ':rocketsleigh:processDebugManifestForPackage': FROM_CACHE,
        ':rocketsleigh:processDebugResources': SUCCESS,
        ':rocketsleigh:processManifestDebugForFeature': FROM_CACHE,
        ':rocketsleigh:stripDebugDebugSymbols': NO_SOURCE,
        ':santa-tracker:assembleDebug': SUCCESS,
        ':santa-tracker:bundleDebugClasses': FROM_CACHE,
        ':santa-tracker:checkDebugAarMetadata': FROM_CACHE,
        ':santa-tracker:checkDebugDuplicateClasses': FROM_CACHE,
        ':santa-tracker:checkDebugLibraries': FROM_CACHE,
        ':santa-tracker:compileDebugAidl': NO_SOURCE,
        ':santa-tracker:compileDebugJavaWithJavac': FROM_CACHE,
        ':santa-tracker:compileDebugKotlin': FROM_CACHE,
        ':santa-tracker:compileDebugRenderscript': NO_SOURCE,
        ':santa-tracker:compileDebugShaders': NO_SOURCE,
        ':santa-tracker:compileDebugSources': UP_TO_DATE,
        ':santa-tracker:compressDebugAssets': FROM_CACHE,
        ':santa-tracker:createDebugCompatibleScreenManifests': FROM_CACHE,
        ':santa-tracker:desugarDebugFileDependencies': SUCCESS,
        ':santa-tracker:dexBuilderDebug': FROM_CACHE,
        ':santa-tracker:extractDeepLinksDebug': FROM_CACHE,
        ':santa-tracker:generateDebugAssets': UP_TO_DATE,
        ':santa-tracker:generateDebugBuildConfig': FROM_CACHE,
        ':santa-tracker:generateDebugFeatureMetadata': FROM_CACHE,
        ':santa-tracker:generateDebugFeatureTransitiveDeps': FROM_CACHE,
        ':santa-tracker:generateDebugResValues': FROM_CACHE,
        ':santa-tracker:generateDebugResources': UP_TO_DATE,
        ':santa-tracker:handleDebugMicroApk': FROM_CACHE,
        ':santa-tracker:kaptDebugKotlin': FROM_CACHE,
        ':santa-tracker:kaptGenerateStubsDebugKotlin': FROM_CACHE,
        ':santa-tracker:mergeDebugAssets': FROM_CACHE,
        ':santa-tracker:mergeDebugJavaResource': FROM_CACHE,
        ':santa-tracker:mergeDebugJniLibFolders': FROM_CACHE,
        ':santa-tracker:mergeDebugNativeDebugMetadata': NO_SOURCE,
        ':santa-tracker:mergeDebugNativeLibs': FROM_CACHE,
        ':santa-tracker:mergeDebugResources': SUCCESS,
        ':santa-tracker:mergeDebugShaders': FROM_CACHE,
        ':santa-tracker:mergeExtDexDebug': FROM_CACHE,
        ':santa-tracker:mergeLibDexDebug': FROM_CACHE,
        ':santa-tracker:mergeProjectDexDebug': FROM_CACHE,
        ':santa-tracker:packageDebug': SUCCESS,
        ':santa-tracker:preBuild': UP_TO_DATE,
        ':santa-tracker:preDebugBuild': FROM_CACHE,
        ':santa-tracker:processDebugJavaRes': NO_SOURCE,
        ':santa-tracker:processDebugMainManifest': FROM_CACHE,
        ':santa-tracker:processDebugManifest': FROM_CACHE,
        ':santa-tracker:processDebugManifestForPackage': FROM_CACHE,
        ':santa-tracker:processDebugResources': SUCCESS,
        ':santa-tracker:signingConfigWriterDebug': FROM_CACHE,
        ':santa-tracker:stripDebugDebugSymbols': NO_SOURCE,
        ':santa-tracker:validateSigningDebug': FROM_CACHE,
        ':santa-tracker:writeDebugAppMetadata': FROM_CACHE,
        ':santa-tracker:writeDebugModuleMetadata': SUCCESS,
        ':santa-tracker:writeDebugSigningConfigVersions': FROM_CACHE,
        ':snowballrun:assembleDebug': SUCCESS,
        ':snowballrun:checkDebugAarMetadata': FROM_CACHE,
        ':snowballrun:checkDebugDuplicateClasses': FROM_CACHE,
        ':snowballrun:compileDebugAidl': NO_SOURCE,
        ':snowballrun:compileDebugJavaWithJavac': FROM_CACHE,
        ':snowballrun:compileDebugRenderscript': NO_SOURCE,
        ':snowballrun:compileDebugShaders': NO_SOURCE,
        ':snowballrun:compileDebugSources': UP_TO_DATE,
        ':snowballrun:compressDebugAssets': FROM_CACHE,
        ':snowballrun:createDebugCompatibleScreenManifests': FROM_CACHE,
        ':snowballrun:desugarDebugFileDependencies': SUCCESS,
        ':snowballrun:dexBuilderDebug': FROM_CACHE,
        ':snowballrun:extractDeepLinksDebug': FROM_CACHE,
        ':snowballrun:featureDebugWriter': SUCCESS,
        ':snowballrun:generateDebugAssets': UP_TO_DATE,
        ':snowballrun:generateDebugBuildConfig': FROM_CACHE,
        ':snowballrun:generateDebugFeatureTransitiveDeps': FROM_CACHE,
        ':snowballrun:generateDebugResValues': FROM_CACHE,
        ':snowballrun:generateDebugResources': UP_TO_DATE,
        ':snowballrun:javaPreCompileDebug': FROM_CACHE,
        ':snowballrun:mergeDebugAssets': FROM_CACHE,
        ':snowballrun:mergeDebugJavaResource': FROM_CACHE,
        ':snowballrun:mergeDebugJniLibFolders': FROM_CACHE,
        ':snowballrun:mergeDebugNativeLibs': NO_SOURCE,
        ':snowballrun:mergeDebugResources': SUCCESS,
        ':snowballrun:mergeDebugShaders': FROM_CACHE,
        ':snowballrun:mergeExtDexDebug': FROM_CACHE,
        ':snowballrun:mergeLibDexDebug': FROM_CACHE,
        ':snowballrun:mergeProjectDexDebug': FROM_CACHE,
        ':snowballrun:packageDebug': SUCCESS,
        ':snowballrun:preBuild': UP_TO_DATE,
        ':snowballrun:preDebugBuild': UP_TO_DATE,
        ':snowballrun:processApplicationManifestDebugForBundle': FROM_CACHE,
        ':snowballrun:processDebugJavaRes': NO_SOURCE,
        ':snowballrun:processDebugMainManifest': FROM_CACHE,
        ':snowballrun:processDebugManifest': FROM_CACHE,
        ':snowballrun:processDebugManifestForPackage': FROM_CACHE,
        ':snowballrun:processDebugResources': SUCCESS,
        ':snowballrun:processManifestDebugForFeature': FROM_CACHE,
        ':snowballrun:stripDebugDebugSymbols': NO_SOURCE,
        ':tracker:assembleDebug': SUCCESS,
        ':tracker:bundleDebugAar': SUCCESS,
        ':tracker:bundleLibCompileToJarDebug': FROM_CACHE,
        ':tracker:bundleLibResDebug': FROM_CACHE,
        ':tracker:bundleLibRuntimeToDirDebug': FROM_CACHE,
        ':tracker:bundleLibRuntimeToJarDebug': FROM_CACHE,
        ':tracker:compileDebugAidl': NO_SOURCE,
        ':tracker:compileDebugJavaWithJavac': SUCCESS,
        ':tracker:compileDebugKotlin': FROM_CACHE,
        ':tracker:compileDebugLibraryResources': SUCCESS,
        ':tracker:compileDebugRenderscript': NO_SOURCE,
        ':tracker:compileDebugShaders': NO_SOURCE,
        ':tracker:compileDebugSources': SUCCESS,
        ':tracker:copyDebugJniLibsProjectAndLocalJars': FROM_CACHE,
        ':tracker:copyDebugJniLibsProjectOnly': FROM_CACHE,
        ':tracker:createFullJarDebug': FROM_CACHE,
        ':tracker:extractDebugAnnotations': FROM_CACHE,
        ':tracker:extractDeepLinksDebug': FROM_CACHE,
        ':tracker:generateDebugAssets': UP_TO_DATE,
        ':tracker:generateDebugBuildConfig': FROM_CACHE,
        ':tracker:generateDebugRFile': FROM_CACHE,
        ':tracker:generateDebugResValues': FROM_CACHE,
        ':tracker:generateDebugResources': UP_TO_DATE,
        ':tracker:kaptDebugKotlin': SUCCESS,
        ':tracker:kaptGenerateStubsDebugKotlin': SUCCESS,
        ':tracker:mergeDebugConsumerProguardFiles': FROM_CACHE,
        ':tracker:mergeDebugGeneratedProguardFiles': FROM_CACHE,
        ':tracker:mergeDebugJavaResource': FROM_CACHE,
        ':tracker:mergeDebugJniLibFolders': FROM_CACHE,
        ':tracker:mergeDebugNativeLibs': NO_SOURCE,
        ':tracker:mergeDebugResources': FROM_CACHE,
        ':tracker:mergeDebugShaders': FROM_CACHE,
        ':tracker:packageDebugAssets': FROM_CACHE,
        ':tracker:packageDebugRenderscript': NO_SOURCE,
        ':tracker:packageDebugResources': FROM_CACHE,
        ':tracker:parseDebugLocalResources': FROM_CACHE,
        ':tracker:preBuild': UP_TO_DATE,
        ':tracker:preDebugBuild': UP_TO_DATE,
        ':tracker:prepareLintJarForPublish': SUCCESS,
        ':tracker:processDebugJavaRes': NO_SOURCE,
        ':tracker:processDebugManifest': FROM_CACHE,
        ':tracker:stripDebugDebugSymbols': NO_SOURCE,
        ':tracker:syncDebugLibJars': FROM_CACHE,
        ':tracker:writeDebugAarMetadata': FROM_CACHE,
        ':wearable:assembleDebug': SUCCESS,
        ':wearable:checkDebugAarMetadata': FROM_CACHE,
        ':wearable:checkDebugDuplicateClasses': FROM_CACHE,
        ':wearable:compileDebugAidl': NO_SOURCE,
        ':wearable:compileDebugJavaWithJavac': FROM_CACHE,
        ':wearable:compileDebugKotlin': FROM_CACHE,
        ':wearable:compileDebugRenderscript': NO_SOURCE,
        ':wearable:compileDebugShaders': NO_SOURCE,
        ':wearable:compileDebugSources': UP_TO_DATE,
        ':wearable:compressDebugAssets': FROM_CACHE,
        ':wearable:createDebugCompatibleScreenManifests': FROM_CACHE,
        ':wearable:desugarDebugFileDependencies': SUCCESS,
        ':wearable:dexBuilderDebug': FROM_CACHE,
        ':wearable:extractDeepLinksDebug': FROM_CACHE,
        ':wearable:generateDebugAssets': UP_TO_DATE,
        ':wearable:generateDebugBuildConfig': FROM_CACHE,
        ':wearable:generateDebugResValues': FROM_CACHE,
        ':wearable:generateDebugResources': UP_TO_DATE,
        ':wearable:kaptDebugKotlin': FROM_CACHE,
        ':wearable:kaptGenerateStubsDebugKotlin': FROM_CACHE,
        ':wearable:mergeDebugAssets': FROM_CACHE,
        ':wearable:mergeDebugJavaResource': FROM_CACHE,
        ':wearable:mergeDebugJniLibFolders': FROM_CACHE,
        ':wearable:mergeDebugNativeDebugMetadata': NO_SOURCE,
        ':wearable:mergeDebugNativeLibs': FROM_CACHE,
        ':wearable:mergeDebugResources': SUCCESS,
        ':wearable:mergeDebugShaders': FROM_CACHE,
        ':wearable:mergeExtDexDebug': FROM_CACHE,
        ':wearable:mergeLibDexDebug': FROM_CACHE,
        ':wearable:mergeProjectDexDebug': FROM_CACHE,
        ':wearable:packageDebug': SUCCESS,
        ':wearable:preBuild': UP_TO_DATE,
        ':wearable:preDebugBuild': UP_TO_DATE,
        ':wearable:processDebugJavaRes': NO_SOURCE,
        ':wearable:processDebugMainManifest': FROM_CACHE,
        ':wearable:processDebugManifest': FROM_CACHE,
        ':wearable:processDebugManifestForPackage': FROM_CACHE,
        ':wearable:processDebugResources': SUCCESS,
        ':wearable:stripDebugDebugSymbols': NO_SOURCE,
        ':wearable:validateSigningDebug': FROM_CACHE,
        ':wearable:writeDebugAppMetadata': FROM_CACHE,
        ':wearable:writeDebugSigningConfigVersions': FROM_CACHE,
    ]

    private static final EXPECTED_RESULTS_4_1 = [
        ':cityquiz:assembleDebug': SUCCESS,
        ':cityquiz:checkDebugAarMetadata': FROM_CACHE,
        ':cityquiz:checkDebugDuplicateClasses': FROM_CACHE,
        ':cityquiz:compileDebugAidl': NO_SOURCE,
        ':cityquiz:compileDebugJavaWithJavac': FROM_CACHE,
        ':cityquiz:compileDebugKotlin': FROM_CACHE,
        ':cityquiz:compileDebugRenderscript': NO_SOURCE,
        ':cityquiz:compileDebugShaders': NO_SOURCE,
        ':cityquiz:compileDebugSources': UP_TO_DATE,
        ':cityquiz:compressDebugAssets': FROM_CACHE,
        ':cityquiz:createDebugCompatibleScreenManifests': FROM_CACHE,
        ':cityquiz:dexBuilderDebug': FROM_CACHE,
        ':cityquiz:extractDeepLinksDebug': FROM_CACHE,
        ':cityquiz:featureDebugWriter': SUCCESS,
        ':cityquiz:generateDebugAssets': UP_TO_DATE,
        ':cityquiz:generateDebugBuildConfig': FROM_CACHE,
        ':cityquiz:generateDebugFeatureTransitiveDeps': FROM_CACHE,
        ':cityquiz:generateDebugResValues': FROM_CACHE,
        ':cityquiz:generateDebugResources': UP_TO_DATE,
        ':cityquiz:javaPreCompileDebug': FROM_CACHE,
        ':cityquiz:mergeDebugAssets': FROM_CACHE,
        ':cityquiz:mergeDebugJavaResource': FROM_CACHE,
        ':cityquiz:mergeDebugJniLibFolders': FROM_CACHE,
        ':cityquiz:mergeDebugNativeLibs': NO_SOURCE,
        ':cityquiz:mergeDebugResources': SUCCESS,
        ':cityquiz:mergeDebugShaders': FROM_CACHE,
        ':cityquiz:mergeExtDexDebug': FROM_CACHE,
        ':cityquiz:mergeLibDexDebug': FROM_CACHE,
        ':cityquiz:mergeProjectDexDebug': FROM_CACHE,
        ':cityquiz:packageDebug': SUCCESS,
        ':cityquiz:preBuild': UP_TO_DATE,
        ':cityquiz:preDebugBuild': UP_TO_DATE,
        ':cityquiz:processApplicationManifestDebugForBundle': FROM_CACHE,
        ':cityquiz:processDebugJavaRes': NO_SOURCE,
        ':cityquiz:processDebugMainManifest': FROM_CACHE,
        ':cityquiz:processDebugManifest': FROM_CACHE,
        ':cityquiz:processDebugManifestForPackage': FROM_CACHE,
        ':cityquiz:processDebugResources': SUCCESS,
        ':cityquiz:processManifestDebugForFeature': FROM_CACHE,
        ':cityquiz:stripDebugDebugSymbols': NO_SOURCE,
        ':common:assembleDebug': SUCCESS,
        ':common:bundleDebugAar': SUCCESS,
        ':common:bundleLibCompileToJarDebug': FROM_CACHE,
        ':common:bundleLibResDebug': FROM_CACHE,
        ':common:bundleLibRuntimeToJarDebug': FROM_CACHE,
        ':common:compileDebugAidl': NO_SOURCE,
        ':common:compileDebugJavaWithJavac': FROM_CACHE,
        ':common:compileDebugKotlin': FROM_CACHE,
        ':common:compileDebugLibraryResources': SUCCESS,
        ':common:compileDebugRenderscript': NO_SOURCE,
        ':common:compileDebugShaders': NO_SOURCE,
        ':common:compileDebugSources': UP_TO_DATE,
        ':common:copyDebugJniLibsProjectAndLocalJars': FROM_CACHE,
        ':common:copyDebugJniLibsProjectOnly': FROM_CACHE,
        ':common:createFullJarDebug': FROM_CACHE,
        ':common:extractDebugAnnotations': FROM_CACHE,
        ':common:extractDeepLinksDebug': FROM_CACHE,
        ':common:generateDebugAssets': UP_TO_DATE,
        ':common:generateDebugBuildConfig': FROM_CACHE,
        ':common:generateDebugRFile': FROM_CACHE,
        ':common:generateDebugResValues': FROM_CACHE,
        ':common:generateDebugResources': UP_TO_DATE,
        ':common:javaPreCompileDebug': FROM_CACHE,
        ':common:mergeDebugConsumerProguardFiles': FROM_CACHE,
        ':common:mergeDebugGeneratedProguardFiles': FROM_CACHE,
        ':common:mergeDebugJavaResource': FROM_CACHE,
        ':common:mergeDebugJniLibFolders': FROM_CACHE,
        ':common:mergeDebugNativeLibs': NO_SOURCE,
        ':common:mergeDebugShaders': FROM_CACHE,
        ':common:packageDebugAssets': FROM_CACHE,
        ':common:packageDebugRenderscript': NO_SOURCE,
        ':common:packageDebugResources': FROM_CACHE,
        ':common:parseDebugLocalResources': FROM_CACHE,
        ':common:preBuild': UP_TO_DATE,
        ':common:preDebugBuild': UP_TO_DATE,
        ':common:prepareLintJarForPublish': SUCCESS,
        ':common:processDebugJavaRes': NO_SOURCE,
        ':common:processDebugManifest': FROM_CACHE,
        ':common:stripDebugDebugSymbols': NO_SOURCE,
        ':common:syncDebugLibJars': FROM_CACHE,
        ':common:writeDebugAarMetadata': FROM_CACHE,
        ':dasherdancer:assembleDebug': SUCCESS,
        ':dasherdancer:checkDebugAarMetadata': FROM_CACHE,
        ':dasherdancer:checkDebugDuplicateClasses': FROM_CACHE,
        ':dasherdancer:compileDebugAidl': NO_SOURCE,
        ':dasherdancer:compileDebugJavaWithJavac': FROM_CACHE,
        ':dasherdancer:compileDebugKotlin': FROM_CACHE,
        ':dasherdancer:compileDebugRenderscript': NO_SOURCE,
        ':dasherdancer:compileDebugShaders': NO_SOURCE,
        ':dasherdancer:compileDebugSources': UP_TO_DATE,
        ':dasherdancer:compressDebugAssets': FROM_CACHE,
        ':dasherdancer:createDebugCompatibleScreenManifests': FROM_CACHE,
        ':dasherdancer:dexBuilderDebug': FROM_CACHE,
        ':dasherdancer:extractDeepLinksDebug': FROM_CACHE,
        ':dasherdancer:featureDebugWriter': SUCCESS,
        ':dasherdancer:generateDebugAssets': UP_TO_DATE,
        ':dasherdancer:generateDebugBuildConfig': FROM_CACHE,
        ':dasherdancer:generateDebugFeatureTransitiveDeps': FROM_CACHE,
        ':dasherdancer:generateDebugResValues': FROM_CACHE,
        ':dasherdancer:generateDebugResources': UP_TO_DATE,
        ':dasherdancer:javaPreCompileDebug': FROM_CACHE,
        ':dasherdancer:mergeDebugAssets': FROM_CACHE,
        ':dasherdancer:mergeDebugJavaResource': FROM_CACHE,
        ':dasherdancer:mergeDebugJniLibFolders': FROM_CACHE,
        ':dasherdancer:mergeDebugNativeLibs': NO_SOURCE,
        ':dasherdancer:mergeDebugResources': SUCCESS,
        ':dasherdancer:mergeDebugShaders': FROM_CACHE,
        ':dasherdancer:mergeExtDexDebug': FROM_CACHE,
        ':dasherdancer:mergeLibDexDebug': FROM_CACHE,
        ':dasherdancer:mergeProjectDexDebug': FROM_CACHE,
        ':dasherdancer:packageDebug': SUCCESS,
        ':dasherdancer:preBuild': UP_TO_DATE,
        ':dasherdancer:preDebugBuild': UP_TO_DATE,
        ':dasherdancer:processApplicationManifestDebugForBundle': FROM_CACHE,
        ':dasherdancer:processDebugJavaRes': NO_SOURCE,
        ':dasherdancer:processDebugMainManifest': FROM_CACHE,
        ':dasherdancer:processDebugManifest': FROM_CACHE,
        ':dasherdancer:processDebugManifestForPackage': FROM_CACHE,
        ':dasherdancer:processDebugResources': SUCCESS,
        ':dasherdancer:processManifestDebugForFeature': FROM_CACHE,
        ':dasherdancer:stripDebugDebugSymbols': NO_SOURCE,
        ':doodles-lib:assembleDebug': SUCCESS,
        ':doodles-lib:bundleDebugAar': SUCCESS,
        ':doodles-lib:bundleLibCompileToJarDebug': FROM_CACHE,
        ':doodles-lib:bundleLibResDebug': NO_SOURCE,
        ':doodles-lib:bundleLibRuntimeToJarDebug': FROM_CACHE,
        ':doodles-lib:compileDebugAidl': NO_SOURCE,
        ':doodles-lib:compileDebugJavaWithJavac': FROM_CACHE,
        ':doodles-lib:compileDebugLibraryResources': SUCCESS,
        ':doodles-lib:compileDebugRenderscript': NO_SOURCE,
        ':doodles-lib:compileDebugShaders': NO_SOURCE,
        ':doodles-lib:compileDebugSources': UP_TO_DATE,
        ':doodles-lib:copyDebugJniLibsProjectAndLocalJars': FROM_CACHE,
        ':doodles-lib:copyDebugJniLibsProjectOnly': FROM_CACHE,
        ':doodles-lib:createFullJarDebug': FROM_CACHE,
        ':doodles-lib:extractDebugAnnotations': FROM_CACHE,
        ':doodles-lib:extractDeepLinksDebug': FROM_CACHE,
        ':doodles-lib:generateDebugAssets': UP_TO_DATE,
        ':doodles-lib:generateDebugBuildConfig': FROM_CACHE,
        ':doodles-lib:generateDebugRFile': FROM_CACHE,
        ':doodles-lib:generateDebugResValues': FROM_CACHE,
        ':doodles-lib:generateDebugResources': UP_TO_DATE,
        ':doodles-lib:javaPreCompileDebug': FROM_CACHE,
        ':doodles-lib:mergeDebugConsumerProguardFiles': FROM_CACHE,
        ':doodles-lib:mergeDebugGeneratedProguardFiles': FROM_CACHE,
        ':doodles-lib:mergeDebugJavaResource': FROM_CACHE,
        ':doodles-lib:mergeDebugJniLibFolders': FROM_CACHE,
        ':doodles-lib:mergeDebugNativeLibs': NO_SOURCE,
        ':doodles-lib:mergeDebugShaders': FROM_CACHE,
        ':doodles-lib:packageDebugAssets': FROM_CACHE,
        ':doodles-lib:packageDebugRenderscript': NO_SOURCE,
        ':doodles-lib:packageDebugResources': FROM_CACHE,
        ':doodles-lib:parseDebugLocalResources': FROM_CACHE,
        ':doodles-lib:preBuild': UP_TO_DATE,
        ':doodles-lib:preDebugBuild': UP_TO_DATE,
        ':doodles-lib:prepareLintJarForPublish': SUCCESS,
        ':doodles-lib:processDebugJavaRes': NO_SOURCE,
        ':doodles-lib:processDebugManifest': FROM_CACHE,
        ':doodles-lib:stripDebugDebugSymbols': NO_SOURCE,
        ':doodles-lib:syncDebugLibJars': FROM_CACHE,
        ':doodles-lib:writeDebugAarMetadata': FROM_CACHE,
        ':gumball:assembleDebug': SUCCESS,
        ':gumball:checkDebugAarMetadata': FROM_CACHE,
        ':gumball:checkDebugDuplicateClasses': FROM_CACHE,
        ':gumball:compileDebugAidl': NO_SOURCE,
        ':gumball:compileDebugJavaWithJavac': FROM_CACHE,
        ':gumball:compileDebugRenderscript': NO_SOURCE,
        ':gumball:compileDebugShaders': NO_SOURCE,
        ':gumball:compileDebugSources': UP_TO_DATE,
        ':gumball:compressDebugAssets': FROM_CACHE,
        ':gumball:createDebugCompatibleScreenManifests': FROM_CACHE,
        ':gumball:dexBuilderDebug': FROM_CACHE,
        ':gumball:extractDeepLinksDebug': FROM_CACHE,
        ':gumball:featureDebugWriter': SUCCESS,
        ':gumball:generateDebugAssets': UP_TO_DATE,
        ':gumball:generateDebugBuildConfig': FROM_CACHE,
        ':gumball:generateDebugFeatureTransitiveDeps': FROM_CACHE,
        ':gumball:generateDebugResValues': FROM_CACHE,
        ':gumball:generateDebugResources': UP_TO_DATE,
        ':gumball:javaPreCompileDebug': FROM_CACHE,
        ':gumball:mergeDebugAssets': FROM_CACHE,
        ':gumball:mergeDebugJavaResource': FROM_CACHE,
        ':gumball:mergeDebugJniLibFolders': FROM_CACHE,
        ':gumball:mergeDebugNativeLibs': FROM_CACHE,
        ':gumball:mergeDebugResources': SUCCESS,
        ':gumball:mergeDebugShaders': FROM_CACHE,
        ':gumball:mergeExtDexDebug': FROM_CACHE,
        ':gumball:mergeLibDexDebug': FROM_CACHE,
        ':gumball:mergeProjectDexDebug': FROM_CACHE,
        ':gumball:packageDebug': SUCCESS,
        ':gumball:preBuild': UP_TO_DATE,
        ':gumball:preDebugBuild': UP_TO_DATE,
        ':gumball:processApplicationManifestDebugForBundle': FROM_CACHE,
        ':gumball:processDebugJavaRes': NO_SOURCE,
        ':gumball:processDebugMainManifest': FROM_CACHE,
        ':gumball:processDebugManifest': FROM_CACHE,
        ':gumball:processDebugManifestForPackage': FROM_CACHE,
        ':gumball:processDebugResources': SUCCESS,
        ':gumball:processManifestDebugForFeature': FROM_CACHE,
        ':gumball:stripDebugDebugSymbols': NO_SOURCE,
        ':jetpack:assembleDebug': SUCCESS,
        ':jetpack:checkDebugAarMetadata': FROM_CACHE,
        ':jetpack:checkDebugDuplicateClasses': FROM_CACHE,
        ':jetpack:compileDebugAidl': NO_SOURCE,
        ':jetpack:compileDebugJavaWithJavac': FROM_CACHE,
        ':jetpack:compileDebugKotlin': FROM_CACHE,
        ':jetpack:compileDebugRenderscript': NO_SOURCE,
        ':jetpack:compileDebugShaders': NO_SOURCE,
        ':jetpack:compileDebugSources': UP_TO_DATE,
        ':jetpack:compressDebugAssets': FROM_CACHE,
        ':jetpack:createDebugCompatibleScreenManifests': FROM_CACHE,
        ':jetpack:dexBuilderDebug': FROM_CACHE,
        ':jetpack:extractDeepLinksDebug': FROM_CACHE,
        ':jetpack:featureDebugWriter': SUCCESS,
        ':jetpack:generateDebugAssets': UP_TO_DATE,
        ':jetpack:generateDebugBuildConfig': FROM_CACHE,
        ':jetpack:generateDebugFeatureTransitiveDeps': FROM_CACHE,
        ':jetpack:generateDebugResValues': FROM_CACHE,
        ':jetpack:generateDebugResources': UP_TO_DATE,
        ':jetpack:javaPreCompileDebug': FROM_CACHE,
        ':jetpack:mergeDebugAssets': FROM_CACHE,
        ':jetpack:mergeDebugJavaResource': FROM_CACHE,
        ':jetpack:mergeDebugJniLibFolders': FROM_CACHE,
        ':jetpack:mergeDebugNativeLibs': NO_SOURCE,
        ':jetpack:mergeDebugResources': SUCCESS,
        ':jetpack:mergeDebugShaders': FROM_CACHE,
        ':jetpack:mergeExtDexDebug': FROM_CACHE,
        ':jetpack:mergeLibDexDebug': FROM_CACHE,
        ':jetpack:mergeProjectDexDebug': FROM_CACHE,
        ':jetpack:packageDebug': SUCCESS,
        ':jetpack:preBuild': UP_TO_DATE,
        ':jetpack:preDebugBuild': UP_TO_DATE,
        ':jetpack:processApplicationManifestDebugForBundle': FROM_CACHE,
        ':jetpack:processDebugJavaRes': NO_SOURCE,
        ':jetpack:processDebugMainManifest': FROM_CACHE,
        ':jetpack:processDebugManifest': FROM_CACHE,
        ':jetpack:processDebugManifestForPackage': FROM_CACHE,
        ':jetpack:processDebugResources': SUCCESS,
        ':jetpack:processManifestDebugForFeature': FROM_CACHE,
        ':jetpack:stripDebugDebugSymbols': NO_SOURCE,
        ':memory:assembleDebug': SUCCESS,
        ':memory:checkDebugAarMetadata': FROM_CACHE,
        ':memory:checkDebugDuplicateClasses': FROM_CACHE,
        ':memory:compileDebugAidl': NO_SOURCE,
        ':memory:compileDebugJavaWithJavac': FROM_CACHE,
        ':memory:compileDebugRenderscript': NO_SOURCE,
        ':memory:compileDebugShaders': NO_SOURCE,
        ':memory:compileDebugSources': UP_TO_DATE,
        ':memory:compressDebugAssets': FROM_CACHE,
        ':memory:createDebugCompatibleScreenManifests': FROM_CACHE,
        ':memory:dexBuilderDebug': FROM_CACHE,
        ':memory:extractDeepLinksDebug': FROM_CACHE,
        ':memory:featureDebugWriter': SUCCESS,
        ':memory:generateDebugAssets': UP_TO_DATE,
        ':memory:generateDebugBuildConfig': FROM_CACHE,
        ':memory:generateDebugFeatureTransitiveDeps': FROM_CACHE,
        ':memory:generateDebugResValues': FROM_CACHE,
        ':memory:generateDebugResources': UP_TO_DATE,
        ':memory:javaPreCompileDebug': FROM_CACHE,
        ':memory:mergeDebugAssets': FROM_CACHE,
        ':memory:mergeDebugJavaResource': FROM_CACHE,
        ':memory:mergeDebugJniLibFolders': FROM_CACHE,
        ':memory:mergeDebugNativeLibs': NO_SOURCE,
        ':memory:mergeDebugResources': SUCCESS,
        ':memory:mergeDebugShaders': FROM_CACHE,
        ':memory:mergeExtDexDebug': FROM_CACHE,
        ':memory:mergeLibDexDebug': FROM_CACHE,
        ':memory:mergeProjectDexDebug': FROM_CACHE,
        ':memory:packageDebug': SUCCESS,
        ':memory:preBuild': UP_TO_DATE,
        ':memory:preDebugBuild': UP_TO_DATE,
        ':memory:processApplicationManifestDebugForBundle': FROM_CACHE,
        ':memory:processDebugJavaRes': NO_SOURCE,
        ':memory:processDebugMainManifest': FROM_CACHE,
        ':memory:processDebugManifest': FROM_CACHE,
        ':memory:processDebugManifestForPackage': FROM_CACHE,
        ':memory:processDebugResources': SUCCESS,
        ':memory:processManifestDebugForFeature': FROM_CACHE,
        ':memory:stripDebugDebugSymbols': NO_SOURCE,
        ':penguinswim:assembleDebug': SUCCESS,
        ':penguinswim:checkDebugAarMetadata': FROM_CACHE,
        ':penguinswim:checkDebugDuplicateClasses': FROM_CACHE,
        ':penguinswim:compileDebugAidl': NO_SOURCE,
        ':penguinswim:compileDebugJavaWithJavac': FROM_CACHE,
        ':penguinswim:compileDebugRenderscript': NO_SOURCE,
        ':penguinswim:compileDebugShaders': NO_SOURCE,
        ':penguinswim:compileDebugSources': UP_TO_DATE,
        ':penguinswim:compressDebugAssets': FROM_CACHE,
        ':penguinswim:createDebugCompatibleScreenManifests': FROM_CACHE,
        ':penguinswim:dexBuilderDebug': FROM_CACHE,
        ':penguinswim:extractDeepLinksDebug': FROM_CACHE,
        ':penguinswim:featureDebugWriter': SUCCESS,
        ':penguinswim:generateDebugAssets': UP_TO_DATE,
        ':penguinswim:generateDebugBuildConfig': FROM_CACHE,
        ':penguinswim:generateDebugFeatureTransitiveDeps': FROM_CACHE,
        ':penguinswim:generateDebugResValues': FROM_CACHE,
        ':penguinswim:generateDebugResources': UP_TO_DATE,
        ':penguinswim:javaPreCompileDebug': FROM_CACHE,
        ':penguinswim:mergeDebugAssets': FROM_CACHE,
        ':penguinswim:mergeDebugJavaResource': FROM_CACHE,
        ':penguinswim:mergeDebugJniLibFolders': FROM_CACHE,
        ':penguinswim:mergeDebugNativeLibs': NO_SOURCE,
        ':penguinswim:mergeDebugResources': SUCCESS,
        ':penguinswim:mergeDebugShaders': FROM_CACHE,
        ':penguinswim:mergeExtDexDebug': FROM_CACHE,
        ':penguinswim:mergeLibDexDebug': FROM_CACHE,
        ':penguinswim:mergeProjectDexDebug': FROM_CACHE,
        ':penguinswim:packageDebug': SUCCESS,
        ':penguinswim:preBuild': UP_TO_DATE,
        ':penguinswim:preDebugBuild': UP_TO_DATE,
        ':penguinswim:processApplicationManifestDebugForBundle': FROM_CACHE,
        ':penguinswim:processDebugJavaRes': NO_SOURCE,
        ':penguinswim:processDebugMainManifest': FROM_CACHE,
        ':penguinswim:processDebugManifest': FROM_CACHE,
        ':penguinswim:processDebugManifestForPackage': FROM_CACHE,
        ':penguinswim:processDebugResources': SUCCESS,
        ':penguinswim:processManifestDebugForFeature': FROM_CACHE,
        ':penguinswim:stripDebugDebugSymbols': NO_SOURCE,
        ':playgames:assembleDebug': SUCCESS,
        ':playgames:bundleDebugAar': SUCCESS,
        ':playgames:bundleLibCompileToJarDebug': FROM_CACHE,
        ':playgames:bundleLibResDebug': NO_SOURCE,
        ':playgames:bundleLibRuntimeToJarDebug': FROM_CACHE,
        ':playgames:compileDebugAidl': NO_SOURCE,
        ':playgames:compileDebugJavaWithJavac': FROM_CACHE,
        ':playgames:compileDebugLibraryResources': SUCCESS,
        ':playgames:compileDebugRenderscript': NO_SOURCE,
        ':playgames:compileDebugShaders': NO_SOURCE,
        ':playgames:compileDebugSources': UP_TO_DATE,
        ':playgames:copyDebugJniLibsProjectAndLocalJars': FROM_CACHE,
        ':playgames:copyDebugJniLibsProjectOnly': FROM_CACHE,
        ':playgames:createFullJarDebug': FROM_CACHE,
        ':playgames:extractDebugAnnotations': FROM_CACHE,
        ':playgames:extractDeepLinksDebug': FROM_CACHE,
        ':playgames:generateDebugAssets': UP_TO_DATE,
        ':playgames:generateDebugBuildConfig': FROM_CACHE,
        ':playgames:generateDebugRFile': FROM_CACHE,
        ':playgames:generateDebugResValues': FROM_CACHE,
        ':playgames:generateDebugResources': UP_TO_DATE,
        ':playgames:javaPreCompileDebug': FROM_CACHE,
        ':playgames:mergeDebugConsumerProguardFiles': FROM_CACHE,
        ':playgames:mergeDebugGeneratedProguardFiles': FROM_CACHE,
        ':playgames:mergeDebugJavaResource': FROM_CACHE,
        ':playgames:mergeDebugJniLibFolders': FROM_CACHE,
        ':playgames:mergeDebugNativeLibs': NO_SOURCE,
        ':playgames:mergeDebugShaders': FROM_CACHE,
        ':playgames:packageDebugAssets': FROM_CACHE,
        ':playgames:packageDebugRenderscript': NO_SOURCE,
        ':playgames:packageDebugResources': FROM_CACHE,
        ':playgames:parseDebugLocalResources': FROM_CACHE,
        ':playgames:preBuild': UP_TO_DATE,
        ':playgames:preDebugBuild': UP_TO_DATE,
        ':playgames:prepareLintJarForPublish': SUCCESS,
        ':playgames:processDebugJavaRes': NO_SOURCE,
        ':playgames:processDebugManifest': FROM_CACHE,
        ':playgames:stripDebugDebugSymbols': NO_SOURCE,
        ':playgames:syncDebugLibJars': FROM_CACHE,
        ':playgames:writeDebugAarMetadata': FROM_CACHE,
        ':presenttoss:assembleDebug': SUCCESS,
        ':presenttoss:checkDebugAarMetadata': FROM_CACHE,
        ':presenttoss:checkDebugDuplicateClasses': FROM_CACHE,
        ':presenttoss:compileDebugAidl': NO_SOURCE,
        ':presenttoss:compileDebugJavaWithJavac': FROM_CACHE,
        ':presenttoss:compileDebugRenderscript': NO_SOURCE,
        ':presenttoss:compileDebugShaders': NO_SOURCE,
        ':presenttoss:compileDebugSources': UP_TO_DATE,
        ':presenttoss:compressDebugAssets': FROM_CACHE,
        ':presenttoss:createDebugCompatibleScreenManifests': FROM_CACHE,
        ':presenttoss:dexBuilderDebug': FROM_CACHE,
        ':presenttoss:extractDeepLinksDebug': FROM_CACHE,
        ':presenttoss:featureDebugWriter': SUCCESS,
        ':presenttoss:generateDebugAssets': UP_TO_DATE,
        ':presenttoss:generateDebugBuildConfig': FROM_CACHE,
        ':presenttoss:generateDebugFeatureTransitiveDeps': FROM_CACHE,
        ':presenttoss:generateDebugResValues': FROM_CACHE,
        ':presenttoss:generateDebugResources': UP_TO_DATE,
        ':presenttoss:javaPreCompileDebug': FROM_CACHE,
        ':presenttoss:mergeDebugAssets': FROM_CACHE,
        ':presenttoss:mergeDebugJavaResource': FROM_CACHE,
        ':presenttoss:mergeDebugJniLibFolders': FROM_CACHE,
        ':presenttoss:mergeDebugNativeLibs': NO_SOURCE,
        ':presenttoss:mergeDebugResources': SUCCESS,
        ':presenttoss:mergeDebugShaders': FROM_CACHE,
        ':presenttoss:mergeExtDexDebug': FROM_CACHE,
        ':presenttoss:mergeLibDexDebug': FROM_CACHE,
        ':presenttoss:mergeProjectDexDebug': FROM_CACHE,
        ':presenttoss:packageDebug': SUCCESS,
        ':presenttoss:preBuild': UP_TO_DATE,
        ':presenttoss:preDebugBuild': UP_TO_DATE,
        ':presenttoss:processApplicationManifestDebugForBundle': FROM_CACHE,
        ':presenttoss:processDebugJavaRes': NO_SOURCE,
        ':presenttoss:processDebugMainManifest': FROM_CACHE,
        ':presenttoss:processDebugManifest': FROM_CACHE,
        ':presenttoss:processDebugManifestForPackage': FROM_CACHE,
        ':presenttoss:processDebugResources': SUCCESS,
        ':presenttoss:processManifestDebugForFeature': FROM_CACHE,
        ':presenttoss:stripDebugDebugSymbols': NO_SOURCE,
        ':rocketsleigh:assembleDebug': SUCCESS,
        ':rocketsleigh:checkDebugAarMetadata': FROM_CACHE,
        ':rocketsleigh:checkDebugDuplicateClasses': FROM_CACHE,
        ':rocketsleigh:compileDebugAidl': NO_SOURCE,
        ':rocketsleigh:compileDebugJavaWithJavac': FROM_CACHE,
        ':rocketsleigh:compileDebugKotlin': FROM_CACHE,
        ':rocketsleigh:compileDebugRenderscript': NO_SOURCE,
        ':rocketsleigh:compileDebugShaders': NO_SOURCE,
        ':rocketsleigh:compileDebugSources': UP_TO_DATE,
        ':rocketsleigh:compressDebugAssets': FROM_CACHE,
        ':rocketsleigh:createDebugCompatibleScreenManifests': FROM_CACHE,
        ':rocketsleigh:dexBuilderDebug': FROM_CACHE,
        ':rocketsleigh:extractDeepLinksDebug': FROM_CACHE,
        ':rocketsleigh:featureDebugWriter': SUCCESS,
        ':rocketsleigh:generateDebugAssets': UP_TO_DATE,
        ':rocketsleigh:generateDebugBuildConfig': FROM_CACHE,
        ':rocketsleigh:generateDebugFeatureTransitiveDeps': FROM_CACHE,
        ':rocketsleigh:generateDebugResValues': FROM_CACHE,
        ':rocketsleigh:generateDebugResources': UP_TO_DATE,
        ':rocketsleigh:javaPreCompileDebug': FROM_CACHE,
        ':rocketsleigh:mergeDebugAssets': FROM_CACHE,
        ':rocketsleigh:mergeDebugJavaResource': FROM_CACHE,
        ':rocketsleigh:mergeDebugJniLibFolders': FROM_CACHE,
        ':rocketsleigh:mergeDebugNativeLibs': NO_SOURCE,
        ':rocketsleigh:mergeDebugResources': SUCCESS,
        ':rocketsleigh:mergeDebugShaders': FROM_CACHE,
        ':rocketsleigh:mergeExtDexDebug': FROM_CACHE,
        ':rocketsleigh:mergeLibDexDebug': FROM_CACHE,
        ':rocketsleigh:mergeProjectDexDebug': FROM_CACHE,
        ':rocketsleigh:packageDebug': SUCCESS,
        ':rocketsleigh:preBuild': UP_TO_DATE,
        ':rocketsleigh:preDebugBuild': UP_TO_DATE,
        ':rocketsleigh:processApplicationManifestDebugForBundle': FROM_CACHE,
        ':rocketsleigh:processDebugJavaRes': NO_SOURCE,
        ':rocketsleigh:processDebugMainManifest': FROM_CACHE,
        ':rocketsleigh:processDebugManifest': FROM_CACHE,
        ':rocketsleigh:processDebugManifestForPackage': FROM_CACHE,
        ':rocketsleigh:processDebugResources': SUCCESS,
        ':rocketsleigh:processManifestDebugForFeature': FROM_CACHE,
        ':rocketsleigh:stripDebugDebugSymbols': NO_SOURCE,
        ':santa-tracker:assembleDebug': SUCCESS,
        ':santa-tracker:bundleDebugClasses': FROM_CACHE,
        ':santa-tracker:checkDebugAarMetadata': FROM_CACHE,
        ':santa-tracker:checkDebugDuplicateClasses': FROM_CACHE,
        ':santa-tracker:checkDebugLibraries': FROM_CACHE,
        ':santa-tracker:compileDebugAidl': NO_SOURCE,
        ':santa-tracker:compileDebugJavaWithJavac': FROM_CACHE,
        ':santa-tracker:compileDebugKotlin': FROM_CACHE,
        ':santa-tracker:compileDebugRenderscript': NO_SOURCE,
        ':santa-tracker:compileDebugShaders': NO_SOURCE,
        ':santa-tracker:compileDebugSources': UP_TO_DATE,
        ':santa-tracker:compressDebugAssets': FROM_CACHE,
        ':santa-tracker:createDebugCompatibleScreenManifests': FROM_CACHE,
        ':santa-tracker:dexBuilderDebug': FROM_CACHE,
        ':santa-tracker:extractDeepLinksDebug': FROM_CACHE,
        ':santa-tracker:generateDebugAssets': UP_TO_DATE,
        ':santa-tracker:generateDebugBuildConfig': FROM_CACHE,
        ':santa-tracker:generateDebugFeatureMetadata': FROM_CACHE,
        ':santa-tracker:generateDebugFeatureTransitiveDeps': FROM_CACHE,
        ':santa-tracker:generateDebugResValues': FROM_CACHE,
        ':santa-tracker:generateDebugResources': UP_TO_DATE,
        ':santa-tracker:handleDebugMicroApk': FROM_CACHE,
        ':santa-tracker:javaPreCompileDebug': FROM_CACHE,
        ':santa-tracker:kaptDebugKotlin': FROM_CACHE,
        ':santa-tracker:kaptGenerateStubsDebugKotlin': FROM_CACHE,
        ':santa-tracker:mergeDebugAssets': FROM_CACHE,
        ':santa-tracker:mergeDebugJavaResource': FROM_CACHE,
        ':santa-tracker:mergeDebugJniLibFolders': FROM_CACHE,
        ':santa-tracker:mergeDebugNativeDebugMetadata': NO_SOURCE,
        ':santa-tracker:mergeDebugNativeLibs': FROM_CACHE,
        ':santa-tracker:mergeDebugResources': SUCCESS,
        ':santa-tracker:mergeDebugShaders': FROM_CACHE,
        ':santa-tracker:mergeExtDexDebug': FROM_CACHE,
        ':santa-tracker:mergeLibDexDebug': FROM_CACHE,
        ':santa-tracker:mergeProjectDexDebug': FROM_CACHE,
        ':santa-tracker:packageDebug': SUCCESS,
        ':santa-tracker:preBuild': UP_TO_DATE,
        ':santa-tracker:preDebugBuild': FROM_CACHE,
        ':santa-tracker:processDebugJavaRes': NO_SOURCE,
        ':santa-tracker:processDebugMainManifest': FROM_CACHE,
        ':santa-tracker:processDebugManifest': FROM_CACHE,
        ':santa-tracker:processDebugManifestForPackage': FROM_CACHE,
        ':santa-tracker:processDebugResources': SUCCESS,
        ':santa-tracker:signingConfigWriterDebug': FROM_CACHE,
        ':santa-tracker:stripDebugDebugSymbols': NO_SOURCE,
        ':santa-tracker:validateSigningDebug': FROM_CACHE,
        ':santa-tracker:writeDebugModuleMetadata': SUCCESS,
        ':snowballrun:assembleDebug': SUCCESS,
        ':snowballrun:checkDebugAarMetadata': FROM_CACHE,
        ':snowballrun:checkDebugDuplicateClasses': FROM_CACHE,
        ':snowballrun:compileDebugAidl': NO_SOURCE,
        ':snowballrun:compileDebugJavaWithJavac': FROM_CACHE,
        ':snowballrun:compileDebugRenderscript': NO_SOURCE,
        ':snowballrun:compileDebugShaders': NO_SOURCE,
        ':snowballrun:compileDebugSources': UP_TO_DATE,
        ':snowballrun:compressDebugAssets': FROM_CACHE,
        ':snowballrun:createDebugCompatibleScreenManifests': FROM_CACHE,
        ':snowballrun:dexBuilderDebug': FROM_CACHE,
        ':snowballrun:extractDeepLinksDebug': FROM_CACHE,
        ':snowballrun:featureDebugWriter': SUCCESS,
        ':snowballrun:generateDebugAssets': UP_TO_DATE,
        ':snowballrun:generateDebugBuildConfig': FROM_CACHE,
        ':snowballrun:generateDebugFeatureTransitiveDeps': FROM_CACHE,
        ':snowballrun:generateDebugResValues': FROM_CACHE,
        ':snowballrun:generateDebugResources': UP_TO_DATE,
        ':snowballrun:javaPreCompileDebug': FROM_CACHE,
        ':snowballrun:mergeDebugAssets': FROM_CACHE,
        ':snowballrun:mergeDebugJavaResource': FROM_CACHE,
        ':snowballrun:mergeDebugJniLibFolders': FROM_CACHE,
        ':snowballrun:mergeDebugNativeLibs': NO_SOURCE,
        ':snowballrun:mergeDebugResources': SUCCESS,
        ':snowballrun:mergeDebugShaders': FROM_CACHE,
        ':snowballrun:mergeExtDexDebug': FROM_CACHE,
        ':snowballrun:mergeLibDexDebug': FROM_CACHE,
        ':snowballrun:mergeProjectDexDebug': FROM_CACHE,
        ':snowballrun:packageDebug': SUCCESS,
        ':snowballrun:preBuild': UP_TO_DATE,
        ':snowballrun:preDebugBuild': UP_TO_DATE,
        ':snowballrun:processApplicationManifestDebugForBundle': FROM_CACHE,
        ':snowballrun:processDebugJavaRes': NO_SOURCE,
        ':snowballrun:processDebugMainManifest': FROM_CACHE,
        ':snowballrun:processDebugManifest': FROM_CACHE,
        ':snowballrun:processDebugManifestForPackage': FROM_CACHE,
        ':snowballrun:processDebugResources': SUCCESS,
        ':snowballrun:processManifestDebugForFeature': FROM_CACHE,
        ':snowballrun:stripDebugDebugSymbols': NO_SOURCE,
        ':tracker:assembleDebug': SUCCESS,
        ':tracker:bundleDebugAar': SUCCESS,
        ':tracker:bundleLibCompileToJarDebug': FROM_CACHE,
        ':tracker:bundleLibResDebug': FROM_CACHE,
        ':tracker:bundleLibRuntimeToJarDebug': FROM_CACHE,
        ':tracker:compileDebugAidl': NO_SOURCE,
        ':tracker:compileDebugJavaWithJavac': SUCCESS,
        ':tracker:compileDebugKotlin': FROM_CACHE,
        ':tracker:compileDebugLibraryResources': SUCCESS,
        ':tracker:compileDebugRenderscript': NO_SOURCE,
        ':tracker:compileDebugShaders': NO_SOURCE,
        ':tracker:compileDebugSources': SUCCESS,
        ':tracker:copyDebugJniLibsProjectAndLocalJars': FROM_CACHE,
        ':tracker:copyDebugJniLibsProjectOnly': FROM_CACHE,
        ':tracker:createFullJarDebug': FROM_CACHE,
        ':tracker:extractDebugAnnotations': FROM_CACHE,
        ':tracker:extractDeepLinksDebug': FROM_CACHE,
        ':tracker:generateDebugAssets': UP_TO_DATE,
        ':tracker:generateDebugBuildConfig': FROM_CACHE,
        ':tracker:generateDebugRFile': FROM_CACHE,
        ':tracker:generateDebugResValues': FROM_CACHE,
        ':tracker:generateDebugResources': UP_TO_DATE,
        ':tracker:javaPreCompileDebug': FROM_CACHE,
        ':tracker:kaptDebugKotlin': SUCCESS,
        ':tracker:kaptGenerateStubsDebugKotlin': SUCCESS,
        ':tracker:mergeDebugConsumerProguardFiles': FROM_CACHE,
        ':tracker:mergeDebugGeneratedProguardFiles': FROM_CACHE,
        ':tracker:mergeDebugJavaResource': FROM_CACHE,
        ':tracker:mergeDebugJniLibFolders': FROM_CACHE,
        ':tracker:mergeDebugNativeLibs': NO_SOURCE,
        ':tracker:mergeDebugResources': FROM_CACHE,
        ':tracker:mergeDebugShaders': FROM_CACHE,
        ':tracker:packageDebugAssets': FROM_CACHE,
        ':tracker:packageDebugRenderscript': NO_SOURCE,
        ':tracker:packageDebugResources': FROM_CACHE,
        ':tracker:parseDebugLocalResources': FROM_CACHE,
        ':tracker:preBuild': UP_TO_DATE,
        ':tracker:preDebugBuild': UP_TO_DATE,
        ':tracker:prepareLintJarForPublish': SUCCESS,
        ':tracker:processDebugJavaRes': NO_SOURCE,
        ':tracker:processDebugManifest': FROM_CACHE,
        ':tracker:stripDebugDebugSymbols': NO_SOURCE,
        ':tracker:syncDebugLibJars': FROM_CACHE,
        ':tracker:writeDebugAarMetadata': FROM_CACHE,
        ':wearable:assembleDebug': SUCCESS,
        ':wearable:checkDebugAarMetadata': FROM_CACHE,
        ':wearable:checkDebugDuplicateClasses': FROM_CACHE,
        ':wearable:compileDebugAidl': NO_SOURCE,
        ':wearable:compileDebugJavaWithJavac': FROM_CACHE,
        ':wearable:compileDebugKotlin': FROM_CACHE,
        ':wearable:compileDebugRenderscript': NO_SOURCE,
        ':wearable:compileDebugShaders': NO_SOURCE,
        ':wearable:compileDebugSources': UP_TO_DATE,
        ':wearable:compressDebugAssets': FROM_CACHE,
        ':wearable:createDebugCompatibleScreenManifests': FROM_CACHE,
        ':wearable:dexBuilderDebug': FROM_CACHE,
        ':wearable:extractDeepLinksDebug': FROM_CACHE,
        ':wearable:generateDebugAssets': UP_TO_DATE,
        ':wearable:generateDebugBuildConfig': FROM_CACHE,
        ':wearable:generateDebugResValues': FROM_CACHE,
        ':wearable:generateDebugResources': UP_TO_DATE,
        ':wearable:javaPreCompileDebug': FROM_CACHE,
        ':wearable:kaptDebugKotlin': FROM_CACHE,
        ':wearable:kaptGenerateStubsDebugKotlin': FROM_CACHE,
        ':wearable:mergeDebugAssets': FROM_CACHE,
        ':wearable:mergeDebugJavaResource': FROM_CACHE,
        ':wearable:mergeDebugJniLibFolders': FROM_CACHE,
        ':wearable:mergeDebugNativeDebugMetadata': NO_SOURCE,
        ':wearable:mergeDebugNativeLibs': FROM_CACHE,
        ':wearable:mergeDebugResources': SUCCESS,
        ':wearable:mergeDebugShaders': FROM_CACHE,
        ':wearable:mergeExtDexDebug': FROM_CACHE,
        ':wearable:mergeLibDexDebug': FROM_CACHE,
        ':wearable:mergeProjectDexDebug': FROM_CACHE,
        ':wearable:packageDebug': SUCCESS,
        ':wearable:preBuild': UP_TO_DATE,
        ':wearable:preDebugBuild': UP_TO_DATE,
        ':wearable:processDebugJavaRes': NO_SOURCE,
        ':wearable:processDebugMainManifest': FROM_CACHE,
        ':wearable:processDebugManifest': FROM_CACHE,
        ':wearable:processDebugManifestForPackage': FROM_CACHE,
        ':wearable:processDebugResources': SUCCESS,
        ':wearable:stripDebugDebugSymbols': NO_SOURCE,
        ':wearable:validateSigningDebug': FROM_CACHE,
    ]

    private static final EXPECTED_RESULTS_4_0 = [
        ':cityquiz:assembleDebug': SUCCESS,
        ':cityquiz:checkDebugDuplicateClasses': FROM_CACHE,
        ':cityquiz:compileDebugAidl': NO_SOURCE,
        ':cityquiz:compileDebugJavaWithJavac': FROM_CACHE,
        ':cityquiz:compileDebugKotlin': FROM_CACHE,
        ':cityquiz:compileDebugRenderscript': NO_SOURCE,
        ':cityquiz:compileDebugShaders': NO_SOURCE,
        ':cityquiz:compileDebugSources': UP_TO_DATE,
        ':cityquiz:createDebugCompatibleScreenManifests': FROM_CACHE,
        ':cityquiz:dexBuilderDebug': FROM_CACHE,
        ':cityquiz:extractDeepLinksDebug': FROM_CACHE,
        ':cityquiz:featureDebugWriter': SUCCESS,
        ':cityquiz:generateDebugAssets': UP_TO_DATE,
        ':cityquiz:generateDebugBuildConfig': FROM_CACHE,
        ':cityquiz:generateDebugFeatureTransitiveDeps': FROM_CACHE,
        ':cityquiz:generateDebugResValues': FROM_CACHE,
        ':cityquiz:generateDebugResources': UP_TO_DATE,
        ':cityquiz:javaPreCompileDebug': FROM_CACHE,
        ':cityquiz:mergeDebugAssets': FROM_CACHE,
        ':cityquiz:mergeDebugJavaResource': SUCCESS,
        ':cityquiz:mergeDebugJniLibFolders': FROM_CACHE,
        ':cityquiz:mergeDebugNativeLibs': FROM_CACHE,
        ':cityquiz:mergeDebugResources': SUCCESS,
        ':cityquiz:mergeDebugShaders': FROM_CACHE,
        ':cityquiz:mergeExtDexDebug': FROM_CACHE,
        ':cityquiz:mergeLibDexDebug': FROM_CACHE,
        ':cityquiz:mergeProjectDexDebug': FROM_CACHE,
        ':cityquiz:packageDebug': SUCCESS,
        ':cityquiz:preBuild': UP_TO_DATE,
        ':cityquiz:preDebugBuild': UP_TO_DATE,
        ':cityquiz:processDebugJavaRes': NO_SOURCE,
        ':cityquiz:processDebugManifest': FROM_CACHE,
        ':cityquiz:processDebugResources': SUCCESS,
        ':cityquiz:stripDebugDebugSymbols': NO_SOURCE,
        ':common:assembleDebug': SUCCESS,
        ':common:bundleDebugAar': SUCCESS,
        ':common:bundleLibCompileToJarDebug': FROM_CACHE,
        ':common:bundleLibResDebug': FROM_CACHE,
        ':common:bundleLibRuntimeToJarDebug': FROM_CACHE,
        ':common:compileDebugAidl': NO_SOURCE,
        ':common:compileDebugJavaWithJavac': FROM_CACHE,
        ':common:compileDebugKotlin': FROM_CACHE,
        ':common:compileDebugLibraryResources': SUCCESS,
        ':common:compileDebugRenderscript': NO_SOURCE,
        ':common:compileDebugShaders': NO_SOURCE,
        ':common:compileDebugSources': UP_TO_DATE,
        ':common:copyDebugJniLibsProjectAndLocalJars': FROM_CACHE,
        ':common:copyDebugJniLibsProjectOnly': FROM_CACHE,
        ':common:createFullJarDebug': FROM_CACHE,
        ':common:extractDebugAnnotations': FROM_CACHE,
        ':common:extractDeepLinksDebug': FROM_CACHE,
        ':common:generateDebugAssets': UP_TO_DATE,
        ':common:generateDebugBuildConfig': FROM_CACHE,
        ':common:generateDebugRFile': FROM_CACHE,
        ':common:generateDebugResValues': FROM_CACHE,
        ':common:generateDebugResources': UP_TO_DATE,
        ':common:javaPreCompileDebug': FROM_CACHE,
        ':common:mergeDebugConsumerProguardFiles': FROM_CACHE,
        ':common:mergeDebugGeneratedProguardFiles': FROM_CACHE,
        ':common:mergeDebugJavaResource': SUCCESS,
        ':common:mergeDebugJniLibFolders': FROM_CACHE,
        ':common:mergeDebugNativeLibs': FROM_CACHE,
        ':common:mergeDebugShaders': FROM_CACHE,
        ':common:packageDebugAssets': FROM_CACHE,
        ':common:packageDebugRenderscript': NO_SOURCE,
        ':common:packageDebugResources': FROM_CACHE,
        ':common:parseDebugLocalResources': FROM_CACHE,
        ':common:preBuild': UP_TO_DATE,
        ':common:preDebugBuild': UP_TO_DATE,
        ':common:prepareLintJarForPublish': SUCCESS,
        ':common:processDebugJavaRes': NO_SOURCE,
        ':common:processDebugManifest': FROM_CACHE,
        ':common:stripDebugDebugSymbols': NO_SOURCE,
        ':common:syncDebugLibJars': SUCCESS,
        ':dasherdancer:assembleDebug': SUCCESS,
        ':dasherdancer:checkDebugDuplicateClasses': FROM_CACHE,
        ':dasherdancer:compileDebugAidl': NO_SOURCE,
        ':dasherdancer:compileDebugJavaWithJavac': FROM_CACHE,
        ':dasherdancer:compileDebugKotlin': FROM_CACHE,
        ':dasherdancer:compileDebugRenderscript': NO_SOURCE,
        ':dasherdancer:compileDebugShaders': NO_SOURCE,
        ':dasherdancer:compileDebugSources': UP_TO_DATE,
        ':dasherdancer:createDebugCompatibleScreenManifests': FROM_CACHE,
        ':dasherdancer:dexBuilderDebug': FROM_CACHE,
        ':dasherdancer:extractDeepLinksDebug': FROM_CACHE,
        ':dasherdancer:featureDebugWriter': SUCCESS,
        ':dasherdancer:generateDebugAssets': UP_TO_DATE,
        ':dasherdancer:generateDebugBuildConfig': FROM_CACHE,
        ':dasherdancer:generateDebugFeatureTransitiveDeps': FROM_CACHE,
        ':dasherdancer:generateDebugResValues': FROM_CACHE,
        ':dasherdancer:generateDebugResources': UP_TO_DATE,
        ':dasherdancer:javaPreCompileDebug': FROM_CACHE,
        ':dasherdancer:mergeDebugAssets': FROM_CACHE,
        ':dasherdancer:mergeDebugJavaResource': SUCCESS,
        ':dasherdancer:mergeDebugJniLibFolders': FROM_CACHE,
        ':dasherdancer:mergeDebugNativeLibs': FROM_CACHE,
        ':dasherdancer:mergeDebugResources': SUCCESS,
        ':dasherdancer:mergeDebugShaders': FROM_CACHE,
        ':dasherdancer:mergeExtDexDebug': FROM_CACHE,
        ':dasherdancer:mergeLibDexDebug': FROM_CACHE,
        ':dasherdancer:mergeProjectDexDebug': FROM_CACHE,
        ':dasherdancer:packageDebug': SUCCESS,
        ':dasherdancer:preBuild': UP_TO_DATE,
        ':dasherdancer:preDebugBuild': UP_TO_DATE,
        ':dasherdancer:processDebugJavaRes': NO_SOURCE,
        ':dasherdancer:processDebugManifest': FROM_CACHE,
        ':dasherdancer:processDebugResources': SUCCESS,
        ':dasherdancer:stripDebugDebugSymbols': NO_SOURCE,
        ':doodles-lib:assembleDebug': SUCCESS,
        ':doodles-lib:bundleDebugAar': SUCCESS,
        ':doodles-lib:bundleLibCompileToJarDebug': FROM_CACHE,
        ':doodles-lib:bundleLibResDebug': NO_SOURCE,
        ':doodles-lib:bundleLibRuntimeToJarDebug': FROM_CACHE,
        ':doodles-lib:compileDebugAidl': NO_SOURCE,
        ':doodles-lib:compileDebugJavaWithJavac': FROM_CACHE,
        ':doodles-lib:compileDebugLibraryResources': SUCCESS,
        ':doodles-lib:compileDebugRenderscript': NO_SOURCE,
        ':doodles-lib:compileDebugShaders': NO_SOURCE,
        ':doodles-lib:compileDebugSources': UP_TO_DATE,
        ':doodles-lib:copyDebugJniLibsProjectAndLocalJars': FROM_CACHE,
        ':doodles-lib:copyDebugJniLibsProjectOnly': FROM_CACHE,
        ':doodles-lib:createFullJarDebug': FROM_CACHE,
        ':doodles-lib:extractDebugAnnotations': FROM_CACHE,
        ':doodles-lib:extractDeepLinksDebug': FROM_CACHE,
        ':doodles-lib:generateDebugAssets': UP_TO_DATE,
        ':doodles-lib:generateDebugBuildConfig': FROM_CACHE,
        ':doodles-lib:generateDebugRFile': FROM_CACHE,
        ':doodles-lib:generateDebugResValues': FROM_CACHE,
        ':doodles-lib:generateDebugResources': UP_TO_DATE,
        ':doodles-lib:javaPreCompileDebug': FROM_CACHE,
        ':doodles-lib:mergeDebugConsumerProguardFiles': FROM_CACHE,
        ':doodles-lib:mergeDebugGeneratedProguardFiles': FROM_CACHE,
        ':doodles-lib:mergeDebugJavaResource': FROM_CACHE,
        ':doodles-lib:mergeDebugJniLibFolders': FROM_CACHE,
        ':doodles-lib:mergeDebugNativeLibs': FROM_CACHE,
        ':doodles-lib:mergeDebugShaders': FROM_CACHE,
        ':doodles-lib:packageDebugAssets': FROM_CACHE,
        ':doodles-lib:packageDebugRenderscript': NO_SOURCE,
        ':doodles-lib:packageDebugResources': FROM_CACHE,
        ':doodles-lib:parseDebugLocalResources': FROM_CACHE,
        ':doodles-lib:preBuild': UP_TO_DATE,
        ':doodles-lib:preDebugBuild': UP_TO_DATE,
        ':doodles-lib:prepareLintJarForPublish': SUCCESS,
        ':doodles-lib:processDebugJavaRes': NO_SOURCE,
        ':doodles-lib:processDebugManifest': FROM_CACHE,
        ':doodles-lib:stripDebugDebugSymbols': NO_SOURCE,
        ':doodles-lib:syncDebugLibJars': FROM_CACHE,
        ':gumball:assembleDebug': SUCCESS,
        ':gumball:checkDebugDuplicateClasses': FROM_CACHE,
        ':gumball:compileDebugAidl': NO_SOURCE,
        ':gumball:compileDebugJavaWithJavac': FROM_CACHE,
        ':gumball:compileDebugRenderscript': NO_SOURCE,
        ':gumball:compileDebugShaders': NO_SOURCE,
        ':gumball:compileDebugSources': UP_TO_DATE,
        ':gumball:createDebugCompatibleScreenManifests': FROM_CACHE,
        ':gumball:dexBuilderDebug': FROM_CACHE,
        ':gumball:extractDeepLinksDebug': FROM_CACHE,
        ':gumball:featureDebugWriter': SUCCESS,
        ':gumball:generateDebugAssets': UP_TO_DATE,
        ':gumball:generateDebugBuildConfig': FROM_CACHE,
        ':gumball:generateDebugFeatureTransitiveDeps': FROM_CACHE,
        ':gumball:generateDebugResValues': FROM_CACHE,
        ':gumball:generateDebugResources': UP_TO_DATE,
        ':gumball:javaPreCompileDebug': FROM_CACHE,
        ':gumball:mergeDebugAssets': FROM_CACHE,
        ':gumball:mergeDebugJavaResource': FROM_CACHE,
        ':gumball:mergeDebugJniLibFolders': FROM_CACHE,
        ':gumball:mergeDebugNativeLibs': FROM_CACHE,
        ':gumball:mergeDebugResources': SUCCESS,
        ':gumball:mergeDebugShaders': FROM_CACHE,
        ':gumball:mergeExtDexDebug': FROM_CACHE,
        ':gumball:mergeLibDexDebug': FROM_CACHE,
        ':gumball:mergeProjectDexDebug': FROM_CACHE,
        ':gumball:packageDebug': SUCCESS,
        ':gumball:preBuild': UP_TO_DATE,
        ':gumball:preDebugBuild': UP_TO_DATE,
        ':gumball:processDebugJavaRes': NO_SOURCE,
        ':gumball:processDebugManifest': FROM_CACHE,
        ':gumball:processDebugResources': SUCCESS,
        ':gumball:stripDebugDebugSymbols': NO_SOURCE,
        ':jetpack:assembleDebug': SUCCESS,
        ':jetpack:checkDebugDuplicateClasses': FROM_CACHE,
        ':jetpack:compileDebugAidl': NO_SOURCE,
        ':jetpack:compileDebugJavaWithJavac': FROM_CACHE,
        ':jetpack:compileDebugKotlin': FROM_CACHE,
        ':jetpack:compileDebugRenderscript': NO_SOURCE,
        ':jetpack:compileDebugShaders': NO_SOURCE,
        ':jetpack:compileDebugSources': UP_TO_DATE,
        ':jetpack:createDebugCompatibleScreenManifests': FROM_CACHE,
        ':jetpack:dexBuilderDebug': FROM_CACHE,
        ':jetpack:extractDeepLinksDebug': FROM_CACHE,
        ':jetpack:featureDebugWriter': SUCCESS,
        ':jetpack:generateDebugAssets': UP_TO_DATE,
        ':jetpack:generateDebugBuildConfig': FROM_CACHE,
        ':jetpack:generateDebugFeatureTransitiveDeps': FROM_CACHE,
        ':jetpack:generateDebugResValues': FROM_CACHE,
        ':jetpack:generateDebugResources': UP_TO_DATE,
        ':jetpack:javaPreCompileDebug': FROM_CACHE,
        ':jetpack:mergeDebugAssets': FROM_CACHE,
        ':jetpack:mergeDebugJavaResource': SUCCESS,
        ':jetpack:mergeDebugJniLibFolders': FROM_CACHE,
        ':jetpack:mergeDebugNativeLibs': FROM_CACHE,
        ':jetpack:mergeDebugResources': SUCCESS,
        ':jetpack:mergeDebugShaders': FROM_CACHE,
        ':jetpack:mergeExtDexDebug': FROM_CACHE,
        ':jetpack:mergeLibDexDebug': FROM_CACHE,
        ':jetpack:mergeProjectDexDebug': FROM_CACHE,
        ':jetpack:packageDebug': SUCCESS,
        ':jetpack:preBuild': UP_TO_DATE,
        ':jetpack:preDebugBuild': UP_TO_DATE,
        ':jetpack:processDebugJavaRes': NO_SOURCE,
        ':jetpack:processDebugManifest': FROM_CACHE,
        ':jetpack:processDebugResources': SUCCESS,
        ':jetpack:stripDebugDebugSymbols': NO_SOURCE,
        ':memory:assembleDebug': SUCCESS,
        ':memory:checkDebugDuplicateClasses': FROM_CACHE,
        ':memory:compileDebugAidl': NO_SOURCE,
        ':memory:compileDebugJavaWithJavac': FROM_CACHE,
        ':memory:compileDebugRenderscript': NO_SOURCE,
        ':memory:compileDebugShaders': NO_SOURCE,
        ':memory:compileDebugSources': UP_TO_DATE,
        ':memory:createDebugCompatibleScreenManifests': FROM_CACHE,
        ':memory:dexBuilderDebug': FROM_CACHE,
        ':memory:extractDeepLinksDebug': FROM_CACHE,
        ':memory:featureDebugWriter': SUCCESS,
        ':memory:generateDebugAssets': UP_TO_DATE,
        ':memory:generateDebugBuildConfig': FROM_CACHE,
        ':memory:generateDebugFeatureTransitiveDeps': FROM_CACHE,
        ':memory:generateDebugResValues': FROM_CACHE,
        ':memory:generateDebugResources': UP_TO_DATE,
        ':memory:javaPreCompileDebug': FROM_CACHE,
        ':memory:mergeDebugAssets': FROM_CACHE,
        ':memory:mergeDebugJavaResource': FROM_CACHE,
        ':memory:mergeDebugJniLibFolders': FROM_CACHE,
        ':memory:mergeDebugNativeLibs': FROM_CACHE,
        ':memory:mergeDebugResources': SUCCESS,
        ':memory:mergeDebugShaders': FROM_CACHE,
        ':memory:mergeExtDexDebug': FROM_CACHE,
        ':memory:mergeLibDexDebug': FROM_CACHE,
        ':memory:mergeProjectDexDebug': FROM_CACHE,
        ':memory:packageDebug': SUCCESS,
        ':memory:preBuild': UP_TO_DATE,
        ':memory:preDebugBuild': UP_TO_DATE,
        ':memory:processDebugJavaRes': NO_SOURCE,
        ':memory:processDebugManifest': FROM_CACHE,
        ':memory:processDebugResources': SUCCESS,
        ':memory:stripDebugDebugSymbols': NO_SOURCE,
        ':penguinswim:assembleDebug': SUCCESS,
        ':penguinswim:checkDebugDuplicateClasses': FROM_CACHE,
        ':penguinswim:compileDebugAidl': NO_SOURCE,
        ':penguinswim:compileDebugJavaWithJavac': FROM_CACHE,
        ':penguinswim:compileDebugRenderscript': NO_SOURCE,
        ':penguinswim:compileDebugShaders': NO_SOURCE,
        ':penguinswim:compileDebugSources': UP_TO_DATE,
        ':penguinswim:createDebugCompatibleScreenManifests': FROM_CACHE,
        ':penguinswim:dexBuilderDebug': FROM_CACHE,
        ':penguinswim:extractDeepLinksDebug': FROM_CACHE,
        ':penguinswim:featureDebugWriter': SUCCESS,
        ':penguinswim:generateDebugAssets': UP_TO_DATE,
        ':penguinswim:generateDebugBuildConfig': FROM_CACHE,
        ':penguinswim:generateDebugFeatureTransitiveDeps': FROM_CACHE,
        ':penguinswim:generateDebugResValues': FROM_CACHE,
        ':penguinswim:generateDebugResources': UP_TO_DATE,
        ':penguinswim:javaPreCompileDebug': FROM_CACHE,
        ':penguinswim:mergeDebugAssets': FROM_CACHE,
        ':penguinswim:mergeDebugJavaResource': FROM_CACHE,
        ':penguinswim:mergeDebugJniLibFolders': FROM_CACHE,
        ':penguinswim:mergeDebugNativeLibs': FROM_CACHE,
        ':penguinswim:mergeDebugResources': SUCCESS,
        ':penguinswim:mergeDebugShaders': FROM_CACHE,
        ':penguinswim:mergeExtDexDebug': FROM_CACHE,
        ':penguinswim:mergeLibDexDebug': FROM_CACHE,
        ':penguinswim:mergeProjectDexDebug': FROM_CACHE,
        ':penguinswim:packageDebug': SUCCESS,
        ':penguinswim:preBuild': UP_TO_DATE,
        ':penguinswim:preDebugBuild': UP_TO_DATE,
        ':penguinswim:processDebugJavaRes': NO_SOURCE,
        ':penguinswim:processDebugManifest': FROM_CACHE,
        ':penguinswim:processDebugResources': SUCCESS,
        ':penguinswim:stripDebugDebugSymbols': NO_SOURCE,
        ':playgames:assembleDebug': SUCCESS,
        ':playgames:bundleDebugAar': SUCCESS,
        ':playgames:bundleLibCompileToJarDebug': FROM_CACHE,
        ':playgames:bundleLibResDebug': NO_SOURCE,
        ':playgames:bundleLibRuntimeToJarDebug': FROM_CACHE,
        ':playgames:compileDebugAidl': NO_SOURCE,
        ':playgames:compileDebugJavaWithJavac': FROM_CACHE,
        ':playgames:compileDebugLibraryResources': SUCCESS,
        ':playgames:compileDebugRenderscript': NO_SOURCE,
        ':playgames:compileDebugShaders': NO_SOURCE,
        ':playgames:compileDebugSources': UP_TO_DATE,
        ':playgames:copyDebugJniLibsProjectAndLocalJars': FROM_CACHE,
        ':playgames:copyDebugJniLibsProjectOnly': FROM_CACHE,
        ':playgames:createFullJarDebug': FROM_CACHE,
        ':playgames:extractDebugAnnotations': FROM_CACHE,
        ':playgames:extractDeepLinksDebug': FROM_CACHE,
        ':playgames:generateDebugAssets': UP_TO_DATE,
        ':playgames:generateDebugBuildConfig': FROM_CACHE,
        ':playgames:generateDebugRFile': FROM_CACHE,
        ':playgames:generateDebugResValues': FROM_CACHE,
        ':playgames:generateDebugResources': UP_TO_DATE,
        ':playgames:javaPreCompileDebug': FROM_CACHE,
        ':playgames:mergeDebugConsumerProguardFiles': FROM_CACHE,
        ':playgames:mergeDebugGeneratedProguardFiles': FROM_CACHE,
        ':playgames:mergeDebugJavaResource': FROM_CACHE,
        ':playgames:mergeDebugJniLibFolders': FROM_CACHE,
        ':playgames:mergeDebugNativeLibs': FROM_CACHE,
        ':playgames:mergeDebugShaders': FROM_CACHE,
        ':playgames:packageDebugAssets': FROM_CACHE,
        ':playgames:packageDebugRenderscript': NO_SOURCE,
        ':playgames:packageDebugResources': FROM_CACHE,
        ':playgames:parseDebugLocalResources': FROM_CACHE,
        ':playgames:preBuild': UP_TO_DATE,
        ':playgames:preDebugBuild': UP_TO_DATE,
        ':playgames:prepareLintJarForPublish': SUCCESS,
        ':playgames:processDebugJavaRes': NO_SOURCE,
        ':playgames:processDebugManifest': FROM_CACHE,
        ':playgames:stripDebugDebugSymbols': NO_SOURCE,
        ':playgames:syncDebugLibJars': FROM_CACHE,
        ':presenttoss:assembleDebug': SUCCESS,
        ':presenttoss:checkDebugDuplicateClasses': FROM_CACHE,
        ':presenttoss:compileDebugAidl': NO_SOURCE,
        ':presenttoss:compileDebugJavaWithJavac': FROM_CACHE,
        ':presenttoss:compileDebugRenderscript': NO_SOURCE,
        ':presenttoss:compileDebugShaders': NO_SOURCE,
        ':presenttoss:compileDebugSources': UP_TO_DATE,
        ':presenttoss:createDebugCompatibleScreenManifests': FROM_CACHE,
        ':presenttoss:dexBuilderDebug': FROM_CACHE,
        ':presenttoss:extractDeepLinksDebug': FROM_CACHE,
        ':presenttoss:featureDebugWriter': SUCCESS,
        ':presenttoss:generateDebugAssets': UP_TO_DATE,
        ':presenttoss:generateDebugBuildConfig': FROM_CACHE,
        ':presenttoss:generateDebugFeatureTransitiveDeps': FROM_CACHE,
        ':presenttoss:generateDebugResValues': FROM_CACHE,
        ':presenttoss:generateDebugResources': UP_TO_DATE,
        ':presenttoss:javaPreCompileDebug': FROM_CACHE,
        ':presenttoss:mergeDebugAssets': FROM_CACHE,
        ':presenttoss:mergeDebugJavaResource': FROM_CACHE,
        ':presenttoss:mergeDebugJniLibFolders': FROM_CACHE,
        ':presenttoss:mergeDebugNativeLibs': FROM_CACHE,
        ':presenttoss:mergeDebugResources': SUCCESS,
        ':presenttoss:mergeDebugShaders': FROM_CACHE,
        ':presenttoss:mergeExtDexDebug': FROM_CACHE,
        ':presenttoss:mergeLibDexDebug': FROM_CACHE,
        ':presenttoss:mergeProjectDexDebug': FROM_CACHE,
        ':presenttoss:packageDebug': SUCCESS,
        ':presenttoss:preBuild': UP_TO_DATE,
        ':presenttoss:preDebugBuild': UP_TO_DATE,
        ':presenttoss:processDebugJavaRes': NO_SOURCE,
        ':presenttoss:processDebugManifest': FROM_CACHE,
        ':presenttoss:processDebugResources': SUCCESS,
        ':presenttoss:stripDebugDebugSymbols': NO_SOURCE,
        ':rocketsleigh:assembleDebug': SUCCESS,
        ':rocketsleigh:checkDebugDuplicateClasses': FROM_CACHE,
        ':rocketsleigh:compileDebugAidl': NO_SOURCE,
        ':rocketsleigh:compileDebugJavaWithJavac': FROM_CACHE,
        ':rocketsleigh:compileDebugKotlin': FROM_CACHE,
        ':rocketsleigh:compileDebugRenderscript': NO_SOURCE,
        ':rocketsleigh:compileDebugShaders': NO_SOURCE,
        ':rocketsleigh:compileDebugSources': UP_TO_DATE,
        ':rocketsleigh:createDebugCompatibleScreenManifests': FROM_CACHE,
        ':rocketsleigh:dexBuilderDebug': FROM_CACHE,
        ':rocketsleigh:extractDeepLinksDebug': FROM_CACHE,
        ':rocketsleigh:featureDebugWriter': SUCCESS,
        ':rocketsleigh:generateDebugAssets': UP_TO_DATE,
        ':rocketsleigh:generateDebugBuildConfig': FROM_CACHE,
        ':rocketsleigh:generateDebugFeatureTransitiveDeps': FROM_CACHE,
        ':rocketsleigh:generateDebugResValues': FROM_CACHE,
        ':rocketsleigh:generateDebugResources': UP_TO_DATE,
        ':rocketsleigh:javaPreCompileDebug': FROM_CACHE,
        ':rocketsleigh:mergeDebugAssets': FROM_CACHE,
        ':rocketsleigh:mergeDebugJavaResource': SUCCESS,
        ':rocketsleigh:mergeDebugJniLibFolders': FROM_CACHE,
        ':rocketsleigh:mergeDebugNativeLibs': FROM_CACHE,
        ':rocketsleigh:mergeDebugResources': SUCCESS,
        ':rocketsleigh:mergeDebugShaders': FROM_CACHE,
        ':rocketsleigh:mergeExtDexDebug': FROM_CACHE,
        ':rocketsleigh:mergeLibDexDebug': FROM_CACHE,
        ':rocketsleigh:mergeProjectDexDebug': FROM_CACHE,
        ':rocketsleigh:packageDebug': SUCCESS,
        ':rocketsleigh:preBuild': UP_TO_DATE,
        ':rocketsleigh:preDebugBuild': UP_TO_DATE,
        ':rocketsleigh:processDebugJavaRes': NO_SOURCE,
        ':rocketsleigh:processDebugManifest': FROM_CACHE,
        ':rocketsleigh:processDebugResources': SUCCESS,
        ':rocketsleigh:stripDebugDebugSymbols': NO_SOURCE,
        ':santa-tracker:assembleDebug': SUCCESS,
        ':santa-tracker:bundleDebugClasses': FROM_CACHE,
        ':santa-tracker:checkDebugDuplicateClasses': FROM_CACHE,
        ':santa-tracker:checkDebugLibraries': FROM_CACHE,
        ':santa-tracker:compileDebugAidl': NO_SOURCE,
        ':santa-tracker:compileDebugJavaWithJavac': FROM_CACHE,
        ':santa-tracker:compileDebugKotlin': FROM_CACHE,
        ':santa-tracker:compileDebugRenderscript': NO_SOURCE,
        ':santa-tracker:compileDebugShaders': NO_SOURCE,
        ':santa-tracker:compileDebugSources': UP_TO_DATE,
        ':santa-tracker:createDebugCompatibleScreenManifests': FROM_CACHE,
        ':santa-tracker:dexBuilderDebug': FROM_CACHE,
        ':santa-tracker:extractDeepLinksDebug': FROM_CACHE,
        ':santa-tracker:generateDebugAssets': UP_TO_DATE,
        ':santa-tracker:generateDebugBuildConfig': FROM_CACHE,
        ':santa-tracker:generateDebugFeatureMetadata': FROM_CACHE,
        ':santa-tracker:generateDebugFeatureTransitiveDeps': FROM_CACHE,
        ':santa-tracker:generateDebugResValues': FROM_CACHE,
        ':santa-tracker:generateDebugResources': UP_TO_DATE,
        ':santa-tracker:handleDebugMicroApk': FROM_CACHE,
        ':santa-tracker:javaPreCompileDebug': FROM_CACHE,
        ':santa-tracker:kaptDebugKotlin': FROM_CACHE,
        ':santa-tracker:kaptGenerateStubsDebugKotlin': FROM_CACHE,
        ':santa-tracker:mergeDebugAssets': FROM_CACHE,
        ':santa-tracker:mergeDebugJavaResource': SUCCESS,
        ':santa-tracker:mergeDebugJniLibFolders': FROM_CACHE,
        ':santa-tracker:mergeDebugNativeLibs': FROM_CACHE,
        ':santa-tracker:mergeDebugResources': SUCCESS,
        ':santa-tracker:mergeDebugShaders': FROM_CACHE,
        ':santa-tracker:mergeExtDexDebug': FROM_CACHE,
        ':santa-tracker:mergeLibDexDebug': FROM_CACHE,
        ':santa-tracker:mergeProjectDexDebug': FROM_CACHE,
        ':santa-tracker:packageDebug': SUCCESS,
        ':santa-tracker:preBuild': UP_TO_DATE,
        ':santa-tracker:preDebugBuild': FROM_CACHE,
        ':santa-tracker:processDebugJavaRes': NO_SOURCE,
        ':santa-tracker:processDebugManifest': FROM_CACHE,
        ':santa-tracker:processDebugResources': SUCCESS,
        ':santa-tracker:signingConfigWriterDebug': FROM_CACHE,
        ':santa-tracker:stripDebugDebugSymbols': NO_SOURCE,
        ':santa-tracker:validateSigningDebug': FROM_CACHE,
        ':santa-tracker:writeDebugModuleMetadata': SUCCESS,
        ':snowballrun:assembleDebug': SUCCESS,
        ':snowballrun:checkDebugDuplicateClasses': FROM_CACHE,
        ':snowballrun:compileDebugAidl': NO_SOURCE,
        ':snowballrun:compileDebugJavaWithJavac': FROM_CACHE,
        ':snowballrun:compileDebugRenderscript': NO_SOURCE,
        ':snowballrun:compileDebugShaders': NO_SOURCE,
        ':snowballrun:compileDebugSources': UP_TO_DATE,
        ':snowballrun:createDebugCompatibleScreenManifests': FROM_CACHE,
        ':snowballrun:dexBuilderDebug': FROM_CACHE,
        ':snowballrun:extractDeepLinksDebug': FROM_CACHE,
        ':snowballrun:featureDebugWriter': SUCCESS,
        ':snowballrun:generateDebugAssets': UP_TO_DATE,
        ':snowballrun:generateDebugBuildConfig': FROM_CACHE,
        ':snowballrun:generateDebugFeatureTransitiveDeps': FROM_CACHE,
        ':snowballrun:generateDebugResValues': FROM_CACHE,
        ':snowballrun:generateDebugResources': UP_TO_DATE,
        ':snowballrun:javaPreCompileDebug': FROM_CACHE,
        ':snowballrun:mergeDebugAssets': FROM_CACHE,
        ':snowballrun:mergeDebugJavaResource': FROM_CACHE,
        ':snowballrun:mergeDebugJniLibFolders': FROM_CACHE,
        ':snowballrun:mergeDebugNativeLibs': FROM_CACHE,
        ':snowballrun:mergeDebugResources': SUCCESS,
        ':snowballrun:mergeDebugShaders': FROM_CACHE,
        ':snowballrun:mergeExtDexDebug': FROM_CACHE,
        ':snowballrun:mergeLibDexDebug': FROM_CACHE,
        ':snowballrun:mergeProjectDexDebug': FROM_CACHE,
        ':snowballrun:packageDebug': SUCCESS,
        ':snowballrun:preBuild': UP_TO_DATE,
        ':snowballrun:preDebugBuild': UP_TO_DATE,
        ':snowballrun:processDebugJavaRes': NO_SOURCE,
        ':snowballrun:processDebugManifest': FROM_CACHE,
        ':snowballrun:processDebugResources': SUCCESS,
        ':snowballrun:stripDebugDebugSymbols': NO_SOURCE,
        ':tracker:assembleDebug': SUCCESS,
        ':tracker:bundleDebugAar': SUCCESS,
        ':tracker:bundleLibCompileToJarDebug': FROM_CACHE,
        ':tracker:bundleLibResDebug': FROM_CACHE,
        ':tracker:bundleLibRuntimeToJarDebug': FROM_CACHE,
        ':tracker:compileDebugAidl': NO_SOURCE,
        ':tracker:compileDebugJavaWithJavac': SUCCESS,
        ':tracker:compileDebugKotlin': FROM_CACHE,
        ':tracker:compileDebugLibraryResources': SUCCESS,
        ':tracker:compileDebugRenderscript': NO_SOURCE,
        ':tracker:compileDebugShaders': NO_SOURCE,
        ':tracker:compileDebugSources': SUCCESS,
        ':tracker:copyDebugJniLibsProjectAndLocalJars': FROM_CACHE,
        ':tracker:copyDebugJniLibsProjectOnly': FROM_CACHE,
        ':tracker:createFullJarDebug': FROM_CACHE,
        ':tracker:extractDebugAnnotations': FROM_CACHE,
        ':tracker:extractDeepLinksDebug': FROM_CACHE,
        ':tracker:generateDebugAssets': UP_TO_DATE,
        ':tracker:generateDebugBuildConfig': FROM_CACHE,
        ':tracker:generateDebugRFile': FROM_CACHE,
        ':tracker:generateDebugResValues': FROM_CACHE,
        ':tracker:generateDebugResources': UP_TO_DATE,
        ':tracker:javaPreCompileDebug': FROM_CACHE,
        ':tracker:kaptDebugKotlin': SUCCESS,
        ':tracker:kaptGenerateStubsDebugKotlin': SUCCESS,
        ':tracker:mergeDebugConsumerProguardFiles': FROM_CACHE,
        ':tracker:mergeDebugGeneratedProguardFiles': FROM_CACHE,
        ':tracker:mergeDebugJavaResource': SUCCESS,
        ':tracker:mergeDebugJniLibFolders': FROM_CACHE,
        ':tracker:mergeDebugNativeLibs': FROM_CACHE,
        ':tracker:mergeDebugResources': FROM_CACHE,
        ':tracker:mergeDebugShaders': FROM_CACHE,
        ':tracker:packageDebugAssets': FROM_CACHE,
        ':tracker:packageDebugRenderscript': NO_SOURCE,
        ':tracker:packageDebugResources': FROM_CACHE,
        ':tracker:parseDebugLocalResources': FROM_CACHE,
        ':tracker:preBuild': UP_TO_DATE,
        ':tracker:preDebugBuild': UP_TO_DATE,
        ':tracker:prepareLintJarForPublish': SUCCESS,
        ':tracker:processDebugJavaRes': NO_SOURCE,
        ':tracker:processDebugManifest': FROM_CACHE,
        ':tracker:stripDebugDebugSymbols': NO_SOURCE,
        ':tracker:syncDebugLibJars': SUCCESS,
        ':wearable:assembleDebug': SUCCESS,
        ':wearable:checkDebugDuplicateClasses': FROM_CACHE,
        ':wearable:compileDebugAidl': NO_SOURCE,
        ':wearable:compileDebugJavaWithJavac': FROM_CACHE,
        ':wearable:compileDebugKotlin': FROM_CACHE,
        ':wearable:compileDebugRenderscript': NO_SOURCE,
        ':wearable:compileDebugShaders': NO_SOURCE,
        ':wearable:compileDebugSources': UP_TO_DATE,
        ':wearable:createDebugCompatibleScreenManifests': FROM_CACHE,
        ':wearable:dexBuilderDebug': FROM_CACHE,
        ':wearable:extractDeepLinksDebug': FROM_CACHE,
        ':wearable:generateDebugAssets': UP_TO_DATE,
        ':wearable:generateDebugBuildConfig': FROM_CACHE,
        ':wearable:generateDebugResValues': FROM_CACHE,
        ':wearable:generateDebugResources': UP_TO_DATE,
        ':wearable:javaPreCompileDebug': FROM_CACHE,
        ':wearable:kaptDebugKotlin': FROM_CACHE,
        ':wearable:kaptGenerateStubsDebugKotlin': FROM_CACHE,
        ':wearable:mergeDebugAssets': FROM_CACHE,
        ':wearable:mergeDebugJavaResource': SUCCESS,
        ':wearable:mergeDebugJniLibFolders': FROM_CACHE,
        ':wearable:mergeDebugNativeLibs': FROM_CACHE,
        ':wearable:mergeDebugResources': SUCCESS,
        ':wearable:mergeDebugShaders': FROM_CACHE,
        ':wearable:mergeExtDexDebug': FROM_CACHE,
        ':wearable:mergeLibDexDebug': FROM_CACHE,
        ':wearable:mergeProjectDexDebug': FROM_CACHE,
        ':wearable:packageDebug': SUCCESS,
        ':wearable:preBuild': UP_TO_DATE,
        ':wearable:preDebugBuild': UP_TO_DATE,
        ':wearable:processDebugJavaRes': NO_SOURCE,
        ':wearable:processDebugManifest': FROM_CACHE,
        ':wearable:processDebugResources': SUCCESS,
        ':wearable:stripDebugDebugSymbols': NO_SOURCE,
        ':wearable:validateSigningDebug': FROM_CACHE,
    ]

    private static final EXPECTED_RESULTS_3_6 = [
        ':cityquiz:assembleDebug': SUCCESS,
        ':cityquiz:checkDebugDuplicateClasses': FROM_CACHE,
        ':cityquiz:compileDebugAidl': NO_SOURCE,
        ':cityquiz:compileDebugJavaWithJavac': FROM_CACHE,
        ':cityquiz:compileDebugKotlin': FROM_CACHE,
        ':cityquiz:compileDebugRenderscript': NO_SOURCE,
        ':cityquiz:compileDebugShaders': FROM_CACHE,
        ':cityquiz:compileDebugSources': UP_TO_DATE,
        ':cityquiz:createDebugCompatibleScreenManifests': FROM_CACHE,
        ':cityquiz:dexBuilderDebug': FROM_CACHE,
        ':cityquiz:extractDeepLinksDebug': FROM_CACHE,
        ':cityquiz:featureDebugWriter': SUCCESS,
        ':cityquiz:generateDebugAssets': UP_TO_DATE,
        ':cityquiz:generateDebugBuildConfig': FROM_CACHE,
        ':cityquiz:generateDebugFeatureTransitiveDeps': FROM_CACHE,
        ':cityquiz:generateDebugResValues': FROM_CACHE,
        ':cityquiz:generateDebugResources': UP_TO_DATE,
        ':cityquiz:javaPreCompileDebug': FROM_CACHE,
        ':cityquiz:mainApkListPersistenceDebug': FROM_CACHE,
        ':cityquiz:mergeDebugAssets': FROM_CACHE,
        ':cityquiz:mergeDebugJavaResource': SUCCESS,
        ':cityquiz:mergeDebugJniLibFolders': FROM_CACHE,
        ':cityquiz:mergeDebugNativeLibs': FROM_CACHE,
        ':cityquiz:mergeDebugResources': FROM_CACHE,
        ':cityquiz:mergeDebugShaders': FROM_CACHE,
        ':cityquiz:mergeExtDexDebug': FROM_CACHE,
        ':cityquiz:mergeLibDexDebug': FROM_CACHE,
        ':cityquiz:mergeProjectDexDebug': FROM_CACHE,
        ':cityquiz:packageDebug': SUCCESS,
        ':cityquiz:preBuild': UP_TO_DATE,
        ':cityquiz:preDebugBuild': UP_TO_DATE,
        ':cityquiz:processDebugJavaRes': NO_SOURCE,
        ':cityquiz:processDebugManifest': FROM_CACHE,
        ':cityquiz:processDebugResources': FROM_CACHE,
        ':cityquiz:stripDebugDebugSymbols': FROM_CACHE,
        ':common:assembleDebug': SUCCESS,
        ':common:bundleDebugAar': SUCCESS,
        ':common:bundleLibCompileDebug': SUCCESS,
        ':common:bundleLibResDebug': SUCCESS,
        ':common:bundleLibRuntimeDebug': SUCCESS,
        ':common:compileDebugAidl': NO_SOURCE,
        ':common:compileDebugJavaWithJavac': FROM_CACHE,
        ':common:compileDebugKotlin': FROM_CACHE,
        ':common:compileDebugLibraryResources': FROM_CACHE,
        ':common:compileDebugRenderscript': NO_SOURCE,
        ':common:compileDebugShaders': FROM_CACHE,
        ':common:compileDebugSources': UP_TO_DATE,
        ':common:copyDebugJniLibsProjectAndLocalJars': FROM_CACHE,
        ':common:copyDebugJniLibsProjectOnly': FROM_CACHE,
        ':common:extractDebugAnnotations': FROM_CACHE,
        ':common:extractDeepLinksDebug': FROM_CACHE,
        ':common:generateDebugAssets': UP_TO_DATE,
        ':common:generateDebugBuildConfig': FROM_CACHE,
        ':common:generateDebugRFile': FROM_CACHE,
        ':common:generateDebugResValues': FROM_CACHE,
        ':common:generateDebugResources': UP_TO_DATE,
        ':common:javaPreCompileDebug': FROM_CACHE,
        ':common:mergeDebugConsumerProguardFiles': SUCCESS,
        ':common:mergeDebugGeneratedProguardFiles': SUCCESS,
        ':common:mergeDebugJavaResource': SUCCESS,
        ':common:mergeDebugJniLibFolders': FROM_CACHE,
        ':common:mergeDebugNativeLibs': FROM_CACHE,
        ':common:mergeDebugShaders': FROM_CACHE,
        ':common:packageDebugAssets': FROM_CACHE,
        ':common:packageDebugRenderscript': NO_SOURCE,
        ':common:packageDebugResources': FROM_CACHE,
        ':common:parseDebugLocalResources': FROM_CACHE,
        ':common:preBuild': UP_TO_DATE,
        ':common:preDebugBuild': UP_TO_DATE,
        ':common:prepareLintJarForPublish': SUCCESS,
        ':common:processDebugJavaRes': NO_SOURCE,
        ':common:processDebugManifest': FROM_CACHE,
        ':common:stripDebugDebugSymbols': FROM_CACHE,
        ':common:syncDebugLibJars': SUCCESS,
        ':dasherdancer:assembleDebug': SUCCESS,
        ':dasherdancer:checkDebugDuplicateClasses': FROM_CACHE,
        ':dasherdancer:compileDebugAidl': NO_SOURCE,
        ':dasherdancer:compileDebugJavaWithJavac': FROM_CACHE,
        ':dasherdancer:compileDebugKotlin': FROM_CACHE,
        ':dasherdancer:compileDebugRenderscript': NO_SOURCE,
        ':dasherdancer:compileDebugShaders': FROM_CACHE,
        ':dasherdancer:compileDebugSources': UP_TO_DATE,
        ':dasherdancer:createDebugCompatibleScreenManifests': FROM_CACHE,
        ':dasherdancer:dexBuilderDebug': FROM_CACHE,
        ':dasherdancer:extractDeepLinksDebug': FROM_CACHE,
        ':dasherdancer:featureDebugWriter': SUCCESS,
        ':dasherdancer:generateDebugAssets': UP_TO_DATE,
        ':dasherdancer:generateDebugBuildConfig': FROM_CACHE,
        ':dasherdancer:generateDebugFeatureTransitiveDeps': FROM_CACHE,
        ':dasherdancer:generateDebugResValues': FROM_CACHE,
        ':dasherdancer:generateDebugResources': UP_TO_DATE,
        ':dasherdancer:javaPreCompileDebug': FROM_CACHE,
        ':dasherdancer:mainApkListPersistenceDebug': FROM_CACHE,
        ':dasherdancer:mergeDebugAssets': FROM_CACHE,
        ':dasherdancer:mergeDebugJavaResource': SUCCESS,
        ':dasherdancer:mergeDebugJniLibFolders': FROM_CACHE,
        ':dasherdancer:mergeDebugNativeLibs': FROM_CACHE,
        ':dasherdancer:mergeDebugResources': FROM_CACHE,
        ':dasherdancer:mergeDebugShaders': FROM_CACHE,
        ':dasherdancer:mergeExtDexDebug': FROM_CACHE,
        ':dasherdancer:mergeLibDexDebug': FROM_CACHE,
        ':dasherdancer:mergeProjectDexDebug': FROM_CACHE,
        ':dasherdancer:packageDebug': SUCCESS,
        ':dasherdancer:preBuild': UP_TO_DATE,
        ':dasherdancer:preDebugBuild': UP_TO_DATE,
        ':dasherdancer:processDebugJavaRes': NO_SOURCE,
        ':dasherdancer:processDebugManifest': FROM_CACHE,
        ':dasherdancer:processDebugResources': FROM_CACHE,
        ':dasherdancer:stripDebugDebugSymbols': FROM_CACHE,
        ':doodles-lib:assembleDebug': SUCCESS,
        ':doodles-lib:bundleDebugAar': SUCCESS,
        ':doodles-lib:bundleLibCompileDebug': SUCCESS,
        ':doodles-lib:bundleLibResDebug': SUCCESS,
        ':doodles-lib:bundleLibRuntimeDebug': SUCCESS,
        ':doodles-lib:compileDebugAidl': NO_SOURCE,
        ':doodles-lib:compileDebugJavaWithJavac': FROM_CACHE,
        ':doodles-lib:compileDebugLibraryResources': FROM_CACHE,
        ':doodles-lib:compileDebugRenderscript': NO_SOURCE,
        ':doodles-lib:compileDebugShaders': FROM_CACHE,
        ':doodles-lib:compileDebugSources': UP_TO_DATE,
        ':doodles-lib:copyDebugJniLibsProjectAndLocalJars': FROM_CACHE,
        ':doodles-lib:copyDebugJniLibsProjectOnly': FROM_CACHE,
        ':doodles-lib:extractDebugAnnotations': FROM_CACHE,
        ':doodles-lib:extractDeepLinksDebug': FROM_CACHE,
        ':doodles-lib:generateDebugAssets': UP_TO_DATE,
        ':doodles-lib:generateDebugBuildConfig': FROM_CACHE,
        ':doodles-lib:generateDebugRFile': FROM_CACHE,
        ':doodles-lib:generateDebugResValues': FROM_CACHE,
        ':doodles-lib:generateDebugResources': UP_TO_DATE,
        ':doodles-lib:javaPreCompileDebug': FROM_CACHE,
        ':doodles-lib:mergeDebugConsumerProguardFiles': SUCCESS,
        ':doodles-lib:mergeDebugGeneratedProguardFiles': SUCCESS,
        ':doodles-lib:mergeDebugJavaResource': FROM_CACHE,
        ':doodles-lib:mergeDebugJniLibFolders': FROM_CACHE,
        ':doodles-lib:mergeDebugNativeLibs': FROM_CACHE,
        ':doodles-lib:mergeDebugShaders': FROM_CACHE,
        ':doodles-lib:packageDebugAssets': FROM_CACHE,
        ':doodles-lib:packageDebugRenderscript': NO_SOURCE,
        ':doodles-lib:packageDebugResources': FROM_CACHE,
        ':doodles-lib:parseDebugLocalResources': FROM_CACHE,
        ':doodles-lib:preBuild': UP_TO_DATE,
        ':doodles-lib:preDebugBuild': UP_TO_DATE,
        ':doodles-lib:prepareLintJarForPublish': SUCCESS,
        ':doodles-lib:processDebugJavaRes': NO_SOURCE,
        ':doodles-lib:processDebugManifest': FROM_CACHE,
        ':doodles-lib:stripDebugDebugSymbols': FROM_CACHE,
        ':doodles-lib:syncDebugLibJars': FROM_CACHE,
        ':gumball:assembleDebug': SUCCESS,
        ':gumball:checkDebugDuplicateClasses': FROM_CACHE,
        ':gumball:compileDebugAidl': NO_SOURCE,
        ':gumball:compileDebugJavaWithJavac': FROM_CACHE,
        ':gumball:compileDebugRenderscript': NO_SOURCE,
        ':gumball:compileDebugShaders': FROM_CACHE,
        ':gumball:compileDebugSources': UP_TO_DATE,
        ':gumball:createDebugCompatibleScreenManifests': FROM_CACHE,
        ':gumball:dexBuilderDebug': FROM_CACHE,
        ':gumball:extractDeepLinksDebug': FROM_CACHE,
        ':gumball:featureDebugWriter': SUCCESS,
        ':gumball:generateDebugAssets': UP_TO_DATE,
        ':gumball:generateDebugBuildConfig': FROM_CACHE,
        ':gumball:generateDebugFeatureTransitiveDeps': FROM_CACHE,
        ':gumball:generateDebugResValues': FROM_CACHE,
        ':gumball:generateDebugResources': UP_TO_DATE,
        ':gumball:javaPreCompileDebug': FROM_CACHE,
        ':gumball:mainApkListPersistenceDebug': FROM_CACHE,
        ':gumball:mergeDebugAssets': FROM_CACHE,
        ':gumball:mergeDebugJavaResource': FROM_CACHE,
        ':gumball:mergeDebugJniLibFolders': FROM_CACHE,
        ':gumball:mergeDebugNativeLibs': FROM_CACHE,
        ':gumball:mergeDebugResources': FROM_CACHE,
        ':gumball:mergeDebugShaders': FROM_CACHE,
        ':gumball:mergeExtDexDebug': FROM_CACHE,
        ':gumball:mergeLibDexDebug': FROM_CACHE,
        ':gumball:mergeProjectDexDebug': FROM_CACHE,
        ':gumball:packageDebug': SUCCESS,
        ':gumball:preBuild': UP_TO_DATE,
        ':gumball:preDebugBuild': UP_TO_DATE,
        ':gumball:processDebugJavaRes': NO_SOURCE,
        ':gumball:processDebugManifest': FROM_CACHE,
        ':gumball:processDebugResources': FROM_CACHE,
        ':gumball:stripDebugDebugSymbols': FROM_CACHE,
        ':jetpack:assembleDebug': SUCCESS,
        ':jetpack:checkDebugDuplicateClasses': FROM_CACHE,
        ':jetpack:compileDebugAidl': NO_SOURCE,
        ':jetpack:compileDebugJavaWithJavac': FROM_CACHE,
        ':jetpack:compileDebugKotlin': FROM_CACHE,
        ':jetpack:compileDebugRenderscript': NO_SOURCE,
        ':jetpack:compileDebugShaders': FROM_CACHE,
        ':jetpack:compileDebugSources': UP_TO_DATE,
        ':jetpack:createDebugCompatibleScreenManifests': FROM_CACHE,
        ':jetpack:dexBuilderDebug': FROM_CACHE,
        ':jetpack:extractDeepLinksDebug': FROM_CACHE,
        ':jetpack:featureDebugWriter': SUCCESS,
        ':jetpack:generateDebugAssets': UP_TO_DATE,
        ':jetpack:generateDebugBuildConfig': FROM_CACHE,
        ':jetpack:generateDebugFeatureTransitiveDeps': FROM_CACHE,
        ':jetpack:generateDebugResValues': FROM_CACHE,
        ':jetpack:generateDebugResources': UP_TO_DATE,
        ':jetpack:javaPreCompileDebug': FROM_CACHE,
        ':jetpack:mainApkListPersistenceDebug': FROM_CACHE,
        ':jetpack:mergeDebugAssets': FROM_CACHE,
        ':jetpack:mergeDebugJavaResource': SUCCESS,
        ':jetpack:mergeDebugJniLibFolders': FROM_CACHE,
        ':jetpack:mergeDebugNativeLibs': FROM_CACHE,
        ':jetpack:mergeDebugResources': FROM_CACHE,
        ':jetpack:mergeDebugShaders': FROM_CACHE,
        ':jetpack:mergeExtDexDebug': FROM_CACHE,
        ':jetpack:mergeLibDexDebug': FROM_CACHE,
        ':jetpack:mergeProjectDexDebug': FROM_CACHE,
        ':jetpack:packageDebug': SUCCESS,
        ':jetpack:preBuild': UP_TO_DATE,
        ':jetpack:preDebugBuild': UP_TO_DATE,
        ':jetpack:processDebugJavaRes': NO_SOURCE,
        ':jetpack:processDebugManifest': FROM_CACHE,
        ':jetpack:processDebugResources': FROM_CACHE,
        ':jetpack:stripDebugDebugSymbols': FROM_CACHE,
        ':memory:assembleDebug': SUCCESS,
        ':memory:checkDebugDuplicateClasses': FROM_CACHE,
        ':memory:compileDebugAidl': NO_SOURCE,
        ':memory:compileDebugJavaWithJavac': FROM_CACHE,
        ':memory:compileDebugRenderscript': NO_SOURCE,
        ':memory:compileDebugShaders': FROM_CACHE,
        ':memory:compileDebugSources': UP_TO_DATE,
        ':memory:createDebugCompatibleScreenManifests': FROM_CACHE,
        ':memory:dexBuilderDebug': FROM_CACHE,
        ':memory:extractDeepLinksDebug': FROM_CACHE,
        ':memory:featureDebugWriter': SUCCESS,
        ':memory:generateDebugAssets': UP_TO_DATE,
        ':memory:generateDebugBuildConfig': FROM_CACHE,
        ':memory:generateDebugFeatureTransitiveDeps': FROM_CACHE,
        ':memory:generateDebugResValues': FROM_CACHE,
        ':memory:generateDebugResources': UP_TO_DATE,
        ':memory:javaPreCompileDebug': FROM_CACHE,
        ':memory:mainApkListPersistenceDebug': FROM_CACHE,
        ':memory:mergeDebugAssets': FROM_CACHE,
        ':memory:mergeDebugJavaResource': FROM_CACHE,
        ':memory:mergeDebugJniLibFolders': FROM_CACHE,
        ':memory:mergeDebugNativeLibs': FROM_CACHE,
        ':memory:mergeDebugResources': FROM_CACHE,
        ':memory:mergeDebugShaders': FROM_CACHE,
        ':memory:mergeExtDexDebug': FROM_CACHE,
        ':memory:mergeLibDexDebug': FROM_CACHE,
        ':memory:mergeProjectDexDebug': FROM_CACHE,
        ':memory:packageDebug': SUCCESS,
        ':memory:preBuild': UP_TO_DATE,
        ':memory:preDebugBuild': UP_TO_DATE,
        ':memory:processDebugJavaRes': NO_SOURCE,
        ':memory:processDebugManifest': FROM_CACHE,
        ':memory:processDebugResources': FROM_CACHE,
        ':memory:stripDebugDebugSymbols': FROM_CACHE,
        ':penguinswim:assembleDebug': SUCCESS,
        ':penguinswim:checkDebugDuplicateClasses': FROM_CACHE,
        ':penguinswim:compileDebugAidl': NO_SOURCE,
        ':penguinswim:compileDebugJavaWithJavac': FROM_CACHE,
        ':penguinswim:compileDebugRenderscript': NO_SOURCE,
        ':penguinswim:compileDebugShaders': FROM_CACHE,
        ':penguinswim:compileDebugSources': UP_TO_DATE,
        ':penguinswim:createDebugCompatibleScreenManifests': FROM_CACHE,
        ':penguinswim:dexBuilderDebug': FROM_CACHE,
        ':penguinswim:extractDeepLinksDebug': FROM_CACHE,
        ':penguinswim:featureDebugWriter': SUCCESS,
        ':penguinswim:generateDebugAssets': UP_TO_DATE,
        ':penguinswim:generateDebugBuildConfig': FROM_CACHE,
        ':penguinswim:generateDebugFeatureTransitiveDeps': FROM_CACHE,
        ':penguinswim:generateDebugResValues': FROM_CACHE,
        ':penguinswim:generateDebugResources': UP_TO_DATE,
        ':penguinswim:javaPreCompileDebug': FROM_CACHE,
        ':penguinswim:mainApkListPersistenceDebug': FROM_CACHE,
        ':penguinswim:mergeDebugAssets': FROM_CACHE,
        ':penguinswim:mergeDebugJavaResource': FROM_CACHE,
        ':penguinswim:mergeDebugJniLibFolders': FROM_CACHE,
        ':penguinswim:mergeDebugNativeLibs': FROM_CACHE,
        ':penguinswim:mergeDebugResources': FROM_CACHE,
        ':penguinswim:mergeDebugShaders': FROM_CACHE,
        ':penguinswim:mergeExtDexDebug': FROM_CACHE,
        ':penguinswim:mergeLibDexDebug': FROM_CACHE,
        ':penguinswim:mergeProjectDexDebug': FROM_CACHE,
        ':penguinswim:packageDebug': SUCCESS,
        ':penguinswim:preBuild': UP_TO_DATE,
        ':penguinswim:preDebugBuild': UP_TO_DATE,
        ':penguinswim:processDebugJavaRes': NO_SOURCE,
        ':penguinswim:processDebugManifest': FROM_CACHE,
        ':penguinswim:processDebugResources': FROM_CACHE,
        ':penguinswim:stripDebugDebugSymbols': FROM_CACHE,
        ':playgames:assembleDebug': SUCCESS,
        ':playgames:bundleDebugAar': SUCCESS,
        ':playgames:bundleLibCompileDebug': SUCCESS,
        ':playgames:bundleLibResDebug': SUCCESS,
        ':playgames:bundleLibRuntimeDebug': SUCCESS,
        ':playgames:compileDebugAidl': NO_SOURCE,
        ':playgames:compileDebugJavaWithJavac': FROM_CACHE,
        ':playgames:compileDebugLibraryResources': FROM_CACHE,
        ':playgames:compileDebugRenderscript': NO_SOURCE,
        ':playgames:compileDebugShaders': FROM_CACHE,
        ':playgames:compileDebugSources': UP_TO_DATE,
        ':playgames:copyDebugJniLibsProjectAndLocalJars': FROM_CACHE,
        ':playgames:copyDebugJniLibsProjectOnly': FROM_CACHE,
        ':playgames:extractDebugAnnotations': FROM_CACHE,
        ':playgames:extractDeepLinksDebug': FROM_CACHE,
        ':playgames:generateDebugAssets': UP_TO_DATE,
        ':playgames:generateDebugBuildConfig': FROM_CACHE,
        ':playgames:generateDebugRFile': FROM_CACHE,
        ':playgames:generateDebugResValues': FROM_CACHE,
        ':playgames:generateDebugResources': UP_TO_DATE,
        ':playgames:javaPreCompileDebug': FROM_CACHE,
        ':playgames:mergeDebugConsumerProguardFiles': SUCCESS,
        ':playgames:mergeDebugGeneratedProguardFiles': SUCCESS,
        ':playgames:mergeDebugJavaResource': FROM_CACHE,
        ':playgames:mergeDebugJniLibFolders': FROM_CACHE,
        ':playgames:mergeDebugNativeLibs': FROM_CACHE,
        ':playgames:mergeDebugShaders': FROM_CACHE,
        ':playgames:packageDebugAssets': FROM_CACHE,
        ':playgames:packageDebugRenderscript': NO_SOURCE,
        ':playgames:packageDebugResources': FROM_CACHE,
        ':playgames:parseDebugLocalResources': FROM_CACHE,
        ':playgames:preBuild': UP_TO_DATE,
        ':playgames:preDebugBuild': UP_TO_DATE,
        ':playgames:prepareLintJarForPublish': SUCCESS,
        ':playgames:processDebugJavaRes': NO_SOURCE,
        ':playgames:processDebugManifest': FROM_CACHE,
        ':playgames:stripDebugDebugSymbols': FROM_CACHE,
        ':playgames:syncDebugLibJars': FROM_CACHE,
        ':presenttoss:assembleDebug': SUCCESS,
        ':presenttoss:checkDebugDuplicateClasses': FROM_CACHE,
        ':presenttoss:compileDebugAidl': NO_SOURCE,
        ':presenttoss:compileDebugJavaWithJavac': FROM_CACHE,
        ':presenttoss:compileDebugRenderscript': NO_SOURCE,
        ':presenttoss:compileDebugShaders': FROM_CACHE,
        ':presenttoss:compileDebugSources': UP_TO_DATE,
        ':presenttoss:createDebugCompatibleScreenManifests': FROM_CACHE,
        ':presenttoss:dexBuilderDebug': FROM_CACHE,
        ':presenttoss:extractDeepLinksDebug': FROM_CACHE,
        ':presenttoss:featureDebugWriter': SUCCESS,
        ':presenttoss:generateDebugAssets': UP_TO_DATE,
        ':presenttoss:generateDebugBuildConfig': FROM_CACHE,
        ':presenttoss:generateDebugFeatureTransitiveDeps': FROM_CACHE,
        ':presenttoss:generateDebugResValues': FROM_CACHE,
        ':presenttoss:generateDebugResources': UP_TO_DATE,
        ':presenttoss:javaPreCompileDebug': FROM_CACHE,
        ':presenttoss:mainApkListPersistenceDebug': FROM_CACHE,
        ':presenttoss:mergeDebugAssets': FROM_CACHE,
        ':presenttoss:mergeDebugJavaResource': FROM_CACHE,
        ':presenttoss:mergeDebugJniLibFolders': FROM_CACHE,
        ':presenttoss:mergeDebugNativeLibs': FROM_CACHE,
        ':presenttoss:mergeDebugResources': FROM_CACHE,
        ':presenttoss:mergeDebugShaders': FROM_CACHE,
        ':presenttoss:mergeExtDexDebug': FROM_CACHE,
        ':presenttoss:mergeLibDexDebug': FROM_CACHE,
        ':presenttoss:mergeProjectDexDebug': FROM_CACHE,
        ':presenttoss:packageDebug': SUCCESS,
        ':presenttoss:preBuild': UP_TO_DATE,
        ':presenttoss:preDebugBuild': UP_TO_DATE,
        ':presenttoss:processDebugJavaRes': NO_SOURCE,
        ':presenttoss:processDebugManifest': FROM_CACHE,
        ':presenttoss:processDebugResources': FROM_CACHE,
        ':presenttoss:stripDebugDebugSymbols': FROM_CACHE,
        ':rocketsleigh:assembleDebug': SUCCESS,
        ':rocketsleigh:checkDebugDuplicateClasses': FROM_CACHE,
        ':rocketsleigh:compileDebugAidl': NO_SOURCE,
        ':rocketsleigh:compileDebugJavaWithJavac': FROM_CACHE,
        ':rocketsleigh:compileDebugKotlin': FROM_CACHE,
        ':rocketsleigh:compileDebugRenderscript': NO_SOURCE,
        ':rocketsleigh:compileDebugShaders': FROM_CACHE,
        ':rocketsleigh:compileDebugSources': UP_TO_DATE,
        ':rocketsleigh:createDebugCompatibleScreenManifests': FROM_CACHE,
        ':rocketsleigh:dexBuilderDebug': FROM_CACHE,
        ':rocketsleigh:extractDeepLinksDebug': FROM_CACHE,
        ':rocketsleigh:featureDebugWriter': SUCCESS,
        ':rocketsleigh:generateDebugAssets': UP_TO_DATE,
        ':rocketsleigh:generateDebugBuildConfig': FROM_CACHE,
        ':rocketsleigh:generateDebugFeatureTransitiveDeps': FROM_CACHE,
        ':rocketsleigh:generateDebugResValues': FROM_CACHE,
        ':rocketsleigh:generateDebugResources': UP_TO_DATE,
        ':rocketsleigh:javaPreCompileDebug': FROM_CACHE,
        ':rocketsleigh:mainApkListPersistenceDebug': FROM_CACHE,
        ':rocketsleigh:mergeDebugAssets': FROM_CACHE,
        ':rocketsleigh:mergeDebugJavaResource': SUCCESS,
        ':rocketsleigh:mergeDebugJniLibFolders': FROM_CACHE,
        ':rocketsleigh:mergeDebugNativeLibs': FROM_CACHE,
        ':rocketsleigh:mergeDebugResources': FROM_CACHE,
        ':rocketsleigh:mergeDebugShaders': FROM_CACHE,
        ':rocketsleigh:mergeExtDexDebug': FROM_CACHE,
        ':rocketsleigh:mergeLibDexDebug': FROM_CACHE,
        ':rocketsleigh:mergeProjectDexDebug': FROM_CACHE,
        ':rocketsleigh:packageDebug': SUCCESS,
        ':rocketsleigh:preBuild': UP_TO_DATE,
        ':rocketsleigh:preDebugBuild': UP_TO_DATE,
        ':rocketsleigh:processDebugJavaRes': NO_SOURCE,
        ':rocketsleigh:processDebugManifest': FROM_CACHE,
        ':rocketsleigh:processDebugResources': FROM_CACHE,
        ':rocketsleigh:stripDebugDebugSymbols': FROM_CACHE,
        ':santa-tracker:assembleDebug': SUCCESS,
        ':santa-tracker:bundleDebugClasses': SUCCESS,
        ':santa-tracker:checkDebugDuplicateClasses': FROM_CACHE,
        ':santa-tracker:checkDebugLibraries': FROM_CACHE,
        ':santa-tracker:compileDebugAidl': NO_SOURCE,
        ':santa-tracker:compileDebugJavaWithJavac': FROM_CACHE,
        ':santa-tracker:compileDebugKotlin': FROM_CACHE,
        ':santa-tracker:compileDebugRenderscript': NO_SOURCE,
        ':santa-tracker:compileDebugShaders': FROM_CACHE,
        ':santa-tracker:compileDebugSources': UP_TO_DATE,
        ':santa-tracker:createDebugCompatibleScreenManifests': FROM_CACHE,
        ':santa-tracker:dexBuilderDebug': FROM_CACHE,
        ':santa-tracker:extractDeepLinksDebug': FROM_CACHE,
        ':santa-tracker:generateDebugAssets': UP_TO_DATE,
        ':santa-tracker:generateDebugBuildConfig': FROM_CACHE,
        ':santa-tracker:generateDebugFeatureMetadata': FROM_CACHE,
        ':santa-tracker:generateDebugFeatureTransitiveDeps': FROM_CACHE,
        ':santa-tracker:generateDebugResValues': FROM_CACHE,
        ':santa-tracker:generateDebugResources': SUCCESS,
        ':santa-tracker:handleDebugMicroApk': SUCCESS,
        ':santa-tracker:javaPreCompileDebug': FROM_CACHE,
        ':santa-tracker:kaptDebugKotlin': FROM_CACHE,
        ':santa-tracker:kaptGenerateStubsDebugKotlin': FROM_CACHE,
        ':santa-tracker:mainApkListPersistenceDebug': FROM_CACHE,
        ':santa-tracker:mergeDebugAssets': FROM_CACHE,
        ':santa-tracker:mergeDebugJavaResource': SUCCESS,
        ':santa-tracker:mergeDebugJniLibFolders': FROM_CACHE,
        ':santa-tracker:mergeDebugNativeLibs': FROM_CACHE,
        ':santa-tracker:mergeDebugResources': FROM_CACHE,
        ':santa-tracker:mergeDebugShaders': FROM_CACHE,
        ':santa-tracker:mergeExtDexDebug': FROM_CACHE,
        ':santa-tracker:mergeLibDexDebug': FROM_CACHE,
        ':santa-tracker:mergeProjectDexDebug': FROM_CACHE,
        ':santa-tracker:packageDebug': SUCCESS,
        ':santa-tracker:preBuild': UP_TO_DATE,
        ':santa-tracker:preDebugBuild': FROM_CACHE,
        ':santa-tracker:processDebugJavaRes': NO_SOURCE,
        ':santa-tracker:processDebugManifest': FROM_CACHE,
        ':santa-tracker:processDebugResources': FROM_CACHE,
        ':santa-tracker:signingConfigWriterDebug': FROM_CACHE,
        ':santa-tracker:stripDebugDebugSymbols': FROM_CACHE,
        ':santa-tracker:validateSigningDebug': FROM_CACHE,
        ':santa-tracker:writeDebugModuleMetadata': SUCCESS,
        ':snowballrun:assembleDebug': SUCCESS,
        ':snowballrun:checkDebugDuplicateClasses': FROM_CACHE,
        ':snowballrun:compileDebugAidl': NO_SOURCE,
        ':snowballrun:compileDebugJavaWithJavac': FROM_CACHE,
        ':snowballrun:compileDebugRenderscript': NO_SOURCE,
        ':snowballrun:compileDebugShaders': FROM_CACHE,
        ':snowballrun:compileDebugSources': UP_TO_DATE,
        ':snowballrun:createDebugCompatibleScreenManifests': FROM_CACHE,
        ':snowballrun:dexBuilderDebug': FROM_CACHE,
        ':snowballrun:extractDeepLinksDebug': FROM_CACHE,
        ':snowballrun:featureDebugWriter': SUCCESS,
        ':snowballrun:generateDebugAssets': UP_TO_DATE,
        ':snowballrun:generateDebugBuildConfig': FROM_CACHE,
        ':snowballrun:generateDebugFeatureTransitiveDeps': FROM_CACHE,
        ':snowballrun:generateDebugResValues': FROM_CACHE,
        ':snowballrun:generateDebugResources': UP_TO_DATE,
        ':snowballrun:javaPreCompileDebug': FROM_CACHE,
        ':snowballrun:mainApkListPersistenceDebug': FROM_CACHE,
        ':snowballrun:mergeDebugAssets': FROM_CACHE,
        ':snowballrun:mergeDebugJavaResource': FROM_CACHE,
        ':snowballrun:mergeDebugJniLibFolders': FROM_CACHE,
        ':snowballrun:mergeDebugNativeLibs': FROM_CACHE,
        ':snowballrun:mergeDebugResources': FROM_CACHE,
        ':snowballrun:mergeDebugShaders': FROM_CACHE,
        ':snowballrun:mergeExtDexDebug': FROM_CACHE,
        ':snowballrun:mergeLibDexDebug': FROM_CACHE,
        ':snowballrun:mergeProjectDexDebug': FROM_CACHE,
        ':snowballrun:packageDebug': SUCCESS,
        ':snowballrun:preBuild': UP_TO_DATE,
        ':snowballrun:preDebugBuild': UP_TO_DATE,
        ':snowballrun:processDebugJavaRes': NO_SOURCE,
        ':snowballrun:processDebugManifest': FROM_CACHE,
        ':snowballrun:processDebugResources': FROM_CACHE,
        ':snowballrun:stripDebugDebugSymbols': FROM_CACHE,
        ':tracker:assembleDebug': SUCCESS,
        ':tracker:bundleDebugAar': SUCCESS,
        ':tracker:bundleLibCompileDebug': SUCCESS,
        ':tracker:bundleLibResDebug': SUCCESS,
        ':tracker:bundleLibRuntimeDebug': SUCCESS,
        ':tracker:compileDebugAidl': NO_SOURCE,
        ':tracker:compileDebugJavaWithJavac': SUCCESS,
        ':tracker:compileDebugKotlin': FROM_CACHE,
        ':tracker:compileDebugLibraryResources': FROM_CACHE,
        ':tracker:compileDebugRenderscript': NO_SOURCE,
        ':tracker:compileDebugShaders': FROM_CACHE,
        ':tracker:compileDebugSources': SUCCESS,
        ':tracker:copyDebugJniLibsProjectAndLocalJars': FROM_CACHE,
        ':tracker:copyDebugJniLibsProjectOnly': FROM_CACHE,
        ':tracker:extractDebugAnnotations': FROM_CACHE,
        ':tracker:extractDeepLinksDebug': FROM_CACHE,
        ':tracker:generateDebugAssets': UP_TO_DATE,
        ':tracker:generateDebugBuildConfig': FROM_CACHE,
        ':tracker:generateDebugRFile': FROM_CACHE,
        ':tracker:generateDebugResValues': FROM_CACHE,
        ':tracker:generateDebugResources': UP_TO_DATE,
        ':tracker:javaPreCompileDebug': FROM_CACHE,
        ':tracker:kaptDebugKotlin': SUCCESS,
        ':tracker:kaptGenerateStubsDebugKotlin': SUCCESS,
        ':tracker:mergeDebugConsumerProguardFiles': SUCCESS,
        ':tracker:mergeDebugGeneratedProguardFiles': SUCCESS,
        ':tracker:mergeDebugJavaResource': SUCCESS,
        ':tracker:mergeDebugJniLibFolders': FROM_CACHE,
        ':tracker:mergeDebugNativeLibs': FROM_CACHE,
        ':tracker:mergeDebugResources': FROM_CACHE,
        ':tracker:mergeDebugShaders': FROM_CACHE,
        ':tracker:packageDebugAssets': FROM_CACHE,
        ':tracker:packageDebugRenderscript': NO_SOURCE,
        ':tracker:packageDebugResources': FROM_CACHE,
        ':tracker:parseDebugLocalResources': FROM_CACHE,
        ':tracker:preBuild': UP_TO_DATE,
        ':tracker:preDebugBuild': UP_TO_DATE,
        ':tracker:prepareLintJarForPublish': SUCCESS,
        ':tracker:processDebugJavaRes': NO_SOURCE,
        ':tracker:processDebugManifest': FROM_CACHE,
        ':tracker:stripDebugDebugSymbols': FROM_CACHE,
        ':tracker:syncDebugLibJars': SUCCESS,
        ':wearable:assembleDebug': SUCCESS,
        ':wearable:checkDebugDuplicateClasses': FROM_CACHE,
        ':wearable:compileDebugAidl': NO_SOURCE,
        ':wearable:compileDebugJavaWithJavac': FROM_CACHE,
        ':wearable:compileDebugKotlin': FROM_CACHE,
        ':wearable:compileDebugRenderscript': NO_SOURCE,
        ':wearable:compileDebugShaders': FROM_CACHE,
        ':wearable:compileDebugSources': UP_TO_DATE,
        ':wearable:createDebugCompatibleScreenManifests': FROM_CACHE,
        ':wearable:dexBuilderDebug': FROM_CACHE,
        ':wearable:extractDeepLinksDebug': FROM_CACHE,
        ':wearable:generateDebugAssets': UP_TO_DATE,
        ':wearable:generateDebugBuildConfig': FROM_CACHE,
        ':wearable:generateDebugResValues': FROM_CACHE,
        ':wearable:generateDebugResources': UP_TO_DATE,
        ':wearable:javaPreCompileDebug': FROM_CACHE,
        ':wearable:kaptDebugKotlin': FROM_CACHE,
        ':wearable:kaptGenerateStubsDebugKotlin': FROM_CACHE,
        ':wearable:mainApkListPersistenceDebug': FROM_CACHE,
        ':wearable:mergeDebugAssets': FROM_CACHE,
        ':wearable:mergeDebugJavaResource': SUCCESS,
        ':wearable:mergeDebugJniLibFolders': FROM_CACHE,
        ':wearable:mergeDebugNativeLibs': FROM_CACHE,
        ':wearable:mergeDebugResources': FROM_CACHE,
        ':wearable:mergeDebugShaders': FROM_CACHE,
        ':wearable:mergeExtDexDebug': FROM_CACHE,
        ':wearable:mergeLibDexDebug': FROM_CACHE,
        ':wearable:mergeProjectDexDebug': FROM_CACHE,
        ':wearable:packageDebug': SUCCESS,
        ':wearable:preBuild': UP_TO_DATE,
        ':wearable:preDebugBuild': UP_TO_DATE,
        ':wearable:processDebugJavaRes': NO_SOURCE,
        ':wearable:processDebugManifest': FROM_CACHE,
        ':wearable:processDebugResources': FROM_CACHE,
        ':wearable:stripDebugDebugSymbols': FROM_CACHE,
        ':wearable:validateSigningDebug': FROM_CACHE,
    ]
}
