/*
 * Decompiled with CFR 0.152.
 */
package dev.hilla.engine;

import dev.hilla.engine.EngineConfiguration;
import dev.hilla.engine.ParserConfiguration;
import dev.hilla.engine.ParserException;
import dev.hilla.parser.core.OpenAPIFileType;
import dev.hilla.parser.core.Parser;
import dev.hilla.parser.core.PluginConfiguration;
import dev.hilla.parser.core.PluginManager;
import dev.hilla.parser.utils.OpenAPIPrinter;
import io.swagger.v3.oas.models.OpenAPI;
import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.attribute.FileAttribute;
import java.util.Collection;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;
import javax.annotation.Nonnull;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public final class ParserProcessor {
    private static final Logger logger = LoggerFactory.getLogger(ParserProcessor.class);
    private final Path baseDir;
    private final ParserConfiguration.PluginsProcessor pluginsProcessor = new ParserConfiguration.PluginsProcessor();
    private final ClassLoader classLoader;
    private final Set<String> classPath;
    private String endpointAnnotationName = "dev.hilla.Endpoint";
    private String endpointExposedAnnotationName = "dev.hilla.EndpointExposed";
    private Collection<String> exposedPackages = List.of();
    private String openAPIBasePath;
    private final Path openAPIFile;

    public ParserProcessor(EngineConfiguration conf, ClassLoader classLoader) {
        this.baseDir = conf.getBaseDir();
        this.openAPIFile = conf.getOpenAPIFile();
        this.classLoader = classLoader;
        this.classPath = conf.getClassPath();
        this.applyConfiguration(conf.getParser());
    }

    private void applyConfiguration(ParserConfiguration parserConfiguration) {
        if (parserConfiguration == null) {
            return;
        }
        parserConfiguration.getEndpointAnnotation().ifPresent(this::applyEndpointAnnotation);
        parserConfiguration.getEndpointExposedAnnotation().ifPresent(this::applyEndpointExposedAnnotation);
        parserConfiguration.getOpenAPIBasePath().ifPresent(this::applyOpenAPIBase);
        parserConfiguration.getPlugins().ifPresent(this::applyPlugins);
        parserConfiguration.getPackages().ifPresent(this::applyExposedPackages);
    }

    public void process() throws ParserException {
        String openAPIString;
        Parser parser = new Parser().classLoader(this.classLoader).classPath(this.classPath).endpointAnnotation(this.endpointAnnotationName).endpointExposedAnnotation(this.endpointExposedAnnotationName).exposedPackages(this.exposedPackages);
        this.preparePlugins(parser);
        this.prepareOpenAPIBase(parser);
        logger.debug("Starting JVM Parser");
        OpenAPI openAPI = parser.execute();
        logger.debug("Saving OpenAPI file to " + this.openAPIFile);
        try {
            Files.createDirectories(this.openAPIFile.getParent(), new FileAttribute[0]);
            openAPIString = new OpenAPIPrinter().pretty().writeAsString(openAPI);
        }
        catch (IOException e) {
            throw new ParserException("Unable to prepare OpenAPI definition", e);
        }
        Optional.of(this.openAPIFile).filter(x$0 -> Files.isRegularFile(x$0, new LinkOption[0])).map(this::readFromFile).filter(openAPIString::equals).ifPresentOrElse(s -> logger.debug("OpenAPI definition has not changed"), () -> {
            try {
                Files.write(this.openAPIFile, openAPIString.getBytes(), new OpenOption[0]);
                logger.debug("OpenAPI definition file saved");
            }
            catch (IOException e) {
                throw new ParserException("Unable to save OpenAPI file", e);
            }
        });
    }

    private String readFromFile(Path path) {
        try {
            return Files.readString(path);
        }
        catch (IOException e) {
            logger.error("Unable to read file", (Throwable)e);
            return null;
        }
    }

    private void applyEndpointAnnotation(@Nonnull String endpointAnnotationName) {
        this.endpointAnnotationName = Objects.requireNonNull(endpointAnnotationName);
    }

    private void applyEndpointExposedAnnotation(@Nonnull String endpointExposedAnnotationName) {
        this.endpointExposedAnnotationName = Objects.requireNonNull(endpointExposedAnnotationName);
    }

    private void applyOpenAPIBase(@Nonnull String openAPIBasePath) {
        this.openAPIBasePath = openAPIBasePath;
    }

    private void applyPlugins(@Nonnull ParserConfiguration.Plugins plugins) {
        this.pluginsProcessor.setConfig(plugins);
    }

    private void applyExposedPackages(@Nonnull List<String> exposedPackages) {
        this.exposedPackages = exposedPackages;
    }

    private void prepareOpenAPIBase(Parser parser) {
        if (this.openAPIBasePath == null) {
            return;
        }
        try {
            Path path = this.baseDir.resolve(this.openAPIBasePath);
            String fileName = path.getFileName().toString();
            if (!(fileName.endsWith("yml") || fileName.endsWith("yaml") || fileName.endsWith("json"))) {
                throw new IOException("No OpenAPI base file found");
            }
            parser.openAPISource(Files.readString(path), fileName.endsWith("json") ? OpenAPIFileType.JSON : OpenAPIFileType.YAML);
        }
        catch (IOException e) {
            throw new ParserException("Failed loading OpenAPI spec file", e);
        }
    }

    private void preparePlugins(Parser parser) {
        List loadedPlugins = this.pluginsProcessor.process().stream().map(plugin -> PluginManager.load((String)plugin.getName(), (PluginConfiguration)plugin.getConfiguration())).collect(Collectors.toList());
        parser.plugins(loadedPlugins);
    }
}

