package io.k8s.api.admissionregistration.v1

import dev.hnaderi.k8s.utils._

/** NamedRuleWithOperations is a tuple of Operations and Resources with ResourceNames. */
final case class NamedRuleWithOperations(
  operations : Option[Seq[String]] = None,
  apiVersions : Option[Seq[String]] = None,
  scope : Option[String] = None,
  apiGroups : Option[Seq[String]] = None,
  resources : Option[Seq[String]] = None,
  resourceNames : Option[Seq[String]] = None
) {

  /** Returns a new data with operations set to new value */
  def withOperations(value: Seq[String]) : NamedRuleWithOperations = copy(operations = Some(value))
  /** Appends new values to operations */
  def addOperations(newValues: String*) : NamedRuleWithOperations = copy(operations = Some(operations.fold(newValues)(_ ++ newValues)))
  /** if operations has a value, transforms to the result of function*/
  def mapOperations(f: Seq[String] => Seq[String]) : NamedRuleWithOperations = copy(operations = operations.map(f))

  /** Returns a new data with apiVersions set to new value */
  def withApiVersions(value: Seq[String]) : NamedRuleWithOperations = copy(apiVersions = Some(value))
  /** Appends new values to apiVersions */
  def addApiVersions(newValues: String*) : NamedRuleWithOperations = copy(apiVersions = Some(apiVersions.fold(newValues)(_ ++ newValues)))
  /** if apiVersions has a value, transforms to the result of function*/
  def mapApiVersions(f: Seq[String] => Seq[String]) : NamedRuleWithOperations = copy(apiVersions = apiVersions.map(f))

  /** Returns a new data with scope set to new value */
  def withScope(value: String) : NamedRuleWithOperations = copy(scope = Some(value))
  /** if scope has a value, transforms to the result of function*/
  def mapScope(f: String => String) : NamedRuleWithOperations = copy(scope = scope.map(f))

  /** Returns a new data with apiGroups set to new value */
  def withApiGroups(value: Seq[String]) : NamedRuleWithOperations = copy(apiGroups = Some(value))
  /** Appends new values to apiGroups */
  def addApiGroups(newValues: String*) : NamedRuleWithOperations = copy(apiGroups = Some(apiGroups.fold(newValues)(_ ++ newValues)))
  /** if apiGroups has a value, transforms to the result of function*/
  def mapApiGroups(f: Seq[String] => Seq[String]) : NamedRuleWithOperations = copy(apiGroups = apiGroups.map(f))

  /** Returns a new data with resources set to new value */
  def withResources(value: Seq[String]) : NamedRuleWithOperations = copy(resources = Some(value))
  /** Appends new values to resources */
  def addResources(newValues: String*) : NamedRuleWithOperations = copy(resources = Some(resources.fold(newValues)(_ ++ newValues)))
  /** if resources has a value, transforms to the result of function*/
  def mapResources(f: Seq[String] => Seq[String]) : NamedRuleWithOperations = copy(resources = resources.map(f))

  /** Returns a new data with resourceNames set to new value */
  def withResourceNames(value: Seq[String]) : NamedRuleWithOperations = copy(resourceNames = Some(value))
  /** Appends new values to resourceNames */
  def addResourceNames(newValues: String*) : NamedRuleWithOperations = copy(resourceNames = Some(resourceNames.fold(newValues)(_ ++ newValues)))
  /** if resourceNames has a value, transforms to the result of function*/
  def mapResourceNames(f: Seq[String] => Seq[String]) : NamedRuleWithOperations = copy(resourceNames = resourceNames.map(f))
}

object NamedRuleWithOperations {

    implicit val encoder : Encoder[io.k8s.api.admissionregistration.v1.NamedRuleWithOperations] = new Encoder[io.k8s.api.admissionregistration.v1.NamedRuleWithOperations] {
        def apply[T : Builder](o: io.k8s.api.admissionregistration.v1.NamedRuleWithOperations) : T = {
          val obj = ObjectWriter[T]()
          obj
            .write("operations", o.operations)
            .write("apiVersions", o.apiVersions)
            .write("scope", o.scope)
            .write("apiGroups", o.apiGroups)
            .write("resources", o.resources)
            .write("resourceNames", o.resourceNames)
            .build
        }
    }

    implicit val decoder: Decoder[NamedRuleWithOperations] = new Decoder[NamedRuleWithOperations] {
      def apply[T : Reader](t: T): Either[String, NamedRuleWithOperations] = for {
          obj <- ObjectReader(t)
          operations <- obj.readOpt[Seq[String]]("operations")
          apiVersions <- obj.readOpt[Seq[String]]("apiVersions")
          scope <- obj.readOpt[String]("scope")
          apiGroups <- obj.readOpt[Seq[String]]("apiGroups")
          resources <- obj.readOpt[Seq[String]]("resources")
          resourceNames <- obj.readOpt[Seq[String]]("resourceNames")
      } yield NamedRuleWithOperations (
          operations = operations,
          apiVersions = apiVersions,
          scope = scope,
          apiGroups = apiGroups,
          resources = resources,
          resourceNames = resourceNames
        )
    }
}

