package io.k8s.api.apps.v1

import dev.hnaderi.k8s.utils._

/** DeploymentSpec is the specification of the desired behavior of the Deployment. */
final case class DeploymentSpec(
  template : io.k8s.api.core.v1.PodTemplateSpec,
  selector : io.k8s.apimachinery.pkg.apis.meta.v1.LabelSelector,
  minReadySeconds : Option[Int] = None,
  progressDeadlineSeconds : Option[Int] = None,
  replicas : Option[Int] = None,
  revisionHistoryLimit : Option[Int] = None,
  paused : Option[Boolean] = None,
  strategy : Option[io.k8s.api.apps.v1.DeploymentStrategy] = None
) {

  /** Returns a new data with template set to new value */
  def withTemplate(value: io.k8s.api.core.v1.PodTemplateSpec) : DeploymentSpec = copy(template = value)
  /** transforms template to result of function */
  def mapTemplate(f: io.k8s.api.core.v1.PodTemplateSpec => io.k8s.api.core.v1.PodTemplateSpec) : DeploymentSpec = copy(template = f(template))

  /** Returns a new data with selector set to new value */
  def withSelector(value: io.k8s.apimachinery.pkg.apis.meta.v1.LabelSelector) : DeploymentSpec = copy(selector = value)
  /** transforms selector to result of function */
  def mapSelector(f: io.k8s.apimachinery.pkg.apis.meta.v1.LabelSelector => io.k8s.apimachinery.pkg.apis.meta.v1.LabelSelector) : DeploymentSpec = copy(selector = f(selector))

  /** Returns a new data with minReadySeconds set to new value */
  def withMinReadySeconds(value: Int) : DeploymentSpec = copy(minReadySeconds = Some(value))
  /** if minReadySeconds has a value, transforms to the result of function*/
  def mapMinReadySeconds(f: Int => Int) : DeploymentSpec = copy(minReadySeconds = minReadySeconds.map(f))

  /** Returns a new data with progressDeadlineSeconds set to new value */
  def withProgressDeadlineSeconds(value: Int) : DeploymentSpec = copy(progressDeadlineSeconds = Some(value))
  /** if progressDeadlineSeconds has a value, transforms to the result of function*/
  def mapProgressDeadlineSeconds(f: Int => Int) : DeploymentSpec = copy(progressDeadlineSeconds = progressDeadlineSeconds.map(f))

  /** Returns a new data with replicas set to new value */
  def withReplicas(value: Int) : DeploymentSpec = copy(replicas = Some(value))
  /** if replicas has a value, transforms to the result of function*/
  def mapReplicas(f: Int => Int) : DeploymentSpec = copy(replicas = replicas.map(f))

  /** Returns a new data with revisionHistoryLimit set to new value */
  def withRevisionHistoryLimit(value: Int) : DeploymentSpec = copy(revisionHistoryLimit = Some(value))
  /** if revisionHistoryLimit has a value, transforms to the result of function*/
  def mapRevisionHistoryLimit(f: Int => Int) : DeploymentSpec = copy(revisionHistoryLimit = revisionHistoryLimit.map(f))

  /** Returns a new data with paused set to new value */
  def withPaused(value: Boolean) : DeploymentSpec = copy(paused = Some(value))
  /** if paused has a value, transforms to the result of function*/
  def mapPaused(f: Boolean => Boolean) : DeploymentSpec = copy(paused = paused.map(f))

  /** Returns a new data with strategy set to new value */
  def withStrategy(value: io.k8s.api.apps.v1.DeploymentStrategy) : DeploymentSpec = copy(strategy = Some(value))
  /** if strategy has a value, transforms to the result of function*/
  def mapStrategy(f: io.k8s.api.apps.v1.DeploymentStrategy => io.k8s.api.apps.v1.DeploymentStrategy) : DeploymentSpec = copy(strategy = strategy.map(f))
}

object DeploymentSpec {

    implicit val encoder : Encoder[io.k8s.api.apps.v1.DeploymentSpec] = new Encoder[io.k8s.api.apps.v1.DeploymentSpec] {
        def apply[T : Builder](o: io.k8s.api.apps.v1.DeploymentSpec) : T = {
          val obj = ObjectWriter[T]()
          obj
            .write("template", o.template)
            .write("selector", o.selector)
            .write("minReadySeconds", o.minReadySeconds)
            .write("progressDeadlineSeconds", o.progressDeadlineSeconds)
            .write("replicas", o.replicas)
            .write("revisionHistoryLimit", o.revisionHistoryLimit)
            .write("paused", o.paused)
            .write("strategy", o.strategy)
            .build
        }
    }

    implicit val decoder: Decoder[DeploymentSpec] = new Decoder[DeploymentSpec] {
      def apply[T : Reader](t: T): Either[String, DeploymentSpec] = for {
          obj <- ObjectReader(t)
          template <- obj.read[io.k8s.api.core.v1.PodTemplateSpec]("template")
          selector <- obj.read[io.k8s.apimachinery.pkg.apis.meta.v1.LabelSelector]("selector")
          minReadySeconds <- obj.readOpt[Int]("minReadySeconds")
          progressDeadlineSeconds <- obj.readOpt[Int]("progressDeadlineSeconds")
          replicas <- obj.readOpt[Int]("replicas")
          revisionHistoryLimit <- obj.readOpt[Int]("revisionHistoryLimit")
          paused <- obj.readOpt[Boolean]("paused")
          strategy <- obj.readOpt[io.k8s.api.apps.v1.DeploymentStrategy]("strategy")
      } yield DeploymentSpec (
          template = template,
          selector = selector,
          minReadySeconds = minReadySeconds,
          progressDeadlineSeconds = progressDeadlineSeconds,
          replicas = replicas,
          revisionHistoryLimit = revisionHistoryLimit,
          paused = paused,
          strategy = strategy
        )
    }
}

