package io.k8s.api.apps.v1

import dev.hnaderi.k8s.utils._

/** StatefulSetCondition describes the state of a statefulset at a certain point. */
final case class StatefulSetCondition(
  status : String,
  `type` : String,
  reason : Option[String] = None,
  lastTransitionTime : Option[io.k8s.apimachinery.pkg.apis.meta.v1.Time] = None,
  message : Option[String] = None
) {

  /** Returns a new data with status set to new value */
  def withStatus(value: String) : StatefulSetCondition = copy(status = value)
  /** transforms status to result of function */
  def mapStatus(f: String => String) : StatefulSetCondition = copy(status = f(status))

  /** Returns a new data with `type` set to new value */
  def withType(value: String) : StatefulSetCondition = copy(`type` = value)
  /** transforms `type` to result of function */
  def mapType(f: String => String) : StatefulSetCondition = copy(`type` = f(`type`))

  /** Returns a new data with reason set to new value */
  def withReason(value: String) : StatefulSetCondition = copy(reason = Some(value))
  /** if reason has a value, transforms to the result of function*/
  def mapReason(f: String => String) : StatefulSetCondition = copy(reason = reason.map(f))

  /** Returns a new data with lastTransitionTime set to new value */
  def withLastTransitionTime(value: io.k8s.apimachinery.pkg.apis.meta.v1.Time) : StatefulSetCondition = copy(lastTransitionTime = Some(value))
  /** if lastTransitionTime has a value, transforms to the result of function*/
  def mapLastTransitionTime(f: io.k8s.apimachinery.pkg.apis.meta.v1.Time => io.k8s.apimachinery.pkg.apis.meta.v1.Time) : StatefulSetCondition = copy(lastTransitionTime = lastTransitionTime.map(f))

  /** Returns a new data with message set to new value */
  def withMessage(value: String) : StatefulSetCondition = copy(message = Some(value))
  /** if message has a value, transforms to the result of function*/
  def mapMessage(f: String => String) : StatefulSetCondition = copy(message = message.map(f))
}

object StatefulSetCondition {

    implicit val encoder : Encoder[io.k8s.api.apps.v1.StatefulSetCondition] = new Encoder[io.k8s.api.apps.v1.StatefulSetCondition] {
        def apply[T : Builder](o: io.k8s.api.apps.v1.StatefulSetCondition) : T = {
          val obj = ObjectWriter[T]()
          obj
            .write("status", o.status)
            .write("type", o.`type`)
            .write("reason", o.reason)
            .write("lastTransitionTime", o.lastTransitionTime)
            .write("message", o.message)
            .build
        }
    }

    implicit val decoder: Decoder[StatefulSetCondition] = new Decoder[StatefulSetCondition] {
      def apply[T : Reader](t: T): Either[String, StatefulSetCondition] = for {
          obj <- ObjectReader(t)
          status <- obj.read[String]("status")
          `type` <- obj.read[String]("type")
          reason <- obj.readOpt[String]("reason")
          lastTransitionTime <- obj.readOpt[io.k8s.apimachinery.pkg.apis.meta.v1.Time]("lastTransitionTime")
          message <- obj.readOpt[String]("message")
      } yield StatefulSetCondition (
          status = status,
          `type` = `type`,
          reason = reason,
          lastTransitionTime = lastTransitionTime,
          message = message
        )
    }
}

