package io.k8s.api.autoscaling.v2

import dev.hnaderi.k8s.utils._

/** HorizontalPodAutoscalerSpec describes the desired functionality of the HorizontalPodAutoscaler. */
final case class HorizontalPodAutoscalerSpec(
  maxReplicas : Int,
  scaleTargetRef : io.k8s.api.autoscaling.v2.CrossVersionObjectReference,
  minReplicas : Option[Int] = None,
  metrics : Option[Seq[io.k8s.api.autoscaling.v2.MetricSpec]] = None,
  behavior : Option[io.k8s.api.autoscaling.v2.HorizontalPodAutoscalerBehavior] = None
) {

  /** Returns a new data with maxReplicas set to new value */
  def withMaxReplicas(value: Int) : HorizontalPodAutoscalerSpec = copy(maxReplicas = value)
  /** transforms maxReplicas to result of function */
  def mapMaxReplicas(f: Int => Int) : HorizontalPodAutoscalerSpec = copy(maxReplicas = f(maxReplicas))

  /** Returns a new data with scaleTargetRef set to new value */
  def withScaleTargetRef(value: io.k8s.api.autoscaling.v2.CrossVersionObjectReference) : HorizontalPodAutoscalerSpec = copy(scaleTargetRef = value)
  /** transforms scaleTargetRef to result of function */
  def mapScaleTargetRef(f: io.k8s.api.autoscaling.v2.CrossVersionObjectReference => io.k8s.api.autoscaling.v2.CrossVersionObjectReference) : HorizontalPodAutoscalerSpec = copy(scaleTargetRef = f(scaleTargetRef))

  /** Returns a new data with minReplicas set to new value */
  def withMinReplicas(value: Int) : HorizontalPodAutoscalerSpec = copy(minReplicas = Some(value))
  /** if minReplicas has a value, transforms to the result of function*/
  def mapMinReplicas(f: Int => Int) : HorizontalPodAutoscalerSpec = copy(minReplicas = minReplicas.map(f))

  /** Returns a new data with metrics set to new value */
  def withMetrics(value: Seq[io.k8s.api.autoscaling.v2.MetricSpec]) : HorizontalPodAutoscalerSpec = copy(metrics = Some(value))
  /** Appends new values to metrics */
  def addMetrics(newValues: io.k8s.api.autoscaling.v2.MetricSpec*) : HorizontalPodAutoscalerSpec = copy(metrics = Some(metrics.fold(newValues)(_ ++ newValues)))
  /** if metrics has a value, transforms to the result of function*/
  def mapMetrics(f: Seq[io.k8s.api.autoscaling.v2.MetricSpec] => Seq[io.k8s.api.autoscaling.v2.MetricSpec]) : HorizontalPodAutoscalerSpec = copy(metrics = metrics.map(f))

  /** Returns a new data with behavior set to new value */
  def withBehavior(value: io.k8s.api.autoscaling.v2.HorizontalPodAutoscalerBehavior) : HorizontalPodAutoscalerSpec = copy(behavior = Some(value))
  /** if behavior has a value, transforms to the result of function*/
  def mapBehavior(f: io.k8s.api.autoscaling.v2.HorizontalPodAutoscalerBehavior => io.k8s.api.autoscaling.v2.HorizontalPodAutoscalerBehavior) : HorizontalPodAutoscalerSpec = copy(behavior = behavior.map(f))
}

object HorizontalPodAutoscalerSpec {

    implicit val encoder : Encoder[io.k8s.api.autoscaling.v2.HorizontalPodAutoscalerSpec] = new Encoder[io.k8s.api.autoscaling.v2.HorizontalPodAutoscalerSpec] {
        def apply[T : Builder](o: io.k8s.api.autoscaling.v2.HorizontalPodAutoscalerSpec) : T = {
          val obj = ObjectWriter[T]()
          obj
            .write("maxReplicas", o.maxReplicas)
            .write("scaleTargetRef", o.scaleTargetRef)
            .write("minReplicas", o.minReplicas)
            .write("metrics", o.metrics)
            .write("behavior", o.behavior)
            .build
        }
    }

    implicit val decoder: Decoder[HorizontalPodAutoscalerSpec] = new Decoder[HorizontalPodAutoscalerSpec] {
      def apply[T : Reader](t: T): Either[String, HorizontalPodAutoscalerSpec] = for {
          obj <- ObjectReader(t)
          maxReplicas <- obj.read[Int]("maxReplicas")
          scaleTargetRef <- obj.read[io.k8s.api.autoscaling.v2.CrossVersionObjectReference]("scaleTargetRef")
          minReplicas <- obj.readOpt[Int]("minReplicas")
          metrics <- obj.readOpt[Seq[io.k8s.api.autoscaling.v2.MetricSpec]]("metrics")
          behavior <- obj.readOpt[io.k8s.api.autoscaling.v2.HorizontalPodAutoscalerBehavior]("behavior")
      } yield HorizontalPodAutoscalerSpec (
          maxReplicas = maxReplicas,
          scaleTargetRef = scaleTargetRef,
          minReplicas = minReplicas,
          metrics = metrics,
          behavior = behavior
        )
    }
}

