package io.k8s.api.batch.v1

import dev.hnaderi.k8s.utils._

/** PodFailurePolicyOnExitCodesRequirement describes the requirement for handling a failed pod based on its container exit codes. In particular, it lookups the .state.terminated.exitCode for each app container and init container status, represented by the .status.containerStatuses and .status.initContainerStatuses fields in the Pod status, respectively. Containers completed with success (exit code 0) are excluded from the requirement check. */
final case class PodFailurePolicyOnExitCodesRequirement(
  operator : String,
  values : Seq[Int],
  containerName : Option[String] = None
) {

  /** Returns a new data with operator set to new value */
  def withOperator(value: String) : PodFailurePolicyOnExitCodesRequirement = copy(operator = value)
  /** transforms operator to result of function */
  def mapOperator(f: String => String) : PodFailurePolicyOnExitCodesRequirement = copy(operator = f(operator))

  /** Returns a new data with values set to new value */
  def withValues(value: Seq[Int]) : PodFailurePolicyOnExitCodesRequirement = copy(values = value)
  /** Appends new values to values */
  def addValues(newValues: Int*) : PodFailurePolicyOnExitCodesRequirement = copy(values = values ++ newValues)
  /** transforms values to result of function */
  def mapValues(f: Seq[Int] => Seq[Int]) : PodFailurePolicyOnExitCodesRequirement = copy(values = f(values))

  /** Returns a new data with containerName set to new value */
  def withContainerName(value: String) : PodFailurePolicyOnExitCodesRequirement = copy(containerName = Some(value))
  /** if containerName has a value, transforms to the result of function*/
  def mapContainerName(f: String => String) : PodFailurePolicyOnExitCodesRequirement = copy(containerName = containerName.map(f))
}

object PodFailurePolicyOnExitCodesRequirement {

    implicit val encoder : Encoder[io.k8s.api.batch.v1.PodFailurePolicyOnExitCodesRequirement] = new Encoder[io.k8s.api.batch.v1.PodFailurePolicyOnExitCodesRequirement] {
        def apply[T : Builder](o: io.k8s.api.batch.v1.PodFailurePolicyOnExitCodesRequirement) : T = {
          val obj = ObjectWriter[T]()
          obj
            .write("operator", o.operator)
            .write("values", o.values)
            .write("containerName", o.containerName)
            .build
        }
    }

    implicit val decoder: Decoder[PodFailurePolicyOnExitCodesRequirement] = new Decoder[PodFailurePolicyOnExitCodesRequirement] {
      def apply[T : Reader](t: T): Either[String, PodFailurePolicyOnExitCodesRequirement] = for {
          obj <- ObjectReader(t)
          operator <- obj.read[String]("operator")
          values <- obj.read[Seq[Int]]("values")
          containerName <- obj.readOpt[String]("containerName")
      } yield PodFailurePolicyOnExitCodesRequirement (
          operator = operator,
          values = values,
          containerName = containerName
        )
    }
}

