package io.k8s.api.core.v1

import dev.hnaderi.k8s.utils._

/** Represents a cinder volume resource in Openstack. A Cinder volume must exist before mounting to a container. The volume must also be in the same region as the kubelet. Cinder volumes support ownership management and SELinux relabeling. */
final case class CinderVolumeSource(
  volumeID : String,
  fsType : Option[String] = None,
  readOnly : Option[Boolean] = None,
  secretRef : Option[io.k8s.api.core.v1.LocalObjectReference] = None
) {

  /** Returns a new data with volumeID set to new value */
  def withVolumeID(value: String) : CinderVolumeSource = copy(volumeID = value)
  /** transforms volumeID to result of function */
  def mapVolumeID(f: String => String) : CinderVolumeSource = copy(volumeID = f(volumeID))

  /** Returns a new data with fsType set to new value */
  def withFsType(value: String) : CinderVolumeSource = copy(fsType = Some(value))
  /** if fsType has a value, transforms to the result of function*/
  def mapFsType(f: String => String) : CinderVolumeSource = copy(fsType = fsType.map(f))

  /** Returns a new data with readOnly set to new value */
  def withReadOnly(value: Boolean) : CinderVolumeSource = copy(readOnly = Some(value))
  /** if readOnly has a value, transforms to the result of function*/
  def mapReadOnly(f: Boolean => Boolean) : CinderVolumeSource = copy(readOnly = readOnly.map(f))

  /** Returns a new data with secretRef set to new value */
  def withSecretRef(value: io.k8s.api.core.v1.LocalObjectReference) : CinderVolumeSource = copy(secretRef = Some(value))
  /** if secretRef has a value, transforms to the result of function*/
  def mapSecretRef(f: io.k8s.api.core.v1.LocalObjectReference => io.k8s.api.core.v1.LocalObjectReference) : CinderVolumeSource = copy(secretRef = secretRef.map(f))
}

object CinderVolumeSource {

    implicit val encoder : Encoder[io.k8s.api.core.v1.CinderVolumeSource] = new Encoder[io.k8s.api.core.v1.CinderVolumeSource] {
        def apply[T : Builder](o: io.k8s.api.core.v1.CinderVolumeSource) : T = {
          val obj = ObjectWriter[T]()
          obj
            .write("volumeID", o.volumeID)
            .write("fsType", o.fsType)
            .write("readOnly", o.readOnly)
            .write("secretRef", o.secretRef)
            .build
        }
    }

    implicit val decoder: Decoder[CinderVolumeSource] = new Decoder[CinderVolumeSource] {
      def apply[T : Reader](t: T): Either[String, CinderVolumeSource] = for {
          obj <- ObjectReader(t)
          volumeID <- obj.read[String]("volumeID")
          fsType <- obj.readOpt[String]("fsType")
          readOnly <- obj.readOpt[Boolean]("readOnly")
          secretRef <- obj.readOpt[io.k8s.api.core.v1.LocalObjectReference]("secretRef")
      } yield CinderVolumeSource (
          volumeID = volumeID,
          fsType = fsType,
          readOnly = readOnly,
          secretRef = secretRef
        )
    }
}

