package io.k8s.api.core.v1

import dev.hnaderi.k8s.utils._

/** HTTPGetAction describes an action based on HTTP Get requests. */
final case class HTTPGetAction(
  port : io.k8s.apimachinery.pkg.util.intstr.IntOrString,
  path : Option[String] = None,
  host : Option[String] = None,
  scheme : Option[String] = None,
  httpHeaders : Option[Seq[io.k8s.api.core.v1.HTTPHeader]] = None
) {

  /** Returns a new data with port set to new value */
  def withPort(value: io.k8s.apimachinery.pkg.util.intstr.IntOrString) : HTTPGetAction = copy(port = value)
  /** transforms port to result of function */
  def mapPort(f: io.k8s.apimachinery.pkg.util.intstr.IntOrString => io.k8s.apimachinery.pkg.util.intstr.IntOrString) : HTTPGetAction = copy(port = f(port))

  /** Returns a new data with path set to new value */
  def withPath(value: String) : HTTPGetAction = copy(path = Some(value))
  /** if path has a value, transforms to the result of function*/
  def mapPath(f: String => String) : HTTPGetAction = copy(path = path.map(f))

  /** Returns a new data with host set to new value */
  def withHost(value: String) : HTTPGetAction = copy(host = Some(value))
  /** if host has a value, transforms to the result of function*/
  def mapHost(f: String => String) : HTTPGetAction = copy(host = host.map(f))

  /** Returns a new data with scheme set to new value */
  def withScheme(value: String) : HTTPGetAction = copy(scheme = Some(value))
  /** if scheme has a value, transforms to the result of function*/
  def mapScheme(f: String => String) : HTTPGetAction = copy(scheme = scheme.map(f))

  /** Returns a new data with httpHeaders set to new value */
  def withHttpHeaders(value: Seq[io.k8s.api.core.v1.HTTPHeader]) : HTTPGetAction = copy(httpHeaders = Some(value))
  /** Appends new values to httpHeaders */
  def addHttpHeaders(newValues: io.k8s.api.core.v1.HTTPHeader*) : HTTPGetAction = copy(httpHeaders = Some(httpHeaders.fold(newValues)(_ ++ newValues)))
  /** if httpHeaders has a value, transforms to the result of function*/
  def mapHttpHeaders(f: Seq[io.k8s.api.core.v1.HTTPHeader] => Seq[io.k8s.api.core.v1.HTTPHeader]) : HTTPGetAction = copy(httpHeaders = httpHeaders.map(f))
}

object HTTPGetAction {

    implicit val encoder : Encoder[io.k8s.api.core.v1.HTTPGetAction] = new Encoder[io.k8s.api.core.v1.HTTPGetAction] {
        def apply[T : Builder](o: io.k8s.api.core.v1.HTTPGetAction) : T = {
          val obj = ObjectWriter[T]()
          obj
            .write("port", o.port)
            .write("path", o.path)
            .write("host", o.host)
            .write("scheme", o.scheme)
            .write("httpHeaders", o.httpHeaders)
            .build
        }
    }

    implicit val decoder: Decoder[HTTPGetAction] = new Decoder[HTTPGetAction] {
      def apply[T : Reader](t: T): Either[String, HTTPGetAction] = for {
          obj <- ObjectReader(t)
          port <- obj.read[io.k8s.apimachinery.pkg.util.intstr.IntOrString]("port")
          path <- obj.readOpt[String]("path")
          host <- obj.readOpt[String]("host")
          scheme <- obj.readOpt[String]("scheme")
          httpHeaders <- obj.readOpt[Seq[io.k8s.api.core.v1.HTTPHeader]]("httpHeaders")
      } yield HTTPGetAction (
          port = port,
          path = path,
          host = host,
          scheme = scheme,
          httpHeaders = httpHeaders
        )
    }
}

