package io.k8s.api.core.v1

import dev.hnaderi.k8s.utils._

/** LoadBalancerIngress represents the status of a load-balancer ingress point: traffic intended for the service should be sent to an ingress point. */
final case class LoadBalancerIngress(
  hostname : Option[String] = None,
  ip : Option[String] = None,
  ipMode : Option[String] = None,
  ports : Option[Seq[io.k8s.api.core.v1.PortStatus]] = None
) {

  /** Returns a new data with hostname set to new value */
  def withHostname(value: String) : LoadBalancerIngress = copy(hostname = Some(value))
  /** if hostname has a value, transforms to the result of function*/
  def mapHostname(f: String => String) : LoadBalancerIngress = copy(hostname = hostname.map(f))

  /** Returns a new data with ip set to new value */
  def withIp(value: String) : LoadBalancerIngress = copy(ip = Some(value))
  /** if ip has a value, transforms to the result of function*/
  def mapIp(f: String => String) : LoadBalancerIngress = copy(ip = ip.map(f))

  /** Returns a new data with ipMode set to new value */
  def withIpMode(value: String) : LoadBalancerIngress = copy(ipMode = Some(value))
  /** if ipMode has a value, transforms to the result of function*/
  def mapIpMode(f: String => String) : LoadBalancerIngress = copy(ipMode = ipMode.map(f))

  /** Returns a new data with ports set to new value */
  def withPorts(value: Seq[io.k8s.api.core.v1.PortStatus]) : LoadBalancerIngress = copy(ports = Some(value))
  /** Appends new values to ports */
  def addPorts(newValues: io.k8s.api.core.v1.PortStatus*) : LoadBalancerIngress = copy(ports = Some(ports.fold(newValues)(_ ++ newValues)))
  /** if ports has a value, transforms to the result of function*/
  def mapPorts(f: Seq[io.k8s.api.core.v1.PortStatus] => Seq[io.k8s.api.core.v1.PortStatus]) : LoadBalancerIngress = copy(ports = ports.map(f))
}

object LoadBalancerIngress {

    implicit val encoder : Encoder[io.k8s.api.core.v1.LoadBalancerIngress] = new Encoder[io.k8s.api.core.v1.LoadBalancerIngress] {
        def apply[T : Builder](o: io.k8s.api.core.v1.LoadBalancerIngress) : T = {
          val obj = ObjectWriter[T]()
          obj
            .write("hostname", o.hostname)
            .write("ip", o.ip)
            .write("ipMode", o.ipMode)
            .write("ports", o.ports)
            .build
        }
    }

    implicit val decoder: Decoder[LoadBalancerIngress] = new Decoder[LoadBalancerIngress] {
      def apply[T : Reader](t: T): Either[String, LoadBalancerIngress] = for {
          obj <- ObjectReader(t)
          hostname <- obj.readOpt[String]("hostname")
          ip <- obj.readOpt[String]("ip")
          ipMode <- obj.readOpt[String]("ipMode")
          ports <- obj.readOpt[Seq[io.k8s.api.core.v1.PortStatus]]("ports")
      } yield LoadBalancerIngress (
          hostname = hostname,
          ip = ip,
          ipMode = ipMode,
          ports = ports
        )
    }
}

